use strict;
use warnings;

use Test::More import => [ qw( BAIL_OUT is isa_ok like ok plan require_ok subtest ) ], tests => 8;
use Test::Fatal qw( exception );
my $class;

BEGIN {
  $class = 'Version::Semantic';
  require_ok $class or BAIL_OUT "Cannot load class '$class'!"
}

like exception { $class->parse( 'v1.2.3' )->increment( 'pre_release' ) }, qr/not implemented/,
  'Unknown version incrementation strategy';

my $strategy;

subtest 'No strategy' => sub {
  plan tests => 6;

  my $start = $class->parse( 'v1.2.3' );
  isa_ok my $self = $start->increment(), $class;
  is $self->major,        1,        'major';
  is $self->minor,        2,        'minor';
  is $self->patch,        4,        'patch';
  is $self->version_core, 'v1.2.4', 'version_core';
  ok $start < $self, 'Incremented'
};

$strategy = 'patch';
subtest "\"$strategy\" strategy" => sub {
  plan tests => 6;

  my $start = $class->parse( '1.2.3' );
  isa_ok my $self = $start->increment( $strategy ), $class;
  is $self->major,        1,       'major';
  is $self->minor,        2,       'minor';
  is $self->patch,        4,       'patch';
  is $self->version_core, '1.2.4', 'version_core';
  ok $start < $self, 'Incremented'
};

subtest "\"$strategy\" strategy with TRIAL pre-release" => sub {
  plan tests => 7;

  my $start = $class->parse( '1.2.3' );
  isa_ok my $self = $start->increment( $strategy, pre_release => 'TRIAL' ), $class;
  is $self->major,        1,             'major';
  is $self->minor,        2,             'minor';
  is $self->patch,        4,             'patch';
  is $self->version_core, '1.2.4',       'version_core';
  is "$self",             '1.2.4-TRIAL', 'Stringification';
  ok $start < $self, 'Incremented'
};

$strategy = 'minor';
subtest "\"$strategy\" strategy" => sub {
  plan tests => 10;

  my $start = $class->parse( 'v1.2.3-beta' );
  isa_ok my $self = $start->increment( $strategy ), $class;
  is $self->major,        1,        'major';
  is $self->minor,        3,        'minor';
  is $self->patch,        0,        'patch';
  is $self->version_core, 'v1.3.0', 'version_core';
  is $self->pre_release,  'beta',   'pre_release';
  ok $self->has_pre_release,  'pre_release is defined';
  ok not( $self->has_build ), 'build is not defined';
  is "$self", 'v1.3.0-beta', 'Stringification';
  ok $start < $self, 'Incremented'
};

$strategy = 'major';
subtest "\"$strategy\" strategy" => sub {
  plan tests => 6;

  my $start = $class->parse( '1.2.3' );
  isa_ok my $self = $start->increment( $strategy ), $class;
  is $self->major,        2,       'major';
  is $self->minor,        0,       'minor';
  is $self->patch,        0,       'patch';
  is $self->version_core, '2.0.0', 'version_core';
  ok $start < $self, 'Incremented'
};

$strategy = 'trial';
subtest "\"$strategy\" strategy" => sub {
  plan tests => 18;

  like exception { $class->parse( 'v1.2.3' )->increment( 'trial' ) }, qr/\ACannot apply '$strategy'/,
    'Version is not a pre-release version';
  like exception { $class->parse( 'v1.2.3-alpha.1' )->increment( 'trial' ) }, qr/does not match/,
    'Invalid pre-release extension';

  my $start = $class->parse( '1.2.3-TRIAL' );
  isa_ok my $self = $start->increment( $strategy ), $class;
  is $self->major,        1,        'major';
  is $self->minor,        2,        'minor';
  is $self->patch,        3,        'patch';
  is $self->version_core, '1.2.3',  'version_core';
  is $self->pre_release,  'TRIAL1', 'pre_release';
  ok $self->has_pre_release, 'pre_release is defined';
  ok $start < $self,         'Incremented';

  $start = $class->parse( 'v1.2.3-TRIAL004' );
  isa_ok $self = $start->increment( $strategy ), $class;
  is $self->major,        1,        'major';
  is $self->minor,        2,        'minor';
  is $self->patch,        3,        'patch';
  is $self->version_core, 'v1.2.3', 'version_core';
  is $self->pre_release,  'TRIAL5', 'pre_release (leading zeros removed!)';
  ok $self->has_pre_release, 'pre_release is defined';
  ok $start < $self,         'Incremented'
}
