/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.vault.support;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.security.GeneralSecurityException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.security.spec.RSAPrivateCrtKeySpec;
import java.security.spec.RSAPublicKeySpec;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;
import org.springframework.util.Base64Utils;
import org.springframework.vault.support.KeyFactories;
import org.springframework.vault.support.KeystoreUtil;

public class PemObject {
    private static final Pattern BEGIN_PATTERN = Pattern.compile("-+BEGIN ([A-Z ]+)-+");
    private static final Pattern END_PATTERN = Pattern.compile("-+END ([A-Z ]+)-+");
    private final PemObjectType objectType;
    private final byte[] content;

    private PemObject(PemObjectType objectType, String content) {
        this.objectType = objectType;
        String sanitized = content.replaceAll("\r", "").replaceAll("\n", "");
        this.content = Base64Utils.decodeFromString((String)sanitized);
    }

    public static boolean isPemEncoded(String content) {
        return BEGIN_PATTERN.matcher(content).find() && END_PATTERN.matcher(content).find();
    }

    public static PemObject fromKey(String content) {
        return PemObject.parse(content).stream().filter(PemObject::isPrivateKey).findFirst().orElseThrow(() -> new IllegalArgumentException("Could not find a PKCS #8 private key"));
    }

    public static PemObject parseFirst(String content) {
        List<PemObject> objects = PemObject.parse(content);
        if (objects.isEmpty()) {
            throw new IllegalArgumentException("Cannot find PEM object");
        }
        return objects.get(0);
    }

    public static List<PemObject> parse(String content) {
        ArrayList<PemObject> objects = new ArrayList<PemObject>();
        try (BufferedReader reader = new BufferedReader(new StringReader(content));){
            PemObject object;
            while ((object = PemObject.readNextSection(reader)) != null) {
                objects.add(object);
            }
        }
        catch (IOException e) {
            throw new RuntimeException("No way this could happen with a StringReader underneath", e);
        }
        return objects;
    }

    @Nullable
    static PemObject readNextSection(BufferedReader reader) throws IOException {
        String title = null;
        StringBuilder keyBuilder = null;
        while (true) {
            Matcher m;
            String line;
            if ((line = reader.readLine()) == null) {
                Assert.isTrue((title == null ? 1 : 0) != 0, (String)("missing end tag " + title));
                return null;
            }
            if (keyBuilder == null) {
                m = BEGIN_PATTERN.matcher(line);
                if (!m.matches()) continue;
                String curTitle = m.group(1);
                keyBuilder = new StringBuilder();
                title = curTitle;
                continue;
            }
            m = END_PATTERN.matcher(line);
            if (m.matches()) {
                String endTitle = m.group(1);
                if (!endTitle.equals(title)) {
                    throw new IllegalArgumentException(String.format("end tag (%s) doesn't match begin tag (%s)", endTitle, title));
                }
                return new PemObject(PemObjectType.of(title), keyBuilder.toString());
            }
            keyBuilder.append(line);
        }
    }

    public boolean isCertificate() {
        return PemObjectType.CERTIFICATE == this.objectType || PemObjectType.X509_CERTIFICATE == this.objectType || PemObjectType.TRUSTED_CERTIFICATE == this.objectType;
    }

    public boolean isPrivateKey() {
        return PemObjectType.PRIVATE_KEY == this.objectType || PemObjectType.EC_PRIVATE_KEY == this.objectType || PemObjectType.RSA_PRIVATE_KEY == this.objectType;
    }

    public boolean isPublicKey() {
        return PemObjectType.PUBLIC_KEY == this.objectType || PemObjectType.RSA_PUBLIC_KEY == this.objectType;
    }

    public X509Certificate getCertificate() {
        if (!this.isCertificate()) {
            throw new IllegalStateException("PEM object is not a certificate");
        }
        try {
            return KeystoreUtil.getCertificate(this.content);
        }
        catch (CertificateException e) {
            throw new IllegalStateException("Cannot obtain Certificate", e);
        }
    }

    public List<X509Certificate> getCertificates() {
        if (!this.isCertificate()) {
            throw new IllegalStateException("PEM object is not a certificate");
        }
        try {
            return Collections.unmodifiableList(KeystoreUtil.getCertificates(this.content));
        }
        catch (CertificateException e) {
            throw new IllegalStateException("Cannot obtain Certificates", e);
        }
    }

    public RSAPrivateCrtKeySpec getRSAPrivateKeySpec() {
        if (!this.isPrivateKey()) {
            throw new IllegalStateException("PEM object is not a private key");
        }
        try {
            return KeyFactories.RSA_PRIVATE.getKey(this.content);
        }
        catch (IOException | GeneralSecurityException e) {
            throw new IllegalStateException("Cannot obtain PrivateKey", e);
        }
    }

    public RSAPublicKeySpec getRSAPublicKeySpec() {
        if (!this.isPublicKey()) {
            throw new IllegalStateException("PEM object is not a public key");
        }
        try {
            return KeyFactories.RSA_PUBLIC.getKey(this.content);
        }
        catch (IOException | GeneralSecurityException e) {
            throw new IllegalStateException("Cannot obtain PrivateKey", e);
        }
    }

    byte[] getContent() {
        return this.content;
    }

    static enum PemObjectType {
        CERTIFICATE_REQUEST("CERTIFICATE REQUEST"),
        NEW_CERTIFICATE_REQUEST("NEW CERTIFICATE REQUEST"),
        CERTIFICATE("CERTIFICATE"),
        TRUSTED_CERTIFICATE("TRUSTED CERTIFICATE"),
        X509_CERTIFICATE("X509 CERTIFICATE"),
        X509_CRL("X509 CRL"),
        PKCS7("PKCS7"),
        CMS("CMS"),
        ATTRIBUTE_CERTIFICATE("ATTRIBUTE CERTIFICATE"),
        EC_PARAMETERS("EC PARAMETERS"),
        PUBLIC_KEY("PUBLIC KEY"),
        RSA_PUBLIC_KEY("RSA PUBLIC KEY"),
        RSA_PRIVATE_KEY("RSA PRIVATE KEY"),
        EC_PRIVATE_KEY("EC PRIVATE KEY"),
        ENCRYPTED_PRIVATE_KEY("ENCRYPTED PRIVATE KEY"),
        PRIVATE_KEY("PRIVATE KEY");

        private static final PemObjectType[] constants;
        private final String name;

        private PemObjectType(String value) {
            this.name = value;
        }

        public String toString() {
            return this.name;
        }

        public static PemObjectType of(String identifier) {
            Assert.hasText((String)identifier, (String)"Identifier must not be empty");
            for (PemObjectType constant : constants) {
                if (!constant.name.equalsIgnoreCase(identifier)) continue;
                return constant;
            }
            throw new IllegalArgumentException(String.format("No enum constant %s", identifier));
        }

        static {
            constants = PemObjectType.values();
        }
    }
}

