/*
 * Decompiled with CFR 0.152.
 */
package org.apache.celeborn.common.network.client;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.SettableFuture;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.GenericFutureListener;
import java.io.Closeable;
import java.io.IOException;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.Nullable;
import org.apache.celeborn.common.network.buffer.NioManagedBuffer;
import org.apache.celeborn.common.network.client.ChunkReceivedCallback;
import org.apache.celeborn.common.network.client.RpcResponseCallback;
import org.apache.celeborn.common.network.client.TransportResponseHandler;
import org.apache.celeborn.common.network.protocol.OneWayMessage;
import org.apache.celeborn.common.network.protocol.PushData;
import org.apache.celeborn.common.network.protocol.PushMergedData;
import org.apache.celeborn.common.network.protocol.RpcRequest;
import org.apache.celeborn.common.network.protocol.StreamChunkSlice;
import org.apache.celeborn.common.network.protocol.TransportMessage;
import org.apache.celeborn.common.network.util.NettyUtils;
import org.apache.celeborn.common.protocol.MessageType;
import org.apache.celeborn.common.protocol.PbChunkFetchRequest;
import org.apache.celeborn.common.read.FetchRequestInfo;
import org.apache.celeborn.common.write.PushRequestInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TransportClient
implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(TransportClient.class);
    private final Channel channel;
    private final TransportResponseHandler handler;
    private volatile boolean timedOut;
    @Nullable
    private String clientId;
    private static final AtomicLong counter = new AtomicLong();

    public TransportClient(Channel channel, TransportResponseHandler handler) {
        this.channel = (Channel)Preconditions.checkNotNull((Object)channel);
        this.handler = (TransportResponseHandler)Preconditions.checkNotNull((Object)handler);
        this.timedOut = false;
    }

    public Channel getChannel() {
        return this.channel;
    }

    public boolean isActive() {
        return !this.timedOut && (this.channel.isOpen() || this.channel.isActive());
    }

    public SocketAddress getSocketAddress() {
        return this.channel.remoteAddress();
    }

    public void fetchChunk(long streamId, int chunkIndex, long fetchDataTimeout, ChunkReceivedCallback callback) {
        this.fetchChunk(streamId, chunkIndex, 0, Integer.MAX_VALUE, fetchDataTimeout, callback);
    }

    public void fetchChunk(long streamId, int chunkIndex, int offset, int len, long fetchDataTimeout, ChunkReceivedCallback callback) {
        final StreamChunkSlice streamChunkSlice = new StreamChunkSlice(streamId, chunkIndex, offset, len);
        if (logger.isDebugEnabled()) {
            logger.debug("Sending fetch chunk request {} to {}.", (Object)streamChunkSlice, (Object)NettyUtils.getRemoteAddress(this.channel));
        }
        StdChannelListener listener = new StdChannelListener(streamChunkSlice){

            @Override
            protected void handleFailure(String errorMsg, Throwable cause) {
                TransportClient.this.handler.handleFetchFailure(streamChunkSlice, errorMsg, cause);
            }
        };
        long dueTime = System.currentTimeMillis() + fetchDataTimeout;
        FetchRequestInfo info = new FetchRequestInfo(dueTime, callback);
        this.handler.addFetchRequest(streamChunkSlice, info);
        ChannelFuture channelFuture = this.channel.writeAndFlush((Object)new RpcRequest(TransportClient.requestId(), new NioManagedBuffer(new TransportMessage(MessageType.CHUNK_FETCH_REQUEST, PbChunkFetchRequest.newBuilder().setStreamChunkSlice(streamChunkSlice.toProto()).build().toByteArray()).toByteBuffer()))).addListener((GenericFutureListener)listener);
        info.setChannelFuture(channelFuture);
    }

    public long sendRpc(ByteBuffer message, RpcResponseCallback callback) {
        if (logger.isTraceEnabled()) {
            logger.trace("Sending RPC to {}", (Object)NettyUtils.getRemoteAddress(this.channel));
        }
        long requestId = TransportClient.requestId();
        this.handler.addRpcRequest(requestId, callback);
        RpcChannelListener listener = new RpcChannelListener(requestId);
        this.channel.writeAndFlush((Object)new RpcRequest(requestId, new NioManagedBuffer(message))).addListener((GenericFutureListener)listener);
        return requestId;
    }

    public void sendRpc(ByteBuffer message) {
        if (logger.isTraceEnabled()) {
            logger.trace("Sending RPC to {}", (Object)NettyUtils.getRemoteAddress(this.channel));
        }
        long requestId = TransportClient.requestId();
        this.channel.writeAndFlush((Object)new RpcRequest(requestId, new NioManagedBuffer(message)));
    }

    public ChannelFuture pushData(PushData pushData, long pushDataTimeout, RpcResponseCallback callback) {
        return this.pushData(pushData, pushDataTimeout, callback, null);
    }

    public ChannelFuture pushData(PushData pushData, long pushDataTimeout, RpcResponseCallback callback, Runnable rpcSendoutCallback) {
        if (logger.isTraceEnabled()) {
            logger.trace("Pushing data to {}", (Object)NettyUtils.getRemoteAddress(this.channel));
        }
        long requestId = TransportClient.requestId();
        long dueTime = System.currentTimeMillis() + pushDataTimeout;
        PushRequestInfo info = new PushRequestInfo(dueTime, callback);
        this.handler.addPushRequest(requestId, info);
        pushData.requestId = requestId;
        PushChannelListener listener = new PushChannelListener(requestId, rpcSendoutCallback);
        ChannelFuture channelFuture = this.channel.writeAndFlush((Object)pushData).addListener((GenericFutureListener)listener);
        info.setChannelFuture(channelFuture);
        return channelFuture;
    }

    public ChannelFuture pushMergedData(PushMergedData pushMergedData, long pushDataTimeout, RpcResponseCallback callback) {
        if (logger.isTraceEnabled()) {
            logger.trace("Pushing merged data to {}", (Object)NettyUtils.getRemoteAddress(this.channel));
        }
        long requestId = TransportClient.requestId();
        long dueTime = System.currentTimeMillis() + pushDataTimeout;
        PushRequestInfo info = new PushRequestInfo(dueTime, callback);
        this.handler.addPushRequest(requestId, info);
        pushMergedData.requestId = requestId;
        PushChannelListener listener = new PushChannelListener(requestId);
        ChannelFuture channelFuture = this.channel.writeAndFlush((Object)pushMergedData).addListener((GenericFutureListener)listener);
        info.setChannelFuture(channelFuture);
        return channelFuture;
    }

    public ByteBuffer sendRpcSync(ByteBuffer message, long timeoutMs) throws IOException {
        final SettableFuture result = SettableFuture.create();
        this.sendRpc(message, new RpcResponseCallback(){

            @Override
            public void onSuccess(ByteBuffer response) {
                try {
                    ByteBuffer copy = ByteBuffer.allocate(response.remaining());
                    copy.put(response);
                    copy.flip();
                    result.set((Object)copy);
                }
                catch (Throwable t) {
                    logger.warn("Error in responding RPC callback", t);
                    result.setException(t);
                }
            }

            @Override
            public void onFailure(Throwable e) {
                result.setException(e);
            }
        });
        try {
            return (ByteBuffer)result.get(timeoutMs, TimeUnit.MILLISECONDS);
        }
        catch (Exception e) {
            throw new IOException("Exception in sendRpcSync to: " + this.getSocketAddress(), e);
        }
    }

    public void sendRpcSync(ByteBuffer message, final RpcResponseCallback callback, long timeoutMs) throws IOException {
        final SettableFuture result = SettableFuture.create();
        this.sendRpc(message, new RpcResponseCallback(){

            @Override
            public void onSuccess(ByteBuffer response) {
                try {
                    ByteBuffer copy = ByteBuffer.allocate(response.remaining());
                    copy.put(response);
                    copy.flip();
                    callback.onSuccess(copy);
                    result.set(null);
                }
                catch (Throwable t) {
                    logger.warn("Error in responding RPC callback", t);
                    callback.onFailure(t);
                    result.set(null);
                }
            }

            @Override
            public void onFailure(Throwable e) {
                callback.onFailure(e);
                result.set(null);
            }
        });
        try {
            result.get(timeoutMs, TimeUnit.MILLISECONDS);
        }
        catch (Exception e) {
            throw new IOException("Exception in sendRpcSync to: " + this.getSocketAddress(), e);
        }
    }

    public void send(ByteBuffer message) {
        this.channel.writeAndFlush((Object)new OneWayMessage(new NioManagedBuffer(message)));
    }

    public void removeRpcRequest(long requestId) {
        this.handler.removeRpcRequest(requestId);
    }

    public void timeOut() {
        this.timedOut = true;
    }

    @VisibleForTesting
    public TransportResponseHandler getHandler() {
        return this.handler;
    }

    @Override
    public void close() {
        this.timedOut = true;
        this.channel.close().awaitUninterruptibly(10L, TimeUnit.SECONDS);
    }

    public String toString() {
        return "TransportClient[remoteAddress=" + this.channel.remoteAddress() + "clientId=" + this.clientId + ",isActive=" + this.isActive() + "]";
    }

    public static long requestId() {
        return counter.getAndIncrement();
    }

    public String getClientId() {
        return this.clientId;
    }

    public void setClientId(String id) {
        Preconditions.checkState((this.clientId == null || this.clientId.equals(id) ? 1 : 0) != 0, (Object)"Client ID has already been set.");
        this.clientId = id;
    }

    private class PushChannelListener
    extends StdChannelListener {
        final long pushRequestId;
        Runnable rpcSendOutCallback;

        PushChannelListener(long pushRequestId) {
            this(pushRequestId, null);
        }

        PushChannelListener(long pushRequestId, Runnable rpcSendOutCallback) {
            super("PUSH " + pushRequestId);
            this.pushRequestId = pushRequestId;
            this.rpcSendOutCallback = rpcSendOutCallback;
        }

        @Override
        public void operationComplete(Future<? super Void> future) throws Exception {
            super.operationComplete(future);
            if (this.rpcSendOutCallback != null) {
                this.rpcSendOutCallback.run();
            }
        }

        @Override
        protected void handleFailure(String errorMsg, Throwable cause) {
            TransportClient.this.handler.handlePushFailure(this.pushRequestId, errorMsg, cause);
        }
    }

    private class RpcChannelListener
    extends StdChannelListener {
        final long rpcRequestId;

        RpcChannelListener(long rpcRequestId) {
            super("RPC " + rpcRequestId);
            this.rpcRequestId = rpcRequestId;
        }

        @Override
        protected void handleFailure(String errorMsg, Throwable cause) {
            TransportClient.this.handler.handleRpcFailure(this.rpcRequestId, errorMsg, cause);
        }
    }

    public class StdChannelListener
    implements GenericFutureListener<Future<? super Void>> {
        final long startTime = System.currentTimeMillis();
        final Object requestId;

        public StdChannelListener(Object requestId) {
            this.requestId = requestId;
        }

        public void operationComplete(Future<? super Void> future) throws Exception {
            if (future.isSuccess()) {
                if (logger.isDebugEnabled()) {
                    long timeTaken = System.currentTimeMillis() - this.startTime;
                    logger.debug("Sending request {} to {} took {} ms", new Object[]{this.requestId, NettyUtils.getRemoteAddress(TransportClient.this.channel), timeTaken});
                }
            } else {
                String errorMsg = String.format("Failed to send request %s to %s: %s, channel will be closed", this.requestId, NettyUtils.getRemoteAddress(TransportClient.this.channel), future.cause());
                logger.warn(errorMsg);
                TransportClient.this.channel.close();
                try {
                    this.handleFailure(errorMsg, future.cause());
                }
                catch (Exception e) {
                    logger.error("Uncaught exception in RPC response callback handler!", (Throwable)e);
                }
            }
        }

        protected void handleFailure(String errorMsg, Throwable cause) {
            logger.error("Error encountered " + errorMsg, cause);
        }
    }
}

