/*
 * Decompiled with CFR 0.152.
 */
package org.apache.dolphinscheduler.plugin.task.api.utils;

import io.fabric8.kubernetes.client.dsl.LogWatch;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.ServiceLoader;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import lombok.Generated;
import lombok.NonNull;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.SystemUtils;
import org.apache.dolphinscheduler.common.thread.ThreadUtils;
import org.apache.dolphinscheduler.common.utils.OSUtils;
import org.apache.dolphinscheduler.common.utils.PropertyUtils;
import org.apache.dolphinscheduler.plugin.task.api.K8sTaskExecutionContext;
import org.apache.dolphinscheduler.plugin.task.api.TaskConstants;
import org.apache.dolphinscheduler.plugin.task.api.TaskException;
import org.apache.dolphinscheduler.plugin.task.api.TaskExecutionContext;
import org.apache.dolphinscheduler.plugin.task.api.am.ApplicationManager;
import org.apache.dolphinscheduler.plugin.task.api.am.KubernetesApplicationManager;
import org.apache.dolphinscheduler.plugin.task.api.am.KubernetesApplicationManagerContext;
import org.apache.dolphinscheduler.plugin.task.api.am.YarnApplicationManagerContext;
import org.apache.dolphinscheduler.plugin.task.api.enums.ResourceManagerType;
import org.apache.dolphinscheduler.plugin.task.api.enums.TaskExecutionStatus;
import org.apache.dolphinscheduler.plugin.task.api.utils.LogUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ProcessUtils {
    @Generated
    private static final Logger log = LoggerFactory.getLogger(ProcessUtils.class);
    private static final Integer SHELL_KILL_WAIT_TIMEOUT = PropertyUtils.getInt((String)"shell.kill.wait.timeout", (int)10);
    private static final Map<ResourceManagerType, ApplicationManager> applicationManagerMap = new HashMap<ResourceManagerType, ApplicationManager>();
    private static final Pattern MACPATTERN;
    private static final Pattern WINDOWSPATTERN;
    private static final Pattern LINUXPATTERN;
    private static final Pattern PID_PATTERN;

    private ProcessUtils() {
        throw new IllegalStateException("Utility class");
    }

    public static boolean kill(@NonNull TaskExecutionContext request) {
        if (request == null) {
            throw new NullPointerException("request is marked non-null but is null");
        }
        try {
            log.info("Begin killing task instance, processId: {}", (Object)request.getProcessId());
            int processId = request.getProcessId();
            if (processId == 0) {
                log.info("Task instance has already finished, no need to kill");
                return true;
            }
            List<Integer> pidList = ProcessUtils.getPidList(processId);
            boolean gracefulKillSuccess = ProcessUtils.sendKillSignal("SIGINT", pidList, request.getTenantCode());
            if (gracefulKillSuccess) {
                log.info("Successfully killed process tree using SIGINT, processId: {}", (Object)processId);
                return true;
            }
            boolean termKillSuccess = ProcessUtils.sendKillSignal("SIGTERM", pidList, request.getTenantCode());
            if (termKillSuccess) {
                log.info("Successfully killed process tree using SIGTERM, processId: {}", (Object)processId);
                return true;
            }
            log.warn("SIGINT & SIGTERM failed, using SIGKILL as a last resort for processId: {}", (Object)processId);
            boolean forceKillSuccess = ProcessUtils.sendKillSignal("SIGKILL", pidList, request.getTenantCode());
            if (forceKillSuccess) {
                log.info("Successfully sent SIGKILL signal to process tree, processId: {}", (Object)processId);
            } else {
                log.error("Error sending SIGKILL signal to process tree, processId: {}", (Object)processId);
            }
            return forceKillSuccess;
        }
        catch (Exception e) {
            log.error("Kill task instance error, processId: {}", (Object)request.getProcessId(), (Object)e);
            return false;
        }
    }

    private static boolean sendKillSignal(String signal, List<Integer> pidList, String tenantCode) {
        if (pidList == null || pidList.isEmpty()) {
            log.info("No process needs to be killed.");
            return true;
        }
        List<Integer> alivePidList = ProcessUtils.getAlivePidList(pidList, tenantCode);
        if (alivePidList.isEmpty()) {
            log.info("All processes already terminated.");
            return true;
        }
        String pids = alivePidList.stream().map(String::valueOf).collect(Collectors.joining(" "));
        try {
            String killCmd = String.format("kill -s %s %s", signal, pids);
            killCmd = OSUtils.getSudoCmd((String)tenantCode, (String)killCmd);
            log.info("Sending {} to process group: {}, command: {}", new Object[]{signal, pids, killCmd});
            OSUtils.exeCmd((String)killCmd);
            long timeoutMillis = TimeUnit.SECONDS.toMillis(SHELL_KILL_WAIT_TIMEOUT.intValue());
            long startTime = System.currentTimeMillis();
            while (!alivePidList.isEmpty() && System.currentTimeMillis() - startTime < timeoutMillis) {
                alivePidList.removeIf(pid -> !ProcessUtils.isProcessAlive(pid, tenantCode));
                if (alivePidList.isEmpty()) continue;
                ThreadUtils.sleep((long)1000L);
            }
            if (alivePidList.isEmpty()) {
                log.debug("Kill command: {}, kill succeeded", (Object)killCmd);
                return true;
            }
            String remainingPids = alivePidList.stream().map(String::valueOf).collect(Collectors.joining(" "));
            log.info("Kill command: {}, timed out, still running PIDs: {}", (Object)killCmd, (Object)remainingPids);
            return false;
        }
        catch (Exception e) {
            log.error("Error sending {} to process: {}", new Object[]{signal, pids, e});
            return false;
        }
    }

    private static List<Integer> getAlivePidList(List<Integer> pidList, String tenantCode) {
        return pidList.stream().filter(pid -> ProcessUtils.isProcessAlive(pid, tenantCode)).collect(Collectors.toList());
    }

    private static boolean isProcessAlive(int pid, String tenantCode) {
        try {
            String checkCmd = String.format("kill -0 %d", pid);
            checkCmd = OSUtils.getSudoCmd((String)tenantCode, (String)checkCmd);
            OSUtils.exeCmd((String)checkCmd);
            return true;
        }
        catch (Exception e) {
            return false;
        }
    }

    public static List<Integer> getPidList(int processId) throws Exception {
        String rawPidStr;
        try {
            if (SystemUtils.IS_OS_MAC) {
                rawPidStr = OSUtils.exeCmd((String)String.format("%s -sp %d", "pstree", processId));
            } else if (SystemUtils.IS_OS_LINUX) {
                rawPidStr = OSUtils.exeCmd((String)String.format("%s -p %d", "pstree", processId));
            } else {
                log.warn("Unsupported OS for pstree: {}. Attempting generic command.", (Object)SystemUtils.OS_NAME);
                rawPidStr = OSUtils.exeCmd((String)String.format("%s -p %d", "pstree", processId));
            }
        }
        catch (Exception ex) {
            log.error("Failed to execute 'pstree' command for process ID: {}", (Object)processId, (Object)ex);
            throw ex;
        }
        String pidsStr = ProcessUtils.parsePidStr(rawPidStr);
        if (StringUtils.isBlank((CharSequence)pidsStr)) {
            log.warn("No PIDs found for process: {}", (Object)processId);
            return Collections.emptyList();
        }
        String[] pidArray = PID_PATTERN.split(pidsStr.trim());
        if (pidArray.length == 0) {
            log.warn("No PIDs parsed for process: {}", (Object)processId);
            return Collections.emptyList();
        }
        ArrayList<Integer> pidList = new ArrayList<Integer>();
        for (String pidStr : pidArray) {
            if (StringUtils.isBlank((CharSequence)(pidStr = pidStr.trim()))) {
                log.error("Empty or blank PID found in output for process: {}, full PIDs string: {}", (Object)processId, (Object)pidsStr);
                throw new IllegalArgumentException("Empty or blank PID found in output from process: " + processId);
            }
            try {
                int pid = Integer.parseInt(pidStr);
                if (pid <= 0) {
                    log.error("Invalid PID value (must be positive): {} for process: {}, full PIDs string: {}", new Object[]{pidStr, processId, pidsStr});
                    throw new IllegalArgumentException("Invalid PID value (must be positive): " + pid);
                }
                pidList.add(pid);
            }
            catch (NumberFormatException e) {
                log.error("Invalid PID format in output: {} for process: {}, full PIDs string: {}", new Object[]{pidStr, processId, pidsStr, e});
                throw new IllegalArgumentException("Invalid PID format in output: '" + pidStr + "' (from process " + processId + ")", e);
            }
        }
        return pidList;
    }

    public static String parsePidStr(String rawPidStr) {
        log.info("prepare to parse pid, raw pid string: {}", (Object)rawPidStr);
        ArrayList<String> allPidList = new ArrayList<String>();
        Matcher mat = null;
        if (SystemUtils.IS_OS_MAC) {
            if (StringUtils.isNotEmpty((CharSequence)rawPidStr)) {
                mat = MACPATTERN.matcher(rawPidStr);
            }
        } else if (SystemUtils.IS_OS_LINUX) {
            if (StringUtils.isNotEmpty((CharSequence)rawPidStr)) {
                mat = LINUXPATTERN.matcher(rawPidStr);
            }
        } else if (StringUtils.isNotEmpty((CharSequence)rawPidStr)) {
            mat = WINDOWSPATTERN.matcher(rawPidStr);
        }
        if (null != mat) {
            while (mat.find()) {
                allPidList.add(mat.group(1));
            }
        }
        return String.join((CharSequence)" ", allPidList).trim();
    }

    public static void cancelApplication(TaskExecutionContext taskExecutionContext) {
        try {
            if (Objects.nonNull(taskExecutionContext.getK8sTaskExecutionContext())) {
                if (!TaskConstants.TASK_TYPE_SET_K8S.contains(taskExecutionContext.getTaskType())) {
                    applicationManagerMap.get((Object)ResourceManagerType.KUBERNETES).killApplication(new KubernetesApplicationManagerContext(taskExecutionContext.getK8sTaskExecutionContext(), taskExecutionContext.getTaskAppId(), ""));
                }
            } else {
                List<String> appIds;
                String host = taskExecutionContext.getHost();
                String executePath = taskExecutionContext.getExecutePath();
                String tenantCode = taskExecutionContext.getTenantCode();
                if (StringUtils.isNotEmpty((CharSequence)taskExecutionContext.getAppIds())) {
                    appIds = Arrays.asList(taskExecutionContext.getAppIds().split(","));
                } else {
                    String logPath = taskExecutionContext.getLogPath();
                    String appInfoPath = taskExecutionContext.getAppInfoPath();
                    if (logPath == null || appInfoPath == null || executePath == null || tenantCode == null) {
                        log.error("Kill yarn job error, the input params is illegal, host: {}, logPath: {}, appInfoPath: {}, executePath: {}, tenantCode: {}", new Object[]{host, logPath, appInfoPath, executePath, tenantCode});
                        throw new TaskException("Cancel application failed!");
                    }
                    log.info("Get appIds from worker {}, taskLogPath: {}", (Object)host, (Object)logPath);
                    appIds = LogUtils.getAppIds(logPath, appInfoPath, PropertyUtils.getString((String)"appId.collect", (String)"log"));
                    taskExecutionContext.setAppIds(String.join((CharSequence)",", appIds));
                }
                if (CollectionUtils.isEmpty(appIds)) {
                    log.info("The appId is empty");
                    return;
                }
                ApplicationManager applicationManager = applicationManagerMap.get((Object)ResourceManagerType.YARN);
                applicationManager.killApplication(new YarnApplicationManagerContext(executePath, tenantCode, appIds));
                log.info("yarn application [{}] is killed or already finished", appIds);
            }
        }
        catch (Exception e) {
            log.error("Cancel application failed: {}", (Object)e.getMessage());
        }
    }

    public static TaskExecutionStatus getApplicationStatus(K8sTaskExecutionContext k8sTaskExecutionContext, String taskAppId) {
        if (Objects.isNull(k8sTaskExecutionContext)) {
            return TaskExecutionStatus.SUCCESS;
        }
        KubernetesApplicationManager applicationManager = (KubernetesApplicationManager)applicationManagerMap.get((Object)ResourceManagerType.KUBERNETES);
        return applicationManager.getApplicationStatus(new KubernetesApplicationManagerContext(k8sTaskExecutionContext, taskAppId, ""));
    }

    public static LogWatch getPodLogWatcher(K8sTaskExecutionContext k8sTaskExecutionContext, String taskAppId, String containerName) {
        KubernetesApplicationManager applicationManager = (KubernetesApplicationManager)applicationManagerMap.get((Object)ResourceManagerType.KUBERNETES);
        return applicationManager.getPodLogWatcher(new KubernetesApplicationManagerContext(k8sTaskExecutionContext, taskAppId, containerName));
    }

    public static void removeK8sClientCache(String taskAppId) {
        KubernetesApplicationManager applicationManager = (KubernetesApplicationManager)applicationManagerMap.get((Object)ResourceManagerType.KUBERNETES);
        applicationManager.removeCache(taskAppId);
    }

    static {
        ServiceLoader.load(ApplicationManager.class).forEach(applicationManager -> applicationManagerMap.put(applicationManager.getResourceManagerType(), (ApplicationManager)applicationManager));
        MACPATTERN = Pattern.compile("-[+|-][-|=]\\s(\\d+)");
        WINDOWSPATTERN = Pattern.compile("\\((\\d+)\\)");
        LINUXPATTERN = Pattern.compile("\\((\\d+)\\)");
        PID_PATTERN = Pattern.compile("\\s+");
    }
}

