// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.apache.doris.fs.obj;

import org.apache.doris.backup.Status;
import org.apache.doris.datasource.property.storage.AzureProperties;
import org.apache.doris.datasource.property.storage.S3Properties;
import org.apache.doris.datasource.property.storage.StorageProperties;
import org.apache.doris.fs.remote.RemoteFile;

import com.azure.core.http.HttpHeaders;
import com.azure.core.http.HttpRequest;
import com.azure.core.http.rest.PagedResponse;
import com.azure.core.util.IterableStream;
import com.azure.storage.blob.BlobContainerClient;
import com.azure.storage.blob.models.BlobItem;
import com.azure.storage.blob.models.BlobItemProperties;
import com.azure.storage.blob.models.ListBlobsOptions;
import mockit.Mock;
import mockit.MockUp;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.jupiter.api.Assertions;

import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.PathMatcher;
import java.nio.file.Paths;
import java.time.OffsetDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

public class AzureObjStorageTest {
    public static class I {
        public String pattern;
        public Long expectedMatchSize;

        public I(String pattern, long expectedMatchSize) {
            this.pattern = pattern;
            this.expectedMatchSize = expectedMatchSize;
        }
    }

    public static List<I> genInputs() {
        // refer genObjectKeys
        List<I> inputs = new ArrayList<I>();
        inputs.add(new I("s3://gavin-test-jp/azure-test/1/*/tmp*", 8196L));
        inputs.add(new I("s3://gavin-test-jp/azure-test/1/tmp*", 4098L));
        inputs.add(new I("s3://gavin-test-jp/azure-test/1/*tmp*", 4098L));
        inputs.add(new I("s3://gavin-test-jp/azure-test/1/**/tmp*", 20490L));
        inputs.add(new I("s3://gavin-test-jp/azure-test/**/tmp*", 32784L));
        inputs.add(new I("s3://gavin-test-jp/azure-test/*", 3L)); // no files at 1st level
        inputs.add(new I("s3://gavin-test-jp/azure-test/2/*", 4098L));
        inputs.add(new I("s3://gavin-test-jp/azure-test/2*/*", 4098L));
        inputs.add(new I("s3://gavin-test-jp/azure-test/2/*I*", 591L));
        inputs.add(new I("s3://gavin-test-jp/azure-test/1", 1L));
        inputs.add(new I("s3://gavin-test-jp/azure-test/2", 1L));
        inputs.add(new I("s3://gavin-test-jp/azure-test/3", 1L));
        inputs.add(new I("s3://gavin-test-jp/azure-test/1/tmp.k*", 61L));
        inputs.add(new I("s3://gavin-test-jp/azure-test/1/tmp.[a-z]*", 1722L));
        inputs.add(new I("s3://gavin-test-jp/azure-test/[12]/tmp.[a-z]*", 3444L));
        inputs.add(new I("s3://gavin-test-jp/azure-test/1/2/tmp.ehJi0Y5QKt", 1L));
        inputs.add(new I("s3://gavin-test-jp/azure-test/non-existed-dir", 0L));
        inputs.add(new I("s3://gavin-test-jp/azure-test/1/2/non-existed-file", 0L));
        inputs.add(new I("s3://gavin-test-jp/azure_backup/__palo_repository_azure_repo1/__ss_*/*", 1L));

        return inputs;
    }

    // this case relies on real azure client and issue HTTP requests, ignore it by default
    // run it if you have storage account(ak), key(sk) and container(bucket)
    @Test
    @Ignore
    public void testGlobList() {
        Map<String, String> props = new HashMap<String, String>();
        props.put(S3Properties.ACCESS_KEY, "${account_name}");
        props.put(S3Properties.SECRET_KEY, "${key}");
        props.put(S3Properties.ENDPOINT, "https://bucket.blob.core.windows.net");
        props.put(S3Properties.BUCKET, "${container}");

        List<I> inputs = genInputs();
        inputs.stream().forEach(i -> {
            AzureProperties azureProps = (AzureProperties) StorageProperties.createPrimary(props);
            AzureObjStorage azs = new AzureObjStorage(azureProps);
            List<RemoteFile> result = new ArrayList<RemoteFile>();
            boolean fileNameOnly = false;
            // FIXME(gavin): Mock the result returned from azure blob to make this UT work when no aksk and network
            Status st = azs.globList(i.pattern, result, fileNameOnly);
            Assertions.assertTrue(st.ok());
            Assertions.assertEquals(i.expectedMatchSize, result.size());
        });
    }

    @Test
    public void testGlobListWithMockedAzureStorage() {
        Map<String, String> props = new HashMap<String, String>();
        props.put(S3Properties.ACCESS_KEY, "gavintestmocked");
        props.put(S3Properties.SECRET_KEY, "sksks");
        props.put(S3Properties.ENDPOINT, "https://bucket.blob.core.windows.net");
        props.put(S3Properties.BUCKET, "gavin-test-mocked");

        List<I> inputs = genInputs();
        inputs.stream().forEach(i -> {
            AzureObjStorage azs = genMockedAzureObjStorage(4/*numBatches, numContinuations*/);
            List<RemoteFile> result = new ArrayList<>();
            boolean fileNameOnly = false;
            // FIXME(gavin): Mock the result returned from azure blob to make this UT work when no aksk and network
            Status st = azs.globList(i.pattern, result, fileNameOnly);
            System.out.println("testGlobListWithMockedAzureStorage pattern: " + i.pattern + " matched " + result.size());
            Assertions.assertTrue(st.ok());
            Assertions.assertEquals(i.expectedMatchSize, result.size());
            for (int j = 0; j < result.size() && j < 10; ++j) {
                System.out.println(result.get(j).getName());
            }

        });
    }

    @Test
    public void testFsGlob() {
        for (I i : genInputs()) {
            String pattern = i.pattern.substring(19); // remove prefix s3://gavin-test-jp/
            PathMatcher matcher = FileSystems.getDefault().getPathMatcher("glob:" + pattern);
            List<String> matchedPaths = new ArrayList<>();
            HashSet<String> directories = new HashSet<>();
            for (String p : genObjectKeys()) {
                java.nio.file.Path blobPath = Paths.get(p);

                while (blobPath != null) {
                    if (matcher.matches(blobPath) && !directories.contains(blobPath.toString())) {
                        matchedPaths.add(blobPath.toString());
                        directories.add(blobPath.toString());
                    }
                    blobPath = blobPath.getParent();
                }
            }
            System.out.println("pattern: " + i.pattern + " matched " + matchedPaths.size());
            System.out.println("====================");
            Assertions.assertEquals(i.expectedMatchSize, matchedPaths.size());
        }
    }

    public static List<String> genObjectKeys() {
        String fileSuffixes = "zMkg8GtbSL MaSolJd8sL gtujCOzywm XBSnMwdoa9 FTxFgeihCa k9mp5K23pY dxjWuBC0dz osaFdxo2mz h4PEGleain aGxFstkIZz lGZbBe35uE gRILmhSPVm Ta3S5IUQiC 730eZqyhum XUhMhqk0DG cL7VKXMQuN DOb5J5MUFV i2dg8BSuTE FIzNOcoekf N24tUXQ8ws c31UcFGP1S S7bkoinC5u CCOKe8YL1N b5qGztFktP C7M1G8mYIP 0JA5yp6VAt lfJpMUYDe5 vXVOGgbZD3 UtM44M7c6F gPcli9bsQ4 cKP1TAjXfC v5k8Wksy9k UNkJtnIWwV 4UB0gbsI9g U4yKzgRbkG XcYaYUubBH bNVYZKq422 AvUoK2bcwS 5t6QDuG5ox Hj8lPHhsZL 08KZbcvC8P 5nYfmpu0Xj dc3VlEN0AS qNZ2I3We3o kMrAdKaEMi sZdFapPTen 0gUMPnQgUY RuRWUuDtpJ OD1QTjrbsv zbIBlZnqXv KUHTedvZrl lmdY0JN50o A4Nlw4uLMK 2oj8EUcjRp tnRP3ZZzBB hDbrkXpEjo ct6ANts67r OZQUcEpcaj 6YRXx10gMD NBga9PU9fv uFxmvtDVbi WvRfkt7Mfh fKRBzT6Sd0 lem9JT5LQ7 Eo7kTuRAws ZeeCMcUmn9 vmgtVQoTtY XmmXjArRkM BNGOY9ISbF Peiddn6Do1 Ztb9FmbxOz nflv6FWhRZ H7xLv79i7y JvCngH1Hbn KzjZbezesb mTbKuZvBST GnR3Fa8sEa SmeUeQHbok h4M172XgKe lvDzmGiAN0 1qVazOgM0N JPWFf5s2hN MUoer5kl2h QCO1QBmB0V 5puvqKkFNH AebQ7k7rGP KtT4mYN4fc LQXh3bZnQg 2oGVUnq4Yz Abla8vQFhO 7DAeCnzWtf kBBW8waA2C CClJkqmMj4 UHziI0EX1s dKuM2ler6P RbZmcXp6o1 5xtwykpwIE GF8y26OQ1n nmtx0Ffphx 1X04f1FKYh bayvAUOr0n BXDjwCbtnH 84N7Pkz7lz uaNn2aPzY3 ruL7x58gi2 GHKT28txuB lgdtaQ9MXS 2pVFkzraLF 78adVyR8vK Gj9ajJUFEl 1pVub3Wfdq 5s8qtDGJyR TbypY3CwB4 D1oMMLBZuH zebvWHPOuY 9lNTAmohtk NE9xMz8BLR nNJVR0qy4s i2q6Bql8A4 XvAkbUJFJT RtOjWL1ty3 6ynujQyXkr h8JePlloXJ vtLSCTdqnx 9PDjcQ2fVh lJdkSjdMfM P4GSuAxSYt ADWADMRtQ4 4pnqGh4Kid Jo2INGxsWc fh4CEGcoM2 8TjRtglqK4 Xvy6u94pti z5P0JE25He WJZvWvqLq2 OHRZ049TgN YrXJXSOBDY D69BXrfc1g Y25VfZauQc MoapvSVNUi LxM0qE4FUI 65BuMVejjW vu8iaE1WMo C3uvhMSpbE tN9lJDjoKe 6GNwnqnEJy KT7pH4FIs5 mqAuiIgdvo AfZ8bf6svY wPl6VxmrA4 oAHXR1Vn9z rpZyD1xGBE yblSUoayVV jrzGJcCJGf dhlOMreZv2 C292YVQttn ntDs7CeT5e 3RLd6SWhTw Fdx5suC1Qr PfVWczs3Lr jrRKQbhL9T vdOt0QkBdl XSi8aIbs77 d6b8wQ0Suk pdSKt7LQtN L9FHSiSwnB TsKfJdMJ4N yoOoBB96V2 a0EvLjGpsr dZzedcmBIW q6TGZasadu 7vs0thA9RH TCD30sEcUk c3IDN5KR41 1XHi7xHJIV soHcBtaSVk UPPmEPmOKi jfMRz3PFNs mcABnbd8yG 21ukYd9ZId Lgtpt0fJXJ LfID20dEsX AbCKea8QBi 9gBnUybFeh Kch0L1mcZj XTsr45dRnv y2JF6VGtTJ omNG5rpDrU heyQ5MBoqH qWFR2d9T1O Re0eJCu7Kd nFivVxwGfX iTDwKvGyKp S2eZCPJCuR xAyZtxMJV3 w22VbEZgxg 8lEm9C307C 6WPM66qi3h 4hPqw6ixU4 pQKtbeSklL goJD7MM0Y6 A4vNARsWFr ryK6r6SWc7 waRT3XC977 w9iTRvkDVM dP5tuNanJ4 vSGgvlHPq5 q8N5CHPo5T LdJikivJcq Lrb4pJsq4b ErHRyCEVr2 wYDu8X22ga PYfYpMffVt axO9RRsCWC 6Z1tOtz6lZ ff79XoUgWy 43cqenGQhh Y4U8BelP1d ZvGEDXVUpW Cqf8pemFKh aVyS2A3cAs dsdfqdyre7 xkZ6A1lQDg FjK4S62NcF x5hfLrknTi pk90m5FcYy OPSRvx7ct2 3PRfaYhL2R bmSPke5wpp hsnHAl7QvF rNlr2jTeDo JyZQ1BcuZ9 zvGZ2hj8wZ mKKHo707MH FovC4gTmRQ jyZovyawXP JzBzUVM5OO mw0bLNzfIj Jt6sRXZKO4 avNgrXcUeM EwKEwjKIru ylIhUDoF56 RQvAKADiuB V2MYn0vfV7 5UcFy4qcda T4s9p13x7r Z72U9EnLQp y0VyPlAjII 6ZtU3I17CO gspSyG02cz WGcB7NqG4I gAJdqFbza1 56a6cKSbnZ fJHRwYZw5B Ud3BPqn53w gKK5WDJUUd wYmNpCoq4I nEF4j4Z1Iy nrYoltub9Y FhahXsdmvH 33oJYhUsu3 DKufRQyYCD 5HQ7aqGA17 0V8dQgxilD sFeYMUnOtI L31awlFfQb NTN8p17rfl xi1hyClCMJ 38YfsMhOOD OuPtNELEWi VYzEuthKgf a8YIeWSVsO h7drDjwg1C JgfbRRivjT bHykHeZtvg jAAjV0fLnm WRabqZLsJw CsJhLgYbJS oRQtkFQbav pTkiVFHijj DMfCGsPoLC 8ib6SjXEcK rLFgUT8o7x JNVBbEDoLu Rn9MJNyf1x lF6DBsy1QY 8zyj4qTtMR mGRgetv5ZR WazA7d4r1z 3VzEpqN7Ps iBc3HNTBkF LY291QAFma aClRF4RicX d9PQI8yiuz la8tppH1Zw ZUo9lVXbah zNfe8cWbYj H1EeqQD7Yo ELKnFsWiS4 2Ljmhpbz8j GLQBdae6Dt j7f2HsPtdm QL8qWdWOZ7 nbcmqntGFd qPaL0BYiG0 BWm5GtkkVu PYZUHYD3s6 9HSG3J86Gn kmlbrU2UAH 0nsePJrocF gq1X7iJJz9 pGoPGolJ59 78d8KZRh00 zvdz6DoFKx wzEvj8RsHx 2S6EHEkDjt sK9o06Prjm xM8w2HbR2K sUEDn5OAAg PDzP2uUbOd xqle4HWkGn yQLrvzvUx7 dv8E3E425U GUgGRuzgtP PJndR9lWs2 ppq0Hv2jm2 83sd8e79kc 89zXvYC3Cl Fyf7S4A1jm kg7MjNAs1D lNymss0wX3 TbHhDWQoFQ X5u3Lu6Hay XJ4seymZSo rTgr3tX9Ss UOJRTEKtLO nZj1wSviq1 aTVeSN0QPD iRDTRwJs27 HZnJYcDpSE WABzQJJvMD Wc129PMQHa 1hTt82uyYi BXsfk6cX75 WQgYSPMeoi OVEztZHWO7 rw6ST9Bljt yvXiZTxfB0 aUClbX41DI 11YAB23Hui TIUKcYoluD fNd9hYE0ue Ga0nJj5D0u 9xTlL6zBeC hKCCuCKNvh t3LIiouEzt sMt53Sr797 N9pvowTMrU GWgvEf4ler 7Hl2Mi46fO W1TjeKpwKQ 8bcdKhCgcQ CXpeNQYsV1 I1AVHopFP7 zIk2rjL1Z2 RHWZXMGPoq M2vcxOhwVB wtem5QOdAB 40MgCQzoXH vNzYFUCJjc Nfp5RFCSlT FAQrbBHJM7 94B4di9g5H NdCMyb4mQv pfwKKfALJP sIjcWKm0pt xfw0idAQ8A dYy5MaIbmu V5xc5qQZZh o69ip04Mi1 jGQASDo0lL uIj5j8OK2Y US422Y8moB HmXi3Kn3Ie 3KzLbQzRsZ 92VshFV5ka fkNyZwNsgz BloNg9EzOX zicAA5JwJ3 7MSr0LGnsx 6itAntAR7C mOv89y0Dyy IMFL4X53Dh 02PN2tfIlW AEdAC7gvb0 0xuXiziiNF Cn9J0h7X7I z7dMIObXfy UAuewcsuWL Czq3NoZi4s 7IOwu5al5U R1YDEkRYir srSpufrQCY t41Ok7Ifrq NAoYhHIFR1 LO3OTnAPtr AOjkxveWFH Pvl0zQejbg SYeccuAbVa jC6kOsdjb4 wG36HL8Oz6 okcXyWs7Pp miugL9t1Js 2LSUGKeeaI NyKxgNxl2w ZmpcsZ5DMn xuObyw75Fl BaKZSX9zSC BPtEhpd0XV AKwVPVvV71 WG4tJ7Qesh 2q5hxMsXxF 72XJd5An4Z imdYGoda9V HK9sW38jpe Og5hFvyoKA 8z0HLdhwMT nP7uYp4NAF q86GTiLD7f 84DcBcADma ujz5oEp9qj cCtKR8SGyF lQ85QoImTS 689cw6yawP K7gBZe1rNR cheuChdkMa ApFDXy2ycP fHo3YtRkEN nFCWb0O2ci EpmmslzHg2 TcsX3CjvFw hFnYhLWSPF 7b9wZb9VHk e411hDGz1B JewizA5IxC frPR73BVUQ pMxs29vOqI yS3HPAJu2n FsPYYHx8SN NzKgyaHStt XjgieUSiUJ yVYZv3ATcm GomnnEm4Uy mm9uWhA1zP oBZDOdPP6s Hmw9z5OgWK jMfC0bUQKC mKymgokFzw YuuxzdtRKZ mcqAYtXp1o zTvvF7Y4L3 foi1GlMar9 ps5fOZkjve HTIWdQRwoN zmYDB41deR 6CC3kJTv8D CBAEREzchZ vf1UDzS3JA cspUo7UJ96 Py9JmSw4VF h34vVtITAG HCiOTK3L5b bkj0y6wS2k a2U7tHAZ3C OeP38J46vk 4fBHgM0QpP bIYAdO6w3r RfUwy7q5ps Ks5p9bzuJ3 Axm4R5fbkt aSS22ayaES 6CSWRCrUJV ZYO9l7cNbL Sgj9j2tCCa 6BWY5PynXw 7r6CUlcnoq QVRI9hVCNh Ey5TBK6LmY 2dGb8wjwwp g2xFrDVxCr NYaD2jvRcl 7xQZzdQoNg twxjMcOcF7 fwHuh6w9jg bwow4nk4sI A4eWDjg4sd jxYdpHEXOw k7exhYefDD UWe8YswYy4 GsAnluEkAv 7gASLSMOcN WCw7yG3Ekd sdYquMbAIT CMskzLWQEP B36idlJesz oQKguBnrKN hdVwqtCuJy u78svmwZSF 8oS2jHRMJO 0ngqaNTcKj zwsCeuJxai rVtNgtld2g mtfz6IaHQt qd1Kh3z6Ko 55LT8euheh JjLVmHHyZl 8GNsFO4bmx 65LSMa5sWU JGOSEx5CjH NGr7q5jCab kABacpX8oq JLOklHHDw1 ykXpNg7Lmv ttPuWnMohz kbAoiH44VC SbIDf0tiGs Uaxuua6ixL 1dxobmQNOC oqdkMb6CjJ 7zyPnX5bsY fZ1WXcd81i Aer71nqRBJ sQylB2UCuB AoYBgKo4Hn 1qN3EUsnwY Rh5sXPyvSP Ap9caWIUws QxmtOajEYp cs90IoYb8C kXCdHb3VZT 0VKhQt7ntb 8ynfA418KR ws17Pf7QCb qmIpglLHH0 tK7GXi16Ky NWkL104YsZ eGHKaJX4sm 7Xjz7xFmq7 UHluqWSFq5 pL7693Gfys QVm9unUba1 Bk6WNk0feM U6ZResxwkW c7qzciGi7P ULSoXkABsZ Z3HY6aY4PU 2XtXV3fU4n ZqU0aEa04Q obRoZU6voS dllwgQy2Wy e5qMoVvzPY kzfodgBAy7 1J9CwTuRKE CED3UV7ScI QMpZleIPo5 uZVaccdsZz zOtMwK9pWv HAVnfYQSes Su9729zNaz Q2AdM5BwEx 7KYXUvG6Hj sW12UOXYfV 45aXFpBQV3 hndRIQU6K4 YCPY1FtMD2 hAKyr5SJab L8j9ks50tG jnKTNjaa89 zGVhH5ROtn ACEBqnljbM Gr4n87W7lf AgPQpkM4Az fpVzxqL5RV 4owC80duvX YrRbdQPymI Mt9UNHMcXD wyvkcx9567 JQ9E4Dt1A6 gv9x6fHXw3 L4TCnXjgkr 4wrNjVLvcs lnamswuHJ3 k4pDfm7mpM QaTtRS3QMO 3QEQJBvpCF JdcBGnCd2i WJEXx3NtwR TVpTcCEjyX wuFIJd1DzG b5u6Dr6XUd 6KfaHYykbY R0qEmdSWkk jT7OmngJhj fuFz85csRw JDZgGNQYkv RCOOP5quu1 Xaof7hH675 4UPbIjK4bc IJhVc0Kd79 D9gAJpnBsr OzyQ3D5aX4 eXiUuTYwdP uFrUe1i7Ye GQ03Q9TUyA WWcSN0fk9x m6DuDR603o N9PlLt2ZNL v8IWJMYljM NlwTk5GFJA a9qlSEehKu UsEHkUKUBo E6zSaJzNJQ Mxta2HtMac 6Plxsg74GZ wgqzpNN0xB TFcatRbSmR uXE3H4Yx3P jfwgLyLdTv ReBdjuvaUI twSJq4iIJA x6BpKCLHCH Q4tLbiwaSq VZMWrw7biL 6VVApia9F1 LNxnN60ttg n0I1UcYxBH FUaPaZQTcw yIxI8Ty2FU mrNYOa7auE 5mwUZoFvPP 5rekxFRLJT UlOrpN1XAj 3wla8LNJTQ OqiAnNB83F mfAbfuWtd5 kdDTzPu6Gw 7wWvKSRCQE 1rq8BKv7pF P2LJhN5N3g cvUAVXEAM2 W1NPof8T4Q FSViduTU1S CKeqNHzp2b yBZ6wdRgzq ngW72wZAx3 SiTB6Q4z75 dMxM2f1N0O 57AkCA6Vpv 5vTRUvns0x QX41o9NX6Y 2ChXi5nMez jD3N108dft AqzAqibR2H UwHsnKRnFn 2atg1HJ2GS xnGkstmsQx f1pqovcvbs Ap9HYSz4fc duHOLzua7I mjdd7wgw92 iERKCldUZT f22ouPQKuY rMcAEhPl8S Zmr9irIfuc OK9KyWhqqg 2P3EO8jheT yxwJ6f0IdF MFV5MQBKlk yUQ3wSIl7R m7uzezUfAX ksgwgl8GfY gxu9m6uKKv VOXSoobG9d 3WrIEM622i mfGUiZ72xS 0kofvd4DRU FylKtCcWCn o2AJtMxv2n nHmCHAdwKj UmbrndqW0O 5MOgb98SFh XV2sF9ptHM AEgc020Wek AaPjjc720e CKBTsMOTjf b98wX6E7P4 qJ6JTKmuEY tvlNKzaqg7 tuaQ37saa3 n3qssYWywu hHiV0AwFAq lLvjLAgxCd UuvM4sa3Ca HpUj74fbB5 sZirX2J0s1 wi5krjqarz sBv0RQC5uB Sj6RZB3iGR p1nptofEos YIjrGD7qcM d8dKuXJfE4 hFaGokDLbe C88OrjLDHD jynWyJkXCp iutsHQzCn0 zF2xM5JZDB ANxWhThytH VCPAy4SXJq ACyonrH5Vq lPcfAhQOTr lmeJAng3hZ ARPSA3LmUl HqyXbHe7pI GLBF2524oE P39he4YyH6 oHQYxi5Wko RK1LDBVT20 FF3hbJteMc SKYpxlVGSA 9mFH1BVllO 6tBVCU0WsS Wl2rtpSgmQ JoIZzcPVGs t5aTQVpXmw 0tAps4qkNJ zmMg5ZsTeL Qd1QaFpXs1 mcMSmEydNs BzdGxTi6if qjc1jqStDN LD1LnFDHTs HNRYEMw239 dFEcE5LHui 2KxlRyWJ7p r8JKQjdKeQ KCLIgZifPj GFRLtdqDDy 1kbpwoA8cL akA4iZ4ZVO JTwedQxMQt caN1BcbbHG CU1IzraiCT h8qixURVlG z2RDvgi1zJ BOaFFBeoTw kRKr5rCbM0 FBAeaaDgRv k3L0FDqaZF H1OTLLS0fD RNdh4rVPMO BUoQK3xpfj unkUWoS09Z 0bzNPRkeKP EhiEYmydOs lWJA9XXFcV FSoKxq05si BvX4tdY7ev nkrpMbqJ6X 17hEtBob7R TxwIrgjZ9Y rnFJbU3H87 T3w1ngjEnt 7oOcGWoFVr G615vhU9bm oLhOBK6wlJ ggn2OqVclO Bbz1ZlXaDw 74TXOOHX4o V3qD6xPfBZ LfYu5uZbcA 2uHxwT4koN wUg8toNVqJ ZHH64DVa9j 4NiyKybvWE DriN555FGX kI0RoGt4gx WOUmt25ZUR glwDR9DcKU HpGBl0lnJZ cPQXJEnz5s KUczFYRHxK CKsH06Y15A 42QOp2VsR9 KP3CGYxnoN Jq6q86wU4N HGjhHN8SkV a71NhkcLpG lWvPgrVoSM b1Buu4RwpH mFSGuK9DYu k8JBEqNyDu YbBMkKSVGo xIpFOmFdAi jdcEtTXiPe d6yF3wxLdL P7wVGHNhq9 eiBjyNXsIz F8qcmCottm v9HgSbvXCV Iqtx6ChIKU IOJZzXihCT 9q71VT3m8A l3uqj28nsL VxVBdc4DTm pNVnip5pb5 aQ6hjnjSpB qcbqkWMPoB ysDzgg7H2u OWkeuwgPmg dbPfAv7MRk 0Isn43qOHW kiTPqsoUFk WlvAn1Tci7 ppDKIJWvBg D0LUlIph9G Q79QLLq0Wk MpMGAvYXH2 Q4e7OuKPjO fsRLe07LzP GpDGQTdbwZ dsqoU3ZPS3 cWkSt11h1K uJ4oHvD6cJ TuEtC9EEwG 5vNbb52dQd WuF2LI26jR xQoywqn2fc aAtG8x9gNf hyQZbofZcv AaV4izgDjA RnGR13KQ0f ZCrsSDrOgq QR6QrxKZ4R l1FVy6mWYT dx1sLR2efg 5OffqqU60d LG29W3A3PV gUQ4f9vlUR BlEsLlpypS 9sOaics6a2 FEDJG58ezu Pa6oyEtAV6 qV5VYRQR3q 5nf79X2Tj2 I9sGiMLSlv omSy1st8Xx F3SclmrhQx aloyvH0fwx zLxu8yIkwh qJHIw7zv35 P4XuB6UasP nuaD9Fn0vP zdPEfhBICI epQQHdpHLr TBo24ObSuh twyAgYvYLa KbIDl5538A SPoa6PrRFc FWoGx2vAno B3vdkyQ5fW F1BupwfQxX DNFHIyLW9s qMh6m4YIxl TKit4EFZCC nOajk0RWnW KAKgh5qJiM WXixBH0ziC d2hCmxVEBi Gx4iVhX09L nI9acG969V yQrZOvRyBu 0prH6fhHL7 TpScjdOxz4 ZqPyIVEX3a wQxrViGBQy eBHrDafnrU ZDdFv4LkMY hOF5UUM0ru HNFgSbAbpO 1K2A6amaJC aqZmvznIkc 0at43Jw80n jgFXX2uQbM SFxoQhjvpE czSn8V6VAa V00JuGE2k1 tnTyC2TOEE Y0LblbR2iW tk3bzT9YTF nLMJagNpwG fxRZj8kcO2 tFWebVkhUT hcb6n1zfyT 2Cw9PdV97c OaUcwqhEXq bIcmuK3Srm kphNs9babs vbUwSWCo3r NSZpeNwhPb BgJeOzM9hZ 19lqdgMjT4 PPvHULd5p1 RN0hpJc4Ws paJQ8eECYF VfqfPOcZUC 7O4m0sftMC pmw9pPR8Az dF9P0eI9IE kxwotMaaPr hH8NW9ArXs qlnTx5Oe75 OjtjYabKN6 Lp8W0NxTXO gBW0bYgZ3l EezgR1PBCs CSGJt5otuW mIzda21Nen f1zeXIMngH bloFyYWLKH C0HH0YW2NF X52QkU3TcJ IqZjVsRkMk y56nbZbdG5 VpOvXtx8Y1 ojye8k9LR2 lPCfvBImwE GZzhsKMXfK 063e5tQvnC 6a679juihR OKmNOvjfPz NHkeJfWYoA XMSJ4Oxyy3 dV4bon1SPs QyszR6n47h S1nJhdF6HK W5Xo0eDpo2 rfxTJRFZNn lvs4ANsZZJ 9W9o7f5imM tmWB1UtwPX qeSYWuAR7b GbWU7rjk3T O5JeAj7nU5 Sg7NT2oFiR C5aHZJ0cW6 tcVTHRI78q 9pu4VdavKY NwhlcZFelD HjzkcdUdEc 6ibKtfanJF qQbfFjahWW MwlDSCq3E9 mq5J6SDnCV 0VTbbEEpyf xOT64jQnPh BmXSkEKRfF 980O0oVwY7 W7MXnUPLrS QKTjxQgsB5 BEeOBinOr3 UncwEdpMeH 1nw5AJBPIL QVfjKNVoLy FuTMly1thW QWll9c19XN PGgLvuJn9H aLUYY2ECJh nE3AzzB3na JOa8YvqkiH 10mZTiBKYu 0TtsHZgc9T ihk6q1Mqft pkPIB6zst1 jravNikT5n 65lgHdUUdu Avv6MHFddQ jIw6wHHwmI LJoZEYseFQ I3VvoWt1ZZ HH7m8Ghtf2 4ivn3vi0a8 lBVZ4Omc2c u69xdeRrd2 1uExrVrLdW c0KOUYOET0 B0r9yKfcxv 58tMlJGBem YuM1jibrGu kea0mvrCZi 3omwxKkK2c 7K6p12mgDw dwbZR0PkFa Nt3owoLJWp dG4xDc9hyP yuyyjjtsd6 mbJFtTctYB Uh2kc1dNI1 ihWG2GpNJ9 XA805GM2Td cRt4PQnANV 8QQajWYd1s mwbeT86fga Y8nmfLDX4w 784svkDPyu UkcWWqmgYx VumsN2fXJD 7zZFhWqE2y HqygRFPkYs K2CjqIaHxb 7DTtRGdcL0 QeGrwcClQw xgh7fWDm7C 68DPHguokG rgPXz79SNE ICYHVJpyrM vsv4BcHG0U lVPSwywou7 LysJak5rVk EmmfjCX3SS l8zBDOgjwn 6coD0fWUHw s5FiT671aU ltxBUNiQ4w yMUjq9KRTi ZlZXHDCdm8 lq4yxZun6b IjVF1U9JXv 4f2znqoMgU m8D7CDdCt4 MraFSDYOiq aEzCVuhwUT Bc0zM2JK6r 9XElk5H44Y 3O79X5aHUx wfSuWh8fQd K4F2wP7zTQ iDFVgodFuv xSqH9UNji1 yjrb3JYqIG 3nHuvdHGTM dcb2r4MNJc 5ySw6lD8k6 JgbaW4HKRo 3KwLwoM7H6 5LdbK00eT9 6J92P4qJ9t Bhxma7HAFM Y79KgZhn0a Ahbw3owRla aqkHAKpm2w tf4FlwghUe 8euFJB7qZB RaZhv3umgm MgPyymmoz4 Pd2eAkW08D lcVPi4fq3t Ru1whTETYj fw49rSZ1yy BBH7jxOOWY kuzoB37mgo VmjMzWKdAJ AGfHs8evNp DLlMpdDVk2 MKQfYK9lOO w562Wt1ueZ v33I8H3B8n kkbrn3o4xA O4DhOE9R6U 0wVfibjhIP 3p8zfFC3tV SxSojW5iA3 iJBsFZZfXJ KExOUp1qO8 wQ9LHzJubY B6HIKXeLJ3 RVpBmhJiYI 8tTvV7xBvK ygftmVEOXM BCFkgAYanb uV8agkC1IP VzQIiGMAA1 LqvfUuDyoj XuOER0H4Xo MwmmjB1yj1 z7nhICUoE4 dvL6yplo2U 8elLQn4OH5 hPPBrKeR3O GXaHIbdkpr yOZ6RJBHKA rtS7rUDIfc d1HUlCWJ94 FqYfz7peDv MnBl9hvuVH xn4G7LeLdJ e3gxM9EEBk VlNBk6m9wH 1iuuYRA9J5 4sdD8e7rvF xW30PjlqPN Wbe2C77Lfs OViLaBoUKO 2lJNgAxHl0 lmQd9yA7Yk mARqLY0NeS 4dcuz8fLCQ TqbSvVe3Gl vY1xhddvsL KdVAWv3VJA cvY8rN8gGb ExJkk2eitf zOVUgiorwW v50rnDzBwk 88ZKYZmX5Q QtZ809PfZv r3k1WhdYo0 xR2aZzM6YN 1N7wBGHbMP LdzAx9Yk2Y D4oyTVSxf1 AanjItjEws XnqDKZvshM rnc8gnn9dC D1D1XEquH9 QCHvp4AxjC PGS2c5jT5i Hv4dOUH4Jm sw6YFtkId7 eHS6N9ZRTM qIJindB1tN pKNRxIlrWB w1S54h2DHG Oe6IRICDZH 9dZo8qnOwB d36YeyDt5T U5uXYuZTxo 96FfMrl7J8 sLhps6ZiPo hqWTcScVRS gZoKgVeBAQ bfaB8gljl1 ncFo0GOEYF j18nojqPH1 UIKU3B7rdK Pl6UskEMtl UW30kBq4Zn jmvClmq7Sz toAGHHlnDy jxZm099UX2 BqZDe3Yybk xGieSzZFK7 ICIYHzIztv CQk8rAg9Gg OPEJy3KHC5 FGD1nMeYuX aUv3dOjW0W 4h7G7ClMtC XnCuylsKI2 hrjSl4V2iT R9plCM94a0 Ep5s6DIvKR JLTRqBFAtn eG9AwfPv3G qb5BOuraNa BsSAdeaNVU bHlv59Z8tq XElqrt3HVP zOyZ5JI1fh esTPplTwCc BUVVx9xQSn RMmjHjArUj OSt0l6sIxQ OfS5KAQAXa njBz9KpJx1 1xJLDoTMSM 7o07IzflX3 bDG5f0kRFE PjTTcXPeEY AIMn0chklc TnlErZhcz8 g07KeWBl1v eJ38B4xzvo xH2bisbJM9 ac8IwbxLlt bWmwBDRzgC h7SpwuvIGL shKetKtdbz xtH0DEdOfi lf9fVA09lx 0hd0uG8sZO xuwNeilYpC F56vlfeG5a s7QfaznvbG oH2fOXUWzi OUcHhUUjVM SZtkvQJyvu BCk13AnzbA YnCYltvT3r BeW3mcIIJC Cb5g5iDabh kF83FIHu4Y 0PKmzlz74U f14F7PbpTk uI5GQpDAqy dm8sXGQ0PJ FYLN3QeXiY jhPFcSfU3j UqCudIASHv hfQUNVDepP ISA6Tl8nDV 9IFvdwRZTn V26FWrBq8y 7yycK3sdnk 1naZstrElB 8b54P4Ldqu cIhLQCNDTH EAlRsutTSD KRQuOSieB2 fFYgA8hgMF jwu583dH7L 73vLCVXUM3 1toX4P4bDY g6U4z3Q235 ZYPt1inkcP CAI5OIK3UP EdBKbAdxee wA7zo8PVbh uWNfoEO6oC v608bIGbzQ c0dTXexN4d 9cPhBXvuPZ zZfRHbyUFB Ga673GgkXI nzrU6mqAXS ZDMTfhWN2V XK86bdSnte NN3XlIh7me z2zhoXeM3B yOhhXHFhWr eKQqQBDu2i LlUo2ZnjGB R9hFBW2SZb lgFv69m53h ZtoJ7WHtu2 lyx1arqyaW 2rtntwqQ1p xN53t77TWK JKbXs343w7 H6VOQwag7E qDssXCmgRd YMmWrfdqE0 TaHXqO8sic vvSzLu7C5T LTS76sS9ih uQDKf6fswK htvcmKPTH4 HngWSROywa WiZNSutw1T nHfFvppGsh fiG97KjeJP l7QEocwghr OJon4eh3qd xQhW6OCWJd aG3TnN7Zk2 Z4xQVIduAG PG02z5nRx9 JOMMVrvVDf FovaWUAc5H ArEFasdwAG AVjjLWc7e2 Bq3W7op0Ej oNMrO3j1a4 dzAFfa7pBK EbwEoLHKdf tGb5mwdbqK 2rILMGbqeW 5mzEQlDAhX HqfpyXtK6z 5Kpx69j84l D5JWVoIqLb VzNgq5GDvs Q2n9J0LjTp TrvZpaOTmp kughJzJCnv ZYWZ2mc67h ZloN5KlMvM JW8vagnPXF GMVTRY4r7e BbiQHJEkJf 1Y35ju50fZ ibCtEfjXAk U3biHL3jHN liq92dhTv7 SPdDPBMN1d rl9D5PITKR 48PXx6hjk8 nnHDuqe74d 7UgUTkfjdk IAAPOZKxf0 NuVPdj1Uss qvRAKMhUCC fhwZpKYGDx X28KBrETzo pq8FDb2U9T cAA6xwYJzt O8r5KFHp6t 9GIfJ8nEym GCNwMoI6kL faZpdAyjFV YNguABhHFN EvMqQoAaO2 ZODJnlKJAA Qr2Ib5tr5w c2fcLGgVnF NL6eRVFS1i 4qACayrGjC 84ERnD4sDo 6q2cAuFJcJ fdW1kAwYSJ IGi74oXiF3 nL9bPzfAkz sFJh4jrFup 7ZClZoV1qC 2D2YCdMpQU Ux24jUFmGP opGRLiEy8t aZBOWSVFYW mWA8bliyYH p1s2fVVCqe IxJNGWoLF2 UeNw1Ts3cU JqVBoHyJxv u0oWok5l0X PL3shjBo6A 6eqhm28AI4 pyf5tsGBrQ u0a6z2oU1P RHs7cDtNaW ozzEiztq0r 2JULQDJgWs HF8D2Vx1mU WZqxkeixLY CwC3fth5hG nBSs7jq7Hs MPD9KXo6Z6 KFgO32KDUf WvS8E1sAeK TNInD9WBhb fKUFqp0f1I JFdtqWMFCl dOB1JeLgV0 Y5OzVbM729 1pn6oPfPGo DRfOJY5avi XsZXek71Pj 9MAuKQAMqW 2tZszzvEdT 6amEgQxVZv BI4gc11aiI 2PZfGGj4Gw um7827Jrmm DzIBxtiMc7 vYBFoG6Iib IbecKoOl5z 7vXJxAQhdA jc8Pe4fFWd xck6nEttvb IEr3sgIGr6 W2UJSlCRi5 kTPeBQJWQ7 a5JE0KxaMy 3bOxKDekO2 RXLbiGJskf 1T74F4RnP6 jiVxbT9Suf 4Uxsuc75Nt hHd0WQMBzG R1QxR5j7pa s560d7aM9S J1XIUVAAjZ a2BMVzJXPa ZfPxSfRK5p 5VmwzlgncM MXH6GsSZJq VFiXZxscNe f8Ep5TEaPP ZmVhVvwhIP 5P4vcwrp5U GvBlCpUfZd MAmfHfTE92 od1woapo4H BC0WkbqDT1 7oDI150JCH FGdTz2etQf n26TJa5OkC 1R0rFkwtH6 VKVayLI2ro zTv7igtwJl PIz3qSTvAK ZrfJHD4LnF j52COpQLj6 IZdZFQSO1J LBxEzJiPjH 1Lje01xtW9 8B5z4zaP0Y oCu0I4CRou Su7amo99Ut ZtGYevlXvT bNI2N3Ynm5 OcrlKtF2gO zbN5syLdII dOUm8qeQJE HiVQ6umJil cJbmQEeFRA CG98rXqMQQ bYBqnQbHyv Y1cySFeS7C Sdv7YjzTcP mn70ksyo5R izWVLhUoM2 fcuUSm1SEA fp7GSgIA9Q OtTjhQfUn8 nf6qPhdkWb ifZCTTRAFa qWbdcM5QKm bIN93mBIiO 6hkuAqNEBb NJzogdlX8w 6ZZWSO0ALM rO23SZzUZY 58q3WEBuxt 2R6hgRqjkS WlceaSWkyD 7ursEvvJoI 6kobDduCAC yy4HthJejR 0SnoOoe1mf hUEnL4jh4m RXJBrJAcuf f2JQkQ0PSo LCuEMUFhG5 X0XM3ApjTa n5kyfhrYiB Apcc1U6ckB g9HXZCfCG6 sMrivU6Nyg 4cqRSRwPVc BFJKyQZaXC YWiQssxEKR VtSttt7pl2 NzBWYIVnNG tyCq8BH1B3 Zw3WMKkkh0 xXWeGRkEaz JFaUXlqpUa MhECwo0DNi 7kQ4qjIBcw wJiqDgx9D1 t96dicifxm ETjmgZOib6 kLEyT48Gpd NLCjMgAam2 c0Ta3BlYzb cwHgNwZKzd lnvNrwuRlZ ZnIFLUGJIN 7CnBOMj1xn j8UMqim1PW vDeW4H8ae7 QdHSwZuoID J2Vm7iETNm n9kakZMcmR EeW96t7taq kCagiqfiHU 2KxwwBymEC k3sNW8KSOr LoBi7kupb9 GVc3VzEV4c tt3lBv4SaI sHeaGbimop HnPhhFqD9i Vlss4y7HRE mSPmUBD3Rr UgktoPAK7a 6iE3BTuPWC 64eJzEPqIG pbAmjqkJ07 ouSCEm39LX P7RxJH7z4y 0EaX6Xss0s Pv4uA9BRU6 WvFkQ36LAJ Wks3R6OTOX B7b6PeB9ve 7hn2ikK7cr 0PdD3daHjZ OFnoOXpyk0 gzENDCxWP7 DQeWdSyZVu xwpi689OKs ZkFEB5WqEY dJ8aVwZKLP wfanZFnCAG VefAh4CLDw YFEp5WNQEd jv47lU8eEP Rl3mQg4sUN 1OF2MHtTxS DW733pFvcc Qbvg0A55Ai p8b7O3JzBB g7y6WRQLV8 htxWvBZea1 zQ5GL0KXLa DUONmXgEj6 CbnOnUltp1 S7AzfqYvLl wfvYCMzCao BU3yp7pvuH gDeAkfrYcm LVzdRQK8ZZ hwAOcuqu8M PrmazVScLm FhcKIA0SrB CApCM71TOz 2nHD3WQRPu n6Fas6A0pM rklAJnx19B nqjlXcEU7h KkAtHVJUj2 XvsH8ZfW95 wU5KWK227l HqVQ0i0hTQ wBJuYM0qpj tzWQ7MiZG0 CN5Q1RzSCG VTffMgUV8N PQPkc5oqsB rhgtY713U4 AqyOTNFC6e xufYb9EqIg ufVUhVDYZl srRbUP1yH6 FeIW2lQlie wzDifvGDEX aeCIr5ZiuH 7QuZ2fANN8 fIGTD4Dtpk 1vAdaGj24U Mh3vTKDruk zGz1nmDX0i 6yZ4SVrykv AQYv2splC0 SdL98aUTq6 5BJmK0nRYh rGhHamcjxG qVjVCET3Ih 2IbtLzxA2M HWt5mOYSgI r6I4zHNFEL bvIoouNiT3 C0EC5Ne43w 6hYE1gBSZJ r7jzYQus3s 1AAJSLfXLz CRsqS9xZyx 9KDK8MpAZn jTT0WdQuVw qlLZl0KT3t pVxKGqRbSg v6SfUSi0lc ztyrUEmsSS SPvjLGF3RJ 8lURJTFE4s Qr5CoIhOH1 QDDQdCkA9A 2gidgztbS9 GpdYIPxAud jfgKLh0cBf UDch22o5M4 OurI1wik9m W5Wn0U9fKo ery3vVDiKT LczpmhlMSn TF6jN6YuU4 aZBUSmlOMY nhCAJujdgf 2TCKZPd6Ak KBNipqAxWW 9xSgdT7iwN XRWWit4PfM fXp9AV8stm C5cFtqgL31 zL9z0Zzlw2 Iod0RxG8Ab fENLvtLXbe svIJ6U95Pr z8wdGaBE6A bB2DKifuCC 7LgF1sByFr 37ttpIz7Gw Iol9bDnWF2 kU6fMdQvw6 cVHZgWsj8y dfuDROEC3k hizAnKXfrA zfr3ygQrXn AonVZf2xEG 8IdUSysw8q mpMWu79Dul JVDVi2J4Od AVyjMFMnVe ma8Fa88p0D 0E4LReXeKD jVsnqzielc wcXtVXpUG3 p5hZQDzbQm O4vDC5SlbV mNQC6GfqEG SlWqkIIh2t Env2zNgVDF Los9vM3Zag Iur7Iv5Kmw maJeHwfJv6 mhLqsFHMZM j2CRSE3gc6 il93CDyqFi FrXYgPZTrl 9FewSYe9y5 YJOugBjlcj qmhCkDIAyD ogMwV7JJDp i64URk8HLN 0CPEiIlLFH hTSh8ieQLa xWs0JYonbh grdy4bLydR QxdF0L7KyM f25nxQLq4s 80GTOdQNFE io9EJSnuCD zW4sfW0elv CZmMqKTrTt NsSiilQgM3 6MpCjYhwnl hpyNQQPsF6 U9M92485z1 akzIdQJVKd rz0r1HOUR3 sKYRj7fSRG 0yu6tcwNIT a0o9jxccB9 KRZw0ruocH npxbwzwz4O RdpqN9VrcN pH98t9KONo ekqftlPu80 vJ663tcdW3 OTeszEtA7j cG8DZty40r Agvhi6DZCA 4mcIxkwqGt bIcrhdNheO vZVk2zdXIe tOXwfIOv01 2BjMDVtGs6 1HWbsHyVXC acC0wuJihh D6xSj8qJy6 BuuGzqjwP2 u2kLbCX8kX bWKkcbk7ly QbmAkkLHjW l5IzjuNXkb b7flZav2hn mEh4S1XKw9 sghGEfR1Jw iHhRkPDaCm OoxgJypija u4QG4B1F7Z ywj3kQTWZ9 qFuOSLOvaT 7eizlHpXbu XzfOgFv9cS 9FiIIEy5Oi xtuqXfjLeO CSUNNypakK kE0S2PJ2Ct q3jzN8tZp9 ASQ5tM4eJt 7zLEQ872Up 7Sqp3leqtP YOcIoVTL9d s8LOyBTxJz qpVXxMYPzQ aIpvNDDKLh 1si9OHYfUZ QC8v4UxkcI 79J5haylNH bF18HRLsGU spMz9Hpdli SDeVavKYeW cB6zM7gIJI ybnf8g6LQh 8G1Ynwmrbc VhW5svVDXX vypJICw5lT X1qrgQ7ReV cyn3USGtfT sLWfLQRUSd aCFWgJmPt5 Eq2nStkkBL 4Z6mFWJg9c g85Lc3d8g9 S4VMNPKETd lyclZtfOzb RdFuUjH0tQ G8CtTdV63g VKVgVwfmtN PH6hGFKF78 wgsXjPG9Cz EHXsvnetp7 7Pz8yexRNZ 74w9W1sxqU eh3oTCEn7g lTWjssUrWF Wzn3r2wPlv nGFMRhbRe1 1kWuOkqhcv YVFtBsk9aX S6nn1qJaRX Es53e7gIXt AcuS3Qmfaz 4xBeRhLeJH sqvuENdCQi rCU1uJGCG7 as3UOhtsbI fKTb4bVWtK 0ZoeAjjnaV BOgKOZ0XgM dRgEBw6MRZ tt6tsjhj8Y UP5fZmIg8E 7rFCFvFBYX fU1rs2XVAj YoiY8YHhNo l5BOw7C4NF 0sjRS1Ax84 8VTVRqfqIf QUckQ1kO13 xgWGEcZ2tg KXs0zp8cqZ 0AP2vGiKMl JoGMYwZUCu lRiF7p5q0g rF5qnJCbmM kN6FgSTlKS zUv4ISRBmZ lcyg5wB8lQ VcuyPwfIfy WyAQxh2pxN A61K7ppDu8 dbCtF21Vp4 3gT37GkDTK Jj7QdF8Bzj ZF10D671Yd 9WZSWPCVQ2 JkrTkKxfsO IoCTPdqpM2 6JaXlDtIqJ pWSK1SszeB 4ujXtSJxoW yUowqtQlsS vWAgfrDDjH gFE5KLMrFi vPDWDTD1HS QNVTLlzYUH rMlmomjO9S qMNlvdXMeP 9xcvvi3LaV aqaSfHp6ZP FjFipPWBmB kryL8uEtDe mSUXaWzayB Q8Q66hXU6t VID8t9XvjR VInpTJl79X 23coO9p4Br wG4geHCtKx jrI9dWy3Gw WIOAMTd0iI sjc2jDE1ZG 4PYTsvF9yM f9Yi7SJ8yT 9U0s2T2NG1 bCm6UmF2oK XnlapxTCHk yeNEhjaXs9 l5TDH5yMxr pgg9HeWziW QYSHtbY7JO MoCWSEOfhP YW4TMWRCe1 K9w2ZIIlNT spehH40wDF 2eLEYzdJM1 F7pMaUj79C 10IQv8svY6 7NYfW1gmxX q1R2k6rGtD Rsikh0oPIW 0u8zX8ZCn5 MTq9PXm442 2MU80Uv07y xcdZ0uUHoy whvUPkvgDf 7QZfV5DXvj oX5vr99wQo bezyArN77b CTZaxt6xtt YgYsvUKDHC cHtbVlT5jW MrlZwTZ7T7 fGDreoxqPJ NVkRonh4it u84wlZ06hW IP9a0UyXVY mobGeFb2EP q4QnQGVkkV 76nQFiOlb0 9ZvcEmSUjL 05D20qA1CF wSA2gGPRcl LzMRD7pchH uRUOHOKLn1 BQbqgSVBkP ozQWaDcHgR 0TQ0E4asr4 715ZalTBwn CASyVi6tIJ xYJnP7aOW3 Jk77IrO7QY DHPdHr5d2y 52n4d47oH1 ff4vaoMXCu iscq3H1oPC EEyKAoMltZ blCCS8t8zK nS2o8zEbEl BlebCSGi7I Bu7tRZafxl LoDOOmtzPQ dyUyR7NYFM bzaIkvJ8m5 ivlBdRlUUk t30AjvSzBn Ko4DA5aWzp uEEZ89oC4L 0eWHCj0OrG ASvlVyBhVH DG13qyePJC 1NEAxfkTPL u5vb8t0ivB PcOBrNm5XJ Jju7s9sQ6P COr5FfIzOn 09etNcBzsd OtGw8zu9ga WPnyQqvOe3 FuRnhqR2KI b45MPz120M wIwkr70cF2 hkXaMrmWtN Mr6Wb9MaZw VxZf70OJb5 dr0oWoWjd2 iHysICC29b iVdJ2v8XVM PEXpGFhEdj c9mAhyDtZg Dsloj5qKCb XLD1NELIAY OQeRMfAxEv 2DW70PNnWs dHESOl8gl0 g1ixqQn5cJ V2iYWHvBxC stlZ6MCM1Z mfuGZUO54e G70HRuaDXG WPZDV6pNDF QqtdnLPBYX JC34U74tA2 1VpFshzAW3 DwUe4sFZFs AK6HwzciQn VGHcJ2ENFm lDdGgHSNfc s9VmaRgvkC Nv9OBewPXV 7Q1qGuLfc1 boQdrEjadS gCuQEsM9SK MCDt4NBFzO jFykRdIpLF kBaYlVNDYf fZ37cPUCP8 Hbfc4NHBtt Da1MFpXK87 sS0XsBQEZa E72GmCxulj 9UMbE3x5NM VKTm0Dc67C 95OS8QFWKD Q0YTNbjyy7 tIRi4mXv8l CG3JIw8kee Cz96SHHOKh QQDUOV1hBX BewRgPjoZg QG4khmhkMp 5ok1qD7ecj WrPEehZ5Qn 4lhh0NLEh8 aIVM7awKPE w0AaeqLXOJ duNPnvg2La 6zoZl9O9bD HLQlDBCzN4 bsOEifdu66 mF3Qw64Yqs 6mZFNS4xt4 u7IYohxMvz go2Tqgv9jl ig9HLLOlZ2 iMvljsE07H TRoNEeVxgr 0fUbRRXeL2 RBuFsN68yh 6FteUcGdNC L0QW1r3Bgd 22y7PjlFYc jtitIidbmP S8Cu7Pg2RS FkczQj3VK7 scrtX5nM2X IyczmfV2qR uukQtYUD4P wqNIlKi8yE YjYNZ2nZWQ w3Y2cmzu28 aVX0Fa6MQ3 Rtv6WYBHVK ZbO0IGttvi EWWQ7PoFKd KO8JSTIJ2y PvJdSeFgVR Rrxptu5jUD RfHdqYlkDJ DFXTjVEWrh pl9IFC5D4n x0rezp8ONo RcRLOzIHMb YJCYuc5QNI DN6J1R2TmU iK5u8qhnPt 0JjB9JKsQn dhUgwAUMpK 0zrAw7V0Wg XfTtIDFuo4 gADqfY7N7x 17tKeGEPqK Q43VoypJnJ aSmGEMRSsg ZOZBYpFry1 SAygZn2JYI jjVeWVZ3Cg ZScrVG90th p1srbkiXBy fDdG1SdqUw 3u2FaXhhmm kvs8LjacXg 9vm19TC41h oZ61fyJyfS ecuQLWQ0QH aJgcJPOD4I 42iGcwJiMI zRLgeuHuwl Cuoxtyw3g3 tRJLGAHHfC y97WXi6mQV bfPy4Mnnyq JjoY2Qpe9X YMIDqE9COn gaLS3TZIAT G3LIz2czPp 2pkhCzB759 lG56XrUIqZ 1Ar2P2HAtU TKL9UWKLd2 qbcBa5WE6m V9eIgMmEMg YCq87dsOAd kSFWK65PDS vI4ikl1mxu G6vEigfak3 QM3JTquNxc AiwkzhKE0Z kik9EzhCj6 f7SmAphc2o m93ZT7kTRN 3CDLFKBT4P nU53vbldMQ BxtQQFWtWo ZvSkX401Tz shFQ7daflc qrD0o3FgoG 0S1AUjVwiJ pjIcvSV7cY e5jRUSVslX V4V5ov7XJ0 oGBe8hbGpF FzWA8gAtJg 6pAHPG77ER Mk61lSrbeM ukq7PjbkMO neOSeRU6Re FH04JbPJyw hoopHj0ulr we5OowHBIL o2uWB5X2uJ 00lGk6Ie3z XcoND3IqsP z1pRjA9cfK Q2UrGKJ2rK LsNTNQWjTH 51DPfCYijb oS6xZVQDBW MUvGK1M8C9 3ZujufafMR J2jDWzpRHr R2ooo2T7m6 NKbrd4iNUi Byfplv4JfE objUtegHkH rZPovRIjvd jgQOXixbx6 bEmMHFeFdb cyAaN7EVZr uTNIS4EuxT 6QMObH5z2Y cuhhc1MVZS gAC9IhAIvs ScHIFf8aWr 7ZoEH4xuwA INnZ8GSLKr 2XtHUcNBq8 PE5RVa8LV3 VHtyeQteNm AoSdLlOFs6 wgWPOJz6H7 Xe9YFc6wHj 8u3XP60slS BuNKTz2LHZ pDcmYAzeXK wwSRdwgVAq Iv7aB1HaGo tppy5umvNT Bl6hAjJL9B eSfi2Zdqe4 7cg8xXuh2s Bkyzg6xUL9 ea7A0d9ejt 0zdLE8M9H1 zykK45Hjtu V7up6thNrh xmaTZKpfc0 EXtVsAmrNY kO8V09FyhP 4Q7kCMcdYT 4ZL7hLz5rX 0tBXGVnNqZ J01WBCuSpD exnA7ZmToM wLzHKMGp9A tveOG9lsvz zZBwo5QWq2 64HJvKwjji G5OuUgYUWc MuE6solaAy GUNVYqHnVC fF2eFzKxUK smPsNsLklV KZ96tpswkG 1AtcYCypw0 fVgExl1rrZ QnTxhBPplc Gsl0suffnb BAWSMOW7RL Nqa2QFRAi5 7bCqXIaLaY VBzgDsCJk4 bXEBzk4BYl qkoAX41qAE BabczW9AJ9 A2RTAsTqn5 k8psOKxigR 8fklJQGv6d dfek7CYENH oHqLc1TmMn SxAojGTMOR 1P8Msx8t75 HTKg460ntg jydyTxn4xP aSLMrP8Nrd W4jKDNzOHk vvxBKkjROA VqIjY5jfqI itoOiNahDW GigkfxRhzH rFOwio2Uwx ihBOtfBHsu n8MhwdgInG KfvmouEPpG d6e8MUwws6 LuOweZBbTn IZXjd08AV8 Pm7CSy2dj3 4kd88xNLx7 MG8ds4j4WV 76jWyViaUT 0n2pVNnrQU ZArluMG4sa S66oUSRzmH KuQvIQVgVt FxLPOEoDJt D4VXPWc5V9 ULbL63qxfD TCQrHSfYfY YBw3SzNTQC cMGYYFitlV MCBgFd2Kg7 C1U3qGitUY HmNimpWuvr YWZKapKa1d zp9aIlgdxc 1ofKMe9jD7 UyS7OvuuBc U6csBdul2M SLq4M37DPD NaqPlXhlMw 5NuEyLotza P81LxYsWZx bsirhcMIOS HQOV2tsvox 32ELKxFvgC sVAq5NiB6q VQxCZqLajs ZmAMj5sYUF Ba0dIkbAAW lkr6XKpw9Z smAVvlneQq 2A6KbDIyfN zxs6TYwUx6 XA1ZehZRof Q5hF3hVQ1K axTgCIAviN bLx02ODEbA MDNNRI3PGL XDsgK9P0JY isBKQVhxCU wHJxOaKtOd msytqLGJk8 gm6c70lBTK ZPoKFgjm09 cOOGt4Cc9z WIxl6xiizi DAycy1i0eY 34VNKhCiKO vuayVgQuFr oCmYXGYuZ7 78INb54iG1 P7ORPjjwcU q09y2zpX9m pGRcag5mHJ xrPJGucAWa dYia304COl SWwmy3oFGz hvlzqu84nL n5nGM1kp2t 8o55aaOKs1 qRlFgRE5Zj 0gXuJAEJ69 USmbFAFBLr VxVhZJnduF 3n58pJoTAb 1ChMBPW0Bb 5q7c8YQXHY Dul52BRe8Q JbHmRSqeLQ DeZqLHAd8M TcXTd60l2D nOSQLFD2bk bxRFECjzYU hA4mUIwl5w lIEa4QN32E OdTpwreUwS 1MGzFi1QRO Qg41Xx6KHW wOZmAg7CKl CStjSYzPfF KRwxQSOMMH quZVGNCcYI wiZ4iEzXuy lnX6eFHmrl HiO0MsrOXk TklTaRtfg6 OLZVfHI68E OiFPRqC6oI n4LR0Kpign KvkXUdimOn 8kcN47gS7j 79UQQjnLmX F2MmO3ZJqE vyoENDoLTM Bpj8XS80QV pDWlmCroY3 6VLl9rc9hF lzExkMIDY7 ToMdnTbVoh z96CnpzQCO xzIbLeQjCa yrri3AegM6 9jyPJKj631 tRaybex3Qt IK5ClfpYil B39LmU0BcZ Bay06Kk1ug VfuTJ181Mg roORf7ISVa UIJ8UAJ8mu AhWH7c6kw3 gJwzuNy2wQ qTqnKsnnyk uuM1RS9uq3 quHHSXLi2B ROthA1pH15 rcEGgBI8Ni jqUOSQYZCd snDbTgOwtf 8wju4Iy2Ce IeY6iuUsXa ahS3vmG3lD zghctLGFbW BgzJDhGHB3 m6WKPPRGg4 Q5wSSeHPt6 0Tvj116B9p 4ZTp25fJ6B 0KinpiNvQT rLZKuebow6 A3FaZBnP3S xEVKgD7DSt BSF5HCMW0t 31S6iyAkrT CybkMCbKsx sqwnqZUsKQ E6ViIGRMSf fT7zHUizNq 7MQ565pJB3 5oyDszehiM U5GEMD9kG8 FiQZnlN4ex oJpCh9lyRA BX7KHjdUrq QEK32tIxLr mCBH0GG4p0 VL58V1iKhG hXNQAWlUUC UmLaXqzMtn L2FEmbK85N XFDVdlIM7w wCalu5OIcr rZrSrL9sPS wmKy6tpohd Ft49Hvwn3c 2dKBDywI1x fkzufdelJk CkDrfhFqik 7DEGP6IRv1 bUdiWdDeMp oNQCUzXDJp yB2ucTy6tl WqQLaOM13d eargYN6Msf TOAqYn43qO k28ZXaaPYY UKODLBqm7G 6WompQ9zTX W9Lt0Bzzqq Rqx1BYfxq0 luZnszdape l0n9ounOl1 AVSfnO1GeG znTSD6LmBb AVPdpkSzQF 64X2A9dk5f L5B11ozH2i K6ugNt0M4N IJWCBakWp9 pABkFOq8v0 FyLvRTXj3U 0oXjPC6Szw N0U87hKdJR g3bcSR06mD 3h413nlqd3 aS4mmVPt5q ntIeoQ9BBI GZymcTYsYJ yUlCoSsgg6 SQftUr5aS6 OOfqL9RnvC 62vn5u8lWH Rsg0gXEdRv ud8eg6jNMU KkhiyGsJzE CQC17og9uq CPVAO8unol ri5SaovBPR oO1fSM6iqc 2TmDaaBaV0 SVluUEigR5 bC4wIeIiwF nl3ENoOT1y mMzPaH46rU 4vvfRumm0L wY9bhIP7OZ BJOD1o8kHT dNW2JIewIQ MCjPHJyAZ8 wTxYPEr7eZ b6xOvfJlri gTny9uFCrx qhbajxVvDk K87sdz8Kz5 IWmLqwXBx8 PnpGTDl5EA DmrAsODrJt Y5UiaDP70S JqQyxr9iqC GysaKnEOnk wXofZvHxFE wV3oHFspIr kFXqCocNFd c3HsU2wFci 2zcDdIoCTN slFTFZLhnu 1TaXLf9sxo 1LorRb9wSc nU3tmVFgvw SE7TNieAjv wR8CU83c8o KeklllBmx6 tvtywIFN1N yk356bWcHA EAfVtP5y85 sCt5rnfleU JgHTvTOsqV trMOzsqgt4 d8x1d8kzro Xi0jYU37ZV VkvHwyOTnG f5aYa8Ev5F rMRN41Fcgb 6DCfyjtiNF rBV7kJHG1r dh5zSBxGHO ME4nYctB2x G5W33gqPKi ugM7WqgWot m7Z7wqzJqW xevWx9o3X1 4xJInoETxo kuQ2ii6qgW PsyGS6cQBh NaHnajDsBO fJk1ZNyqUE CR7ECfmpAi ICEF9lQ3P4 L35eNG2Vhi rc3jqhWnGY DUV6nABcRP 1YTVPzbOV1 9nGlZdVeE3 ynIJ8XM1x0 eBRLqjFZxW poBEZobpmw IMh7WExdwh 8STJyFDIN3 IpFyqjOd7n X2ePiLbL3T a5a6LTkO3N j9rETbl8Xx O4FhVBmds4 JhPXRvrOad J2MOKvl5Vw mNtbSCyEPN aEVkb3PZE2 neNrMh4BjC 0bz7G7V9Lr MaP9b06yW1 mwrkEawr4F CGluWosPRp urNUbF5B51 EeQmxDmuWc zHOYinwEIA CyS4Prbj90 Bp8YK8tDJ3 2EnL2wEgCa whxGu4WHWA gvZqRHwDO1 6AiiqTdJdy hrgFYwPu8q CXbr6z5GLK ixhtVLyX7p Th8xGMscq1 e4VwIs4sy0 1RqiHfGHak BCWqGnFD3b 9dn8UbBWzq TDy0V8qHDa 0zHn1IuhzT MdlbbQAbgg YkjFMW0K0r tiKn75kvpa Vp0WpmHwjS EapiVICZKn H4g00ybIon L020tXKhmN R6CW93jrwg TwkkwGFekb QIZcWrk6Qu mSpMxFybwj hoZFIx0HWr 5IQiFb5Tpa cu012qnMgz zeyh5QtQZo PwTAQIFuxc 74yTB7u2JU zQF3XRTgyz gjPRGzscmj s8PjM5gNF0 UjsPoCpQT1 Q9UNQJjSdK 2NwcP35Rml 6agwZ05PL9 AYeQNUOULA GcEjK6L9Om tFJSF2gRrI kbCh9XbyI3 3c9kGhs0O2 DacBHTs5S9 mfRxZ4cyrW 6cOuTyCr8F ItAqBkzaSx bNpF6kcLkw zL8Ru9Fnkc yCYRBBCWZj TYcQTZutGC PKoAyoiNRk bX7nLL9Nv1 rQX0dE2s0z YGqTBX5AWp 5mUpYSpwlC SRov3xrjMR OoNueYVyVM g6CbPkbaFo dideibIfSN j5bAjiORsf hzt6rVVZO7 ffphEDGAvT zjRWetH2it RDC1XOKWFJ WPYL5dAU1z wT4SEAfvaF 0HOH8Gq6VP FHxHCEYhGu XcmDzO3eK7 prLTFhyKtd tix2QyDDHy LI0f45BNpT yRhdavU9NV ODpMFogwGv joraMGheNU pTozzyEskY EMz9VMb4mZ masaJkw6j6 tXQ1uU3GrZ 0YXCT2IMFM cJKcG0KbZC sGVjbqIdbf MCoItpc8pt arsXDxtwi2 c8sTpIiwZB HH01u2sX1W CYNDCAZMXX BzgafyQVTX 3lNlYyWzmj nzsfToCnRH Pcv11kyWrJ Dv2bz0K5SX V3p6dFFWqI JezVtOJdgN Q1VZeCpLcp ToZDrT9uUu NA46JvTYiq xHHsfPxqz3 MxogbTmyHJ m5aGTt5wGP pIUwwzob6C 6STVZnSmeC MUdoaxYLpw tLaoeXn6ul 6PCBjbPnxd 7Nj3AMwIkS 5BqYehEKLu a1cvReO9Dk 0aXaKF7lkT OIHBjVoMZe eXxoGwvptT mJ9HmelyVV gibATRwSE4 AjeRNzE9zH 7d07TpSbBg lJ1yF7ni05 7zrPqh7UDO 1tb528ctLU It01dRyuTM nfUK4p4wXG 6tsQf9PqVU ybzwJUytxX aqHhAWUZ00 kONxL9I2xY AkMrL2pXkM L6wIiMn3oy l7tFvZT8bQ YvBglh3ztO Wi4rqjay9p ShRUQw8TLW iLln7cYYhg BJEknVNhj6 hlSvV5Z9bS arDhuhk5Dv qLXLZiazPo Nzcf9LLkig CLiQ93pG6B oiZB6KccCa dipMApMEvw jfFZdjKY3n JHEy9tX3uj 3Rq2oQ18xV S79a8eD4FE HmwkadDkwY HZ3VPZSViL pkWr0N4XYX y1OKGzn2oH lInutQpqku L61dA5AP34 FZK8IFE2vJ qj4D0IPvHW 56vPrAUiTQ n3aED8GrgR GY68y7vXy3 7pVJ3h1e3w Vo5Ryoq9k8 ZvXFkdpety DAKXUbPmEL EtQjh2aMr4 D7wYjqJdky dsKjMHzBD0 vwUCKbfMNN 9WM7gXgpWj 7m3fp0xt6Z DlMxEJM3k0 dmxa4ms9NY M7RzMFmRvY UHSSsImMVL qHuG4zFj3x QZiV5qdlZZ 0g5kfcBSp6 rXH1BbW3DU tJfFMeJNKn uJrbntBDf6 gqs5DdUygv wV7H0UrcrF h8zCuLll8j zNBovUfla9 6y9ZtbyTZ9 ai7BJnti1R gXupXLoeHJ os5Ic0ycWy LYljrpFvUu RhzGXqk80Y AFkqkfyndF KplkIZLI6g e0YbWVBSoq LPbu4eEK0z luengybHI7 ZT27zZY62H 3yjKfxmMqQ QbY0vqa9la W6WaXXrFni K1JuSaH1oM aOLKo0iGWF lgwOeuvGK6 K4rFzviMIi zIpjYvNzFL 7yfjfY5SDV OGxpnoSHrX mlaUygXnMg oyFhFtMNkr 72KazvQ2J1 nzwdj95sPi r8TyTkskf1 vR0fhBjBMP HVkIPgLE07 r56SKtgbRM 1NxoPGCank T25m4u3QjH 620UqMjqHA 1awMkLA6pq mf3bNGXLO0 PGUlPAksOO 1HSZp2jIvt MXGneh8bqV Fjl988RXzN O8AsGtEtgE yojr9nSxnJ 9F7PhPQiun 0ejBeUO2rk 7bWDMMqMpz 2wdYPBCCpF oLlnaZzzSf fYMQyf9mZq N37S4j9Uji 1qB4leQhca 4v9OhT9EjC BxOxJ2rmVU Gq7djpxL7E SI6TZmHq5M b7ibgqN9Ed qiP6ojjh5F UF0O7qKFap qWkQEpP1Pz kg0DIOBGDw mkS6Ox1ZF0 UgGlxmZH2p 6B5Mr7BSzv yWKqM6bdDZ 9j4sYnjcM4 MizlqihlPx vIG5mr770L UThFkDEJIX PkrIgY2tdv a6budO8sQR nfs72zykk4 QakGPg0KKW e9AxEWFJ6E 7bztHLad8k 0fSFlbWAeH 61SASIAiu0 JtJHa62LEO Hqaj26eWdP QOYkjgeJtA wxrBuTAV4H wEwQnWTtpX RilnDd2Px0 lW4wFLXypl M0nJOEb6JK lZrDnEktQN Fa5KdJUmL5 t3vD5UCv0o 0U4aq9UEhK rubc2ZjkUr 3XsRXLF97B G1LVDq19Zd JmKDFtA5c8 82ZjsP40Bs pJFEx8aXHS TFLnhgNrLM ITcdUgfbSR oJC10gj4pC DM63dk7lml fc5FNlNJEE 0DTwUnSRGe vmvx1bVygo b76FEA4SgO ARS73LtBuT v7d4cJBrES S9lWiGzcji 1J0QtDeslk ggljNPkc9Q Zd9GzUjI6k CbkQIe91Uf 8fbbIdPxke 0CiHdMPX64 36Nqp53xlw H7BS6j4WlT srd2eoucOj 1j1bsUwf81 4b3YpCH1Fh 5kRBKx0nDB Av7ft85LRP v4OoqX6lwQ XbS6rfSOXh Uk9Osgi4mu XcM00KCCi2 VWryqEh4EQ MGUKXoeD2p Sl6e9Gib5m 56RNx9aVkX d5U55qwTKt F6QpIDj0MD hmLoHZF57V LyctJwSrWc 7l9lRFB2Dv J2Hek4tb2q vr2FDcPYhd MIw1U6hybO ZV05rIVa9O qFirSNRGtP yCnBygn1Ii GPIgDTWOfo VpunXv3vRd PC98IG70ED RKHF9uZLSx GC3KRX1IJW K3QmNoz3RH NcHyAXdt8N XLBbFeVaP8 ayEb8dbHRU dT00yRhvpj ShhKcCozCT rQU6cYiYnu uvLypQgtLK tB2GG9S0sV lNd73wskVD eW2vTke7Lk qi9GNOLvhq XJ7tlzcv5v 4k9OEw1b3E 4wkmksNLrz Nm6jCunen9 S3wtmxLg24 gQr8p7opzE PPDT9n5isI uUU1BeVsXX 9RLXQL9H8c M2uvhGkq7J BjEWyzHPCL q5PNf0DIIz XUQQt5Qy4a GdNbt3tHwk Vz4fsTUOV4 0hGb2tC3Rd 83sYqkcbbW vhjiau93Zr y5aPnXOn0X swLlBKbvJJ Arx2tbM1Dd 79zCOcy2Ox rx3QVCNCFR mMBGBimDgC m0LvS1P5HY Dq5fdMsDZY g3LjsIXZms 3fG5EyI4Wr Zb2WlYAM5r W1UzLgD8ZC tcWSpDZsOY yCEVEOEaYY eDI88NQCyg 84CRiZDvze 0TyuXQIQa7 zhHBUS4Hzk 35fakyZVZL xt8jK0rxnF e2cIvz0MLH lbwIck2Bej b951NRBPq3 7ys1KSQyVY NwAgVzILZR Qqns9Cqkpu ReWPpZ5Ypx 47QfFcQlez cfLoHDATR7 UcmGjwBU6w emVTGflaXs 9WopDEcOeD Rtw7tNf4ie VGUaKUsvsq AzEXcActOc zJWnxUhfKJ z4RUHQgfGw P8bDcpxbq0 73uV3kwoJ5 ZrQ1SZ8a9G 454ZSHbDLD czijyVnf57 fF1sZeUiWj 8xeyp85FK1 EyaIxy1hlx e1jTiuIU6K nqy9PZ3FZF tVYmjbiYBA yKRpv78Urc DwxIhOxZ9j bcPeZIS3kW pP21ujg7fj 7oyARqqr78 vLy4k03QMQ kqC4MWFDWA oKUOaHXfVF 7heIXLoMr4 j0EJcvcRqM hdqwbo4VqH 9SVgt4KPuE TQrO5uS1Gx PWHAnxzG79 wr3u0T8CCK kqqrxkMZhA VEzgIUJUCU szci6uBbFw HuHCTXDfKc jF0OVRR4rZ 40ZSzqzUWJ mrSrpB7L80 y7818NXH6T FwpSF4xOFu CcwKPMYk0Z 8rX3rOKS7h aitMI0zXHB MnvPtKPr4t 1RHeCS88nD mJQ7cheyvf G2eAaoXnP6 7Lw4MIhIHs Hjab6knRv8 BCgjBZfxiQ d0bBe1d3rl 42YzHDcBZo B6VaPkoYw4 HqmURsML5j jDup4FzJns KLU0apKDL4 GRzkDmIGa2 zTy4nGIq0Y n5364wkYaH 4y41wqWSOo wqsD9PiwXL pGgzSfUbE3 aLBxc8hIhG gfOTQqhQ2a z2VcIDcMms PyYliqBk3M 7L7p8JgElZ wnibW4lR8N 6hQEc5f61F bRgdXtQKte d3PNAbBu2G lI7KoP37OD BqyuuV5hnl hRIlbksHak ouvlAEvxUb aMRmwxCaOo hzuZCYVtIQ mXobr9SBVK VGHrz4Qv1n ab7Zzmg7as FB3TUekRdE FCsOBgKX0W q7GRVdba95 opooS0avMm pAVJAOhxk7 xR5YisMI3G qLV2Qlevsv dLFrJ7nY12 3CYLztkX9J qPHzuUAkHc bt2edR3Vu2 xeJYOWRV39 zB2k7h2GoD nEx7Oh3p6Y 4KgsZP02Sx NK9k9KxF9f uTynEJikpS jk6LAtCCya 4SHYOD1Jk3 XZfVTY7Yos v9gvzwkJyU g9c3CmQu3q ikjkro62xq RSBiqwMzAM 7XS9Neq03q 6UoHkF01te nCkySePE0I jIpKyLaVqw oVTwhZV1FL Bqx1eryOzD LMxzVxZP8t t0JZJfAZKB b2vA0VsHoG 8v7zEgJqfH aoOOE68oIO gMrTGRqE6X X4GBwJdlab 2TdxapWCG2 n13PliAQuW irFbiX04sV 5AzDjTY1FV 9Gw4LInErO G8I2yGzD9K qfn6DJPMpO CUvO5VXtbg pzDRolz51G 5aGVncANIA avZByxbhaB 51TVvBOPox nCYErZ8FQE 7m4qDkWpSi c0DkqnqSYz 44WvUmO5C8 Fs0Bk6ky7s 5JPAkp48uP yrTI62l0Ym Vg2CvJVnEn vbaJlZWwDN hgPH6M3NHt pSQ5jYvwuw XvXQsGpQmx MQ9UqDknge RTS5ix3t7B XNA2kSsWOs xTa1RNtSUS dTbKJW0Phh UIvOvToZTd Q6Aj5ek5So TlHh4pnCCO h0jMUSIBB8 pVWX9BV4U9 cjsE0nnltF 4mHQEA2Q73 f4ROzMltVb nudCKgHr84 8V0i4qSQk0 oLK2WI8Iu3 013ajbKjw6 kufBkw6eGH 4l21B1YKQp QG3qi6tn7S mJoPVWsrIz xmgHfmHVTE BKA7yCDGNF MG3cNRI2b4 yJo3YzzRk1 rfBJXthFcy 9QxUs01SXJ ValipZRPxr K04yYFEYio 4VijL8Wqxu nJVrZ4FqZo Hvcl1XUA39 KtTYRpvNjd q2UQ60oA64 4c4nOZsXCT Hp4claGm0Q BaIMK3NQLD Xatyu2ojcJ syq3pZbjlU LiMpbDPCah cJlhAXVI6i uWyNZfFAsD OWD7C4zNOs yMnTvgWUgO mGFUtqMAlT T20vXKCkqQ cI6QdhPsOa YHo7rbfXqZ A8E1f9FbbY dH1RerGM6Z wyBjq7JSCU XKzLPE7Tbp Lvxsl3nYyN SBc4XnFtif CPDX0J2alg ryXgAyzdbl QdVUKN6JmB 1OwS83hUcc WgGQmMdwIq eP4uDgHGSq FXBmejdPVZ ZGgoOA52tE tHTZuRA2S2 KhOcnVr3lo D37rpS8IMB igHPYb7fDX kGhBvMT0hH XCDKOApW6Y Oi53OB9qen Ua9e0Wn8oR m1xGZ2M6VI tA8G82Reuk 0XOtp3DSup DIq0ie5bBU 6ZpBDFvl8d AfR03PkfXj OTq0cI33yp bbOLBJiqdE DzOOncj6nk VcFZOAlK9t ivb4gWgPzF lUQzSdWFHZ 9B5f9OfkmH QFa3HMko3w s6vkXR6lpd poKIuXD2m3 Pwr1RtxiwF dSilzXHl99 GVCn23sKNM DlcQOTIqY0 RTFaXrGokI 0XKrNy3l6a Z6MvYbYula 6hy3aMMWYg SjBv8HVWqe ZhfS13OSjf LLEoPr6PgR CSdfBjw3Ah 2eNtQmiqO3 s1bBwQFXkz hcvNnxEZq9 WbZtxqosPW 0DGSxTfXBB 19AwA5Ctbq RbP0NFTlxE nGg5UuPnhJ M0nC47vV2R I4BOsxTgwc HG5FUzHmuS 64gw7hdPeS xrTXYNE1Fm OFESWhHmqr UzlRMmu1WL LIs8SRan4X JzX3ejcwfZ w8kRXTzNwZ cNh5mujCex Sp9WtT0hGo 8u3FcDYxtS MdHIv92YrF ZLo8FGdTtf tcRfAEkKf5 j1opCzOxKg 3vq6CsNHh8 LJm2LqHYgL 0UpVIP21dk ep5o0vxJLF qHfYAtdrRN jJf5zvhKya GoGXAcEiRe BgdMaZ5RNA u2PF0TaC8w TrQsgVvR1g MaV9vGp6oq XhKICz7ipu LsPwMVY26F qCPrseg66p KKmQzViZmS 7UH8vnSK2Q azTNSXp9wr j3oPhvmc49 JmtpzPkZA1 gOryqOr9ae whegKvJOgn hmlFKBRBPP LLuxFWXmA7 Omw4haAkgy CRqijKTT9l xNQc2kqym2 tXekFlHeS6 Ptfc4mWsK0 PNHCNU9Dkb DxjkGOzzLi YHG2RE4isl 8RglZ7fNu0 AU736HnKwH mKgRiUyDfV VjkR3TpGoL PEjM8kRvHV EUWOe95QD4 j3GKRqapek n8sKCAbf6p U4bOb8SjDc 2ebAvsr4eO Dp86s2w3yC ByDn9HxzQL uPDi8MWDOf OTC3iOlzeB plnqpEnfbD elw0C29GtD tpRFQi7d4Q Ez0vNdIPpS s1sDjD2iU9 7lXb0nVPnk xOBs7bZX4c CDfOgAHltw Y3CxdTjlbp 70KE3HDUbT k8HbqZ4o0g RJTGe5bl5f fSxLh7eHZu TxSz709Hpb ORsXc1yTZk qw2DtVjR1U hwANK11UCX FleFYZ8Hjb gP4zz3oC79 c5hN09H87G q6LHKhc9kM wZ6XRfXfo3 aEmlZXc59l QuLiv935yV b7FZMYiNoO Vxs42ZpBl9 b6UIXPVsDa 32gweB1Z10 iMsVgbBsOF jo6Nz6hI5k 7T7EVIrs0V CG2NMie0As 2XWeC4kHRl 30n6HFPIBp lX0a6xCJru RlTHI8nXcP mzo8gx05Ym qktwNunVEY Cmtpoi6XOe mOR1To5zl4 487OUM9sh1 uZWvmxXbFM pGQXSSkyKZ YW4iT4mMgJ znBAnYhPZZ X02TFGNhcx TuQEBzr8XM 59RYfauiTp IwAf7dOYi9 vppXbYTenJ D4p1g0PESl L8IkHM3PXh FfWbGy1WFp DSVBcPfSnt yqbJwoHQbq uLgFs91cVe jzWXy4Wtvh AgX5oeUxUB h24stUZG4d hNT9hvkCbp 0RmzeBaBZY nc1CJWaHUG xd6BO6p9Qf EtDdRqbuEf MIWIKlO2vd NgdsYZmweZ xNfkB62eAn 7Gt3N4wYva wnkFEHaGtb mYBcU98vcC vcMFSD1SHa gHOZ9QzKjC nH8QdudCGu x2ryONW8HD QEGFbrrYm8 9lDsi7hwX8 VkXQR2o6HC vQu4e56cSm Pt5VAXbBAi LG6CSNqGYW iN7k4fDVHC 8EBawOmsy5 gsFfnEvpD6 tQezfduEfA RPvHlQS8AJ MTJs2BCnzy x9o3EYBRaL njdKgVYE7W A8U28ajp3r XWfSoWEJQq Dc1Fztgst5 VGWiqTWCuN EfTnSC2khh n0BCYqC53J EAXgcNz8Tj cslJaayZwS 1XJu7GRhoi xkf7Ya8t7n stBB6Tm249 mIgIzsB8UH GQGl7r3c2A NwbzAZ7CLH iBVTssjhX6 UZPLIAY58P iQaRQWu3D3 MUcfYWyzrM 8bxT5NyKX2 xrVsTXP1YQ mwuKBs8BZg jra6cknN44 VhCQmc80GR a5KWLp9M1s zEbEtNw3Yq GYOhwzMXON yWkPGGZB7O kzq0U4YWbA ZqT7o5CNqG P2lpLqK8qr 1y4QL29yKg LWYmspbrNa HkPETdq7Sh olwb1LUH0N b0BLi66eDb bHehG6In8M snzg5jH4nd ZUHrVLDMIv eDtbBhorcW U8NYaw13dT D5jzo3fGxb Crap2DKnCR jmFeqpCbKh FpCCDlz5B0 NnxkxbphTM dtfgNRBhxj 3kM5sSoYaV iCop4vGvwx 9ymNCKofmk oVi2v2neUS dldquw20Se 8bX69xIF0x VPyE4mSkgk IWSfEDDxUA ZmPd7tC8kN ZrYpIHjrQP HV6ISHn87t 23mAkDaaon A450EwGtEg 5X1EW1r3fc Ngx70kQcj8 IeCde9edM1 CFtlgN1P5L KS4deucYWR gEw16INBQN 8YPbGnAwAb eOEFedYbU0 wkU5wcSAC1 iN6nFEzcfs JrUP9MxQIh 06w8XY6GvV SP7uNr2cag 0xxGcKY2C7 JkXsMi9fSV xoKwsUe2up DwVUd8TURH I5wukLKysr RUriiuEdv5 QOLDdSPsG7 CLcRGoYC9E kv03AstejU LSF42buVBu NgXRRPXsXB qfMNhGF4tN LE8V9PhZNu ffo388KWuK KLY1KbpPFV jnSPHy1vEN e4jE0JDhPr jPWJvTW3RE HBRnDBGBZe x8pxki2Vqo jBfTJtrKfF 34xJ0Sd7LS Dc38Nbxm72 Dx4RkpgfQN ItexgLkoSA cOL8PDFyCF 391kZaXSce bsZzPvL88H j6AWCHICul Xx5kpmrX2Z RA3yhCMwkG hbzP6S0TNL 0lB2eC8mk8 QoYuZ4sLhn Ne0sWJfNUq gvrdipahWn PNthNQWRyi mutUy9UK91 n5xk6YlhRx u3yef0WFS7 V6SfmEUXjP z7tLvm8WVK nxX83leyI7 dVkLvUHHEI bdCZA3tcJv LOV7SbmUmF dtTFXdojA4 EJufmFIBUL hfpsDMXA39 gMjEwevfax JCkKmJp1VS BDZV7S2ntE fjNIfYgnCg vBBOsNEl1p IgiQoJJYyJ gGgOT3LALy TZVFdGnj7o j3W1HIkln6 2QsHGQxUY2 frBSGcRZ3w lfG0Yx4lmV VNTGNnfaJp S6lrQVMsMO FeY6q46eqs dtCsZmx1Wa 6fM1kcFqKk 1xNwaSKCRU 3qMJquQFl0 2f4Yx97nla OoI2ELmdEk 3cPovUQMqt I95v3chbLF lgddzuSOXt 76b8qMWepl hxXUm0iV6V RSEdHaPgTU FelCBln4yl xVpISPGpEK fjcXR49H2w RKZIAuwYML FRgcOzv8yK 66tjp7qKUQ 9lQDiNjsHY c99MOMDK6v 05KZ042l1p FSfjruv6lV O2h2wAg6rZ HmYTsfNpeC imrGafzU7x 1uksljipZO 39jZBujRlq YbJIdptdly DpIW64PMcH 7Y0RzzpZcR fPKuxRQoFW 0zpYJT5IJu UXgK2WByMx MSX467kvNk i2By2ZkWHT fiYDcLzZam K9DLbIKwsn 7ciKVOgKMR c8JbWW3xh8 DNV6a8VxoN gSXKP2Pa9l np8p8Yrvze asbkZKo4ta C24OKmMTH2 9skwS1quFi 7WZTETEFCe 1vi6HnnEZr hf8AKtKBSb nJqEDDVjr7 soVjbSt0fZ 41UZp9mqWu 7o8Fyd0gTl xKlOqrFgAV gmLy0MYejF pCBIJAFCfQ z9n2wmAsU7 QQaYvhHU8X CNJ4ODEGIx GaMn5tG5ro 7ZKsbPZI2y l6yxxbqvHE YgM8QbnNr9 ZajRHtjDO3 99wcfMZaGx qCg2cTB5zS lLl88TOb45 bKQQ4QQPHZ sU2AyEL5Ic YCCpkvolM1 5VcSHNCz6V V7JYMli5Cz zJ9I4f75LM OQv1VgEsFx cpf2ZhYtvC b3Mp7OUEt1 R24hpmBwCT iG6QFqcUrK LLVAgVV27t zZEV0jfoBQ sFRpqb5eNf 4oYVflocvw 8eP1hzDd78 ELAkWenKeA LpfWpW72Hr vkB08g5zOk dfjWb7IsCl yPvnU71Wjw i1Cwllu8bZ n4aza8JMsZ IkkIYiBcC7 HZzvW7AdW7 jOBljjdWZz nb37NSABBx PLIyLe2Hrk 8NIh3KQt6h q9olQESKNr jPJ1eMd3W9 zQiWhGryD3 imRWYxy5tV pfN5pNnsE1 uiPmukPkrK IhSv1tpa8N Tr1HlAH0ti dKTWezeDnW hpsyWcyGaT hCuqxUS5zF DGzvNHMd7i rxu44JeLPU l7K5pD7ViP 5FQIG9BPKH 75Je9NE5pL LFbbTBHFi8 X3l0MX498r pPSnjrZVAe 34DMBE00eu s9rNVu5qKE PUPv1NRyDW iMPB3CBBJH F80qfSwp9I mGNcCnYJGu GvIbmHmEJX JGvUiHgjOD UY2EKMFNqZ 0PFQWqO86g LWIu797dqa RBBKOWhoh7 LGUljBIIBu pHrplTXcjP sd7EhcZ27p X6AWZo3j66 VSXZaoiL3y MoAwGHpmiG 6wHQ0zUnJk eZDsQiXStI T1Xm710ano u6Coc6oaLx VbPPREnlVh 9Zh8PLHGIX Pdv1lp3qKC 5ShVFJHPLy XgIWPjSm3d hfO0IQpuqP ZDYkcSn1fA xBV3eflnxR p4kh4s2llo FNmMKKPB7i 7r9IaXu51h nbwtv8xz98 y0g0cYdWc8 Vi2PvMuwcy pMsuvF52K2 ad28KvkPUD S2qsM2vrwO JzuSBRpPHg iPegigeKN7 ocJ1UIvv6a p7cZiYghNt 8ZizSknIRa cA0g1t108k cmnQq1lo2D H5zHc2W4Dc NBqj1M7RVV M3cyLl2DiV tzTXiSqsyU 0u8x5doSgr TFme8pmccW VARuSHU9L7 6Mq9bsXcVw Co3jVNJlSd JZizHQx7J1 EQlI1ulmvb iVRgs0mzhx RZ78UpZ5t0 n4bh6I2MRj MacfI35XhV 0ggIzCzlBw 7BRACN2n7L WvYTt9Z2jC aTsGYJ15qp mAzoNGzbqW Seoa9TdvkE jWW5MRI00v XdbIZj0ypO c0VdJbE8wS 9aW30Z8ZFt z2XdjBDGBB TEU88TvbzG njQmHzqZng Xu76MgraGZ quMeuYI9Hn XJbfsTZKeu P6L355rvvm sGKAjDXsec FsxLjp1Tbf UnXyOl2AIq rKQnkZyswz W732CPx5UB FzGIFtKsIk qz1KXmcuMX coXfcxLRZL sALlwv020U aUPvywXKzZ Tb55PK5brX pX9crZbREx waslYK0ATT c9I8lONOLZ 35irtn1jqo FfepEpZgBV WiEzAlh9YA v2dNeruukI eYYFqJGsLL KLaqSamdcQ P87IbcFZ9i BW39XqOsww 8NcwfRRSts HRydtAhK4X q2OpYXA9YT IE6ItNvkxZ DMQwMYZy1N OM1uPgznvp 53koIwGhwx ZkZUBgbj7Z jbc2Knebez iCNvZqw3k5 n6GZlFEDwY kcipu3P3tw DhnGTUaERe 4aUO4Mz9k3 59xvdmf0A7 ML476JQyA4 0XFLParqKN 51xB4Na6MR eNptZhuxdB QnV6VZbxGP 4sQlY5HEWc LbTRUClTg8 MUaPaKpNYa DhDVw5gcF8 sZYM7LjgHN 1QhSam2FOe ZnCmSRmdM1 XhYmh59xh5 rAYauoK4vl rTBX1KLzgr 3K4gVlkUdJ GHZzm4Hhst etRUF2CM9k XyRhwh7Ln8 rtvGmQsP7J M1j4Z1Tc0w PYK9ueznU3 K7Bu0D2jgm vN93BstbLc niyTLgaMmO Y9cToSVQg9 jPbJ2KjQJ2 W0JEfiWEvs F2JNKrqXqX 3Ug1J8FWlL ZzK9LDFQNP k8DGn89Ous frKkNQ5lyH u326IjAXzQ 5Yy3ViIpnz S4zadNFmyA LipenCxoeT q5J6xwLwmA auF65FUvvZ 2V8SDF9IHh rwgzW1EffK xRJT5g9Bbg yShoE4zCKo S1Km7ch09e F17LfUR6EJ pTBzsPKJuZ HO57p9BbaR ufBHX2Vhsb LQQT579ncb 0Oqjnw9Brl L156tUJVxV rGhD1o6GRB zqLvW2rkfX HsksGEf2uP KzUDNu99Fy YjtZi4U4Jw d4G4tNfwsw OVyihj5vpH pKcxCd2g7J RQVAsropry vPaurBS0y0 rvIfnQFrRU 0sjqzChdkY RkAAOdXutt n22cYdetRY 1W7d0j5XPN ann7428vCN tc5ePMjkmK fnwvLpQozw hqPOwLyRav Kg5kFKqevQ F4XKd7B5Ae mamKopC6wQ NijwEjtLai 7EbDb7t1lD BCQPrNIZht IgQbvucQyy XYa7Z5G0TK dUs6FQ5kCT OaLtbi9a2g HIxY4mXza7 Bp9mrfGyeR GdWfGEjRK9 AlvkAqFLdd c3ob7PGSID 7IRKVjjCtA 0LMe1YZ1RK cPrgZx9eJG 3UZbJ8ZrgE 8v1L94odWk ZQJt83Y9Xk NhSFDFfhol 1WElcXmQix KFnau2MzIu wFjW03KBy0 48SUnJmzGe Hnq6jKgKF2 hhLBxnWlzY iqlcd1I2MI bYicXFm2fE Z8BBzdFfls V9goAQfHS3 eiZqbW7YDW w8EBu5A5NI K12bjVuXjf 7NMXKUcHlk r53vJXNr2k 07IXRkIqDh gg93ihLelm Iz3e5CazSJ r9s0mCQpG2 vYgZtq7uky PSEh0KbSI0 9u22Bw3gxY NaGMf26FX4 NlH7aDT7LO 9yAff1qOY0 wBctOT3jaO SP7w5Z6iJa RTdT9N1fq1 fDUdlE6Ulo bHIaYfg9bq ZIbShrcfXT T2X8oGyidS SVzFiS6miA f6dPCz0uus znG7az81Vp nhh9PLakMx 9U1ogXeZfu xPRg5GrnvQ jMByR0hF0W hRKTO73JpC oT0cngjswB iznS63ACHd hXjupxTIQM rqMZKy4miW m95oFeXFIr Og3bSb4r4N g30b7FaCQQ WOIdcQaTVA Ie3oOYcpuW PNv3YmnxVC pOrh2QMHJE udteoZIZnr HzD861Xnge tPVSQlzHBH 7vZjRFYqbT ngbHzpoYia YYUpVHADvo JNgTWDFrFQ YqgmAj4xQH RafVpBtEiE 5DlebwoqCE 2jUSAZjmUT dFAQMsgrlx k4I5l0um5y s9aPOvHTG0 jYWPXgdlQk rI7JamAoqQ YvsiWqyg1z YmVkpzdQBt gA7q1eW1aE E0VMhLWqUJ 91t0lTG5cA HqlnR6V4d0 zpytUcXut8 fTnxeZwK4K Lbu7h0By7d OvT4yAdCqY Jskn7LXzrU AW9pkIDldB YbLkV1to6E ETOafJRGHC p8LdVJh4id 0KHi0Th0mS kXzgAKVS27 93twdQgGrH bHotmryrGn WT8EQiQMyr cgSYpMtzhJ AT33so2y5m Ds2vyybSaF 4UwrZguBtX 9FMNt4uvHG ncy6qZSV59 c1yhCQL2yS QFq1o2bF4K 1CACN1kjtw 8oxm7oRBTT KLqjSgl8bL wdUU7G9s3X y3iehmpzRU WenbeTwcwv S6rzR1b0wh VvLpKQdXZo ywBOTvO7H4 FUVJVTR8dG DTGuICrvGE HcCAzUQAgO 7gJ91SUdda LAfM2EgHRy Kj2eTLOA2L g87vwoSQPU R3xsvekjjM O2RbdXpzSj ilSkZcbsWP 31qnYIelYI JuEhuunsnh uijN8N6U2n DVaC27TmVl 00aDHXHtYh KSU3FspYXz XOv1EAFrLn 8xCRo5A33g wZMoRFG8m3 2ObMofmH0C 0Z3bj8DRiw OdiD2J9zwN IWUmky9OC6 itKxoFyYdE jWBp8zSPHO v0tpk7MPjk MKgFeCai0F tHxOosykdN FSFHLFJ8kM Ym50nwgRox HFJY6m2Lyn VB4xAOaUIk s21z0VS0Ol iYiZGd6ZuL 5NZg216lO5 vmzJPjlbXI wpWgyKrurE 9uwZ0C7Kvq rNWwII0nb0 P6Tnri8xqL VMN2hteqKs kfrbkxXWMS AvyIb9GYo9 dalCC2UUsd jHWfrJGWKt BWNBFnuvx6 c4TB2o643G t1GN8mteOG 6oQhvREAk7 LrEDxKiqGk bLwBzENlLV eIeWJcsSBu bv1hjP6ibj LQ0E8Q72a3 Skj2YmbtT9 OZzW7pEPHr A8VeueWphX RG60Fxb1kP Jr2NanUJkg klDlwuiG1C 0TlcgaunO9 LHM7jJHk7W 9Vc1fYYWDq h4o29bd5rk MQVgJuyFNL PYSDmvoxgU eF3WPbifHj PhNlBZIUl4 Py0BmUcXx7 RgUeDGMMIx Dah9es81ZY vZ29KrNYeC S4WsbMkDYp xgkEfH2xe2 vAW8dZdjn2 HVQ8WWC55n SQqpX8ZKuL kwNM9hJZKH KZ4muFs5iR xXdwLjvM7X X1S4z5qjoT ONNgq106it EKQyMNAceK 1JhAKBB09U bL5rKOziNV QCJ3ZM5AH7 eRm3PmlHCq ORUWzBn5im St3ffZoSC9 UIqqRnfU55 7yG0ZFKnBv 4e5mD3ZTWH ZM6yi8ilPu vjDsGcrxFz OHECVV4kap gUZTZvslFY VziZtiFtDb kLdAAFhik1 vqbS6pJc5A naYNb1osJG LYFEZGyRIQ b7DtgxsQJV jR9GERu9JY nondYbt67r tIgpLBgrY3 Wfgj3geVmu DntrNCEgCp UrNIfWctGX AI5FhFLN7B n4gyTTJB6x nKGbHl1NTw K38MLP1w6h 8qvbvDK14M nZ6miJv5cY VtuBIqWl79 7IrPPBMBch HA6bztKJJV IueKLyQ9oc r6oOW8KnVI veJeCIYagG IggLLFruGB JNQlatAScV zke4s5L8Dh F4NdScXiYE 5WmWTExXZn EStDmSXMHO cNBlOJzbNo Z6Me0QFoOC zXkhddDYC3 k5SuNes33z ymU7rPyGEj fifsQ9fUvP Q06JR0hHHF fpvaNHcaRG sUYw7kfRNq 7hFl44SiCS wvMkyMvqaU E5q94VCk64 9TkKZpsFsT QVfW8NV0Ce 37I43iHOgt H9DSRzONIO ObKN4qa7H9 DVj1PFsAND ddeK0NVgfw 8fVUGmGmzH jc5fbsDPfy 9c0jEsJFBm HJCobNNr6a CBMQXXKCOz FhYI6sEIzO SEYXNnlFRj qXrsi0Ev0a 37XOsld9MS xOWc3f0rjV 9PrLV3IQuX daEzlorO9t sL3ORDZWfd IrT89tZhqt 09NOriCkW4 zYl5JzJSXi ThRLljE7UD nYgfDb9xSU 37j55uEbhI UpXs2lCZks rNSK9hLt6R ceRBy7IJYJ wNSHhpq8zc eY6pNY2Uye E7FpJ6kWwH 2kPLkFk1WF iYs08sUvLh EfKKKy4aBY VEbRFwR5SR ZXWYx1wE3g NFRvDdlaON Cb0LOcb9tg Vkp6TNQCys K2G4ZLVRXp 2X3Iz7Ly3s l1l0zHfmtZ SOfE4jwaw6 wcbZRZfVw4 DrtGP6TUId RKokd8krU5 I1kmubFww5 5oPlVdJ7tp X6IMY2U8B2 LNvqjHG5KV 5r1VpZkal7 TB5lo5CUGl jCXdmHjWvW XlRxHCVI6Y 31lGLYdwUB iQuriBYfiC E98dCd7a6I pdvmQJXazG yzZA9Vpl24 JvvGg3vHbl jSdhj2uiAP lJHKyT4JPf SJ6LUX1RGO 3RcOP5jkT3 NuRk5joc6m LVZIUTmL9c HfERlyDLGm Vzz8SeML0i LRGoYv8FLo vNZdfIhpZe XkbEIOUYJX Wq9IW5zhbV HosSAQOUBI BDgJCUoaVW luAowFf1ms WDS97wDB6M up1CtTxKTE N9iwX5vRhM Lo6rZhfL8L YiXY5p4ChO PVHKgKwtsH QVrskn9ZwS EibTyPu15E XvIlrMErda HYh8LPDcSR 3ZvuNmCKMw FPF8nlvJ7m jBvdgomDQC hapg2l4oLb LMw0pLjlUe 6hvxf1MstW dP8j6Sd7WW K7bAyzXVV3 HLSsJJGnz8 BTEfLOZ3A0 hMBCsoPvgX ymdeCryqHi BU5opLNU9k sPrr8Rhs2g 4UoXOEVkvS ZoxESP5YFA z0ixW4eNuA 2usgAPMHeJ DREZzE4DqS CApGSqHz7w aA94s2AfVs BPhbYe2gFu 1OGPuYkCeV dWgcKSucdu L2PtQxgidv yLRAAe8JRZ thguFxlowm Mu6TXINcq2 8VgNUwHbKa 1w5rurCrnM lJqXWk4M5c Nv8P0TtvG8 AXgp7XPdol pyFlxNog8c iVL5coM0VR vsEfSICgRI LRcqxXOpD1 PKTYhST4mz uPm7X3ysRQ jNglOrzTdp VOXB3K10HD QQheAfKkKp KKGzRT9kDK yGUY7p5ZHh SaM6Lj1gpG 0U7PnxnBWh jSvrZ85jqO gIE6SYikwd ycCb3kd47w Z6oRjuCRIR Rir1Ig11Wu xjyp235996 A9OzpW42hr 2C8m5oTI0n Kv502TXSFr 6Kk55MQgA7 ZS9IiS2u4M pVBIYgv4tz EzkcOhjp3O wyxpKnTc59 mxn1B3h5J0 hyT7DkVQlP oW3EFnVPGL gJYGEzf2iN Pwi4STsmmz 7fespM43iI yUeL1d7gIB tWA82qk7Yt d73vlPPb4T ceR0Oj0WHc KpSM7GaGTd wXnVx7zYR0 GBw5JGz2ON YlFTwky67K f8TgpYTv45 md1Ji1LXE5 zIfq2XCBx4 162y7vjcF9 VJNGlxEMm3 lKWI6lS8sT EMbduWRlr1 XsfXHVXFfB ItuqQJeAoG 43By3TB0mp RqaJOpu1Q2 aRG47TFA7b lJtmY2ihmn oJKN60eV6g 6FpFFmixl6 ryN9bG1cxa ZlgDo5zOGE YTZkZJUFqo ez6clAf9ed pUj9pVHUXR 9b18joCHMJ GBcWm1x0Ic bpPVlmLLeR WHDOuKHHun KKUSYpl5l7 0hsOFe0K43 M6VEajTzXX Z2VodQxLZu ehJi0Y5QKt vqVqzVSwTV hnb38yZAfV W72KwY9SHz";
        List<String> ret = new ArrayList<>();
        for (String i : fileSuffixes.split(" ")) {
            ret.add("azure-test/1/tmp." + i);
            ret.add("azure-test/1/2/tmp." + i);
            ret.add("azure-test/1/2/1/tmp." + i);
            ret.add("azure-test/1/2/2/tmp." + i);
            ret.add("azure-test/1/2/3/tmp." + i);
            ret.add("azure-test/1/3/tmp." + i);
            ret.add("azure-test/2/tmp." + i);
            ret.add("azure-test/3/tmp." + i);
        }
        ret.add("azure_backup/__palo_repository_azure_repo1/__ss_yyq/__meta.ff9b669c1505f51993d5fb448a345811");
        ret.sort(String::compareTo);
        return ret;
    }

    @Test
    public void testMockObj() {
        AzureObjStorage azs = genMockedAzureObjStorage(2);
        List<RemoteFile> result = new ArrayList<RemoteFile>();
        boolean fileNameOnly = false;
        Status st = azs.globList("s3://gavin-test-jp/azure-test/1/*", result, fileNameOnly);
        Assertions.assertTrue(st.ok());
        // for (RemoteFile i : result) {
        //     System.out.println(i.getName());
        // }
    }

    /**
     * Mock an AzureObjStorage of which getPagedBlobItems() will return objects and split into multiple batches
     * for testing continuations
     *
     * @param numBatch control how many PagedResponse will return when calling getPagedBlobItems()
     * @return a mocked AzureObjStorage
     */
    public static AzureObjStorage genMockedAzureObjStorage(int numBatch) {
        Map<String, String> props = new HashMap<String, String>();
        props.put(S3Properties.ACCESS_KEY, "gavintestus");
        props.put(S3Properties.SECRET_KEY, "sksksksksksksk");
        props.put(S3Properties.ENDPOINT, "https://blobz.blob.core.windows.net");
        props.put(S3Properties.BUCKET, "gavin-test-us");
        AzureProperties azureProps = (AzureProperties) StorageProperties.createPrimary(props);
        AzureObjStorage azs = new AzureObjStorage(azureProps);
        List<String> allBlobKeys = genObjectKeys();
        final Integer[] batchIndex = {0}; // from 0 to numBatch
        new MockUp<AzureObjStorage>(AzureObjStorage.class) {
            @Mock
            public PagedResponse<BlobItem> getPagedBlobItems(BlobContainerClient client, ListBlobsOptions options,
                                                             String newContinuationToken) {
                batchIndex[0]++;
                return new PagedResponse<BlobItem>() {
                    public final List<List<String>> batches = new ArrayList<List<String>>() {
                        {
                            List<String> all = allBlobKeys.stream().filter(x -> x.startsWith(options.getPrefix()))
                                    .collect(Collectors.toList());
                            int numPerBatch = all.size() / numBatch;
                            int cnt = 0;
                            for (int i = 0; i < numBatch; ++i) {
                                List<String> batch = new ArrayList<>();
                                for (int j = 0; j < numPerBatch; ++j) {
                                    batch.add(all.get(cnt++));
                                }
                                add(batch);
                            }
                            for (; cnt < all.size(); ++cnt) { // remainder
                                get(size() - 1).add(all.get(cnt));
                            }
                        }
                    };

                    @Override
                    public IterableStream<BlobItem> getElements() {
                        List<BlobItem> blobItems = new ArrayList<>();
                        batches.get(batchIndex[0] - 1).forEach(x -> blobItems.add(new BlobItem().setName(x).setIsPrefix(false)
                                .setProperties(new BlobItemProperties().setContentLength(17L)
                                .setLastModified(OffsetDateTime.now()))));
                        return new IterableStream<>(blobItems);
                    }

                    @Override
                    public String getContinuationToken() {
                        return (batchIndex[0] >= numBatch) ? null : "what ever it is";
                    }

                    @Override
                    public int getStatusCode() {
                        return 200; // OK
                    }

                    @Override
                    public HttpHeaders getHeaders() {
                        return null;
                    }

                    @Override
                    public HttpRequest getRequest() {
                        return null;
                    }

                    @Override
                    public void close() throws IOException {

                    }
                };
            }
        };
        return azs;
    }
}
