/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.query.filter;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;
import org.apache.druid.java.util.common.StringUtils;
import org.apache.hive.common.util.Murmur3;

public class BloomKFilter {
    public static final float DEFAULT_FPP = 0.05f;
    public static final int START_OF_SERIALIZED_LONGS = 5;
    private static final int DEFAULT_BLOCK_SIZE = 8;
    private static final int DEFAULT_BLOCK_SIZE_BITS = (int)(Math.log(8.0) / Math.log(2.0));
    private static final int DEFAULT_BLOCK_OFFSET_MASK = 7;
    private static final int DEFAULT_BIT_OFFSET_MASK = 63;
    private static final ThreadLocal<byte[]> BYTE_ARRAY_4 = ThreadLocal.withInitial(() -> new byte[4]);
    private final BitSet bitSet;
    private final int m;
    private final int k;
    private final int totalBlockCount;

    public BloomKFilter(long maxNumEntries) {
        BloomKFilter.checkArgument(maxNumEntries > 0L, "expectedEntries should be > 0");
        long numBits = BloomKFilter.optimalNumOfBits(maxNumEntries, 0.05f);
        this.k = BloomKFilter.optimalNumOfHashFunctions(maxNumEntries, numBits);
        int nLongs = (int)Math.ceil((double)numBits / 64.0);
        int padLongs = 8 - nLongs % 8;
        this.m = (nLongs + padLongs) * 64;
        this.bitSet = new BitSet(this.m);
        BloomKFilter.checkArgument(this.bitSet.data.length % 8 == 0, "bitSet has to be block aligned");
        this.totalBlockCount = this.bitSet.data.length / 8;
    }

    public BloomKFilter(long[] bits, int numFuncs) {
        this.bitSet = new BitSet(bits);
        this.m = bits.length * 64;
        this.k = numFuncs;
        BloomKFilter.checkArgument(this.bitSet.data.length % 8 == 0, "bitSet has to be block aligned");
        this.totalBlockCount = this.bitSet.data.length / 8;
    }

    static void checkArgument(boolean expression, String message) {
        if (!expression) {
            throw new IllegalArgumentException(message);
        }
    }

    static int optimalNumOfHashFunctions(long n, long m) {
        return Math.max(1, (int)Math.round((double)m / (double)n * Math.log(2.0)));
    }

    static long optimalNumOfBits(long n, double p) {
        return (long)((double)(-n) * Math.log(p) / (Math.log(2.0) * Math.log(2.0)));
    }

    public static void serialize(OutputStream out, BloomKFilter bloomFilter) throws IOException {
        DataOutputStream dataOutputStream = new DataOutputStream(out);
        dataOutputStream.writeByte(bloomFilter.k);
        dataOutputStream.writeInt(bloomFilter.getBitSet().length);
        for (long value : bloomFilter.getBitSet()) {
            dataOutputStream.writeLong(value);
        }
    }

    public static BloomKFilter deserialize(InputStream in) throws IOException {
        if (in == null) {
            throw new IOException("Input stream is null");
        }
        try {
            DataInputStream dataInputStream = new DataInputStream(in);
            byte numHashFunc = dataInputStream.readByte();
            int bitsetArrayLen = dataInputStream.readInt();
            long[] data = new long[bitsetArrayLen];
            for (int i = 0; i < bitsetArrayLen; ++i) {
                data[i] = dataInputStream.readLong();
            }
            return new BloomKFilter(data, numHashFunc);
        }
        catch (RuntimeException e) {
            IOException io = new IOException("Unable to deserialize BloomKFilter");
            io.initCause(e);
            throw io;
        }
    }

    public static void mergeBloomFilterBytes(byte[] bf1Bytes, int bf1Start, int bf1Length, byte[] bf2Bytes, int bf2Start, int bf2Length) {
        int idx;
        if (bf1Length != bf2Length) {
            throw new IllegalArgumentException("bf1Length " + bf1Length + " does not match bf2Length " + bf2Length);
        }
        for (idx = 0; idx < 5; ++idx) {
            if (bf1Bytes[bf1Start + idx] == bf2Bytes[bf2Start + idx]) continue;
            throw new IllegalArgumentException("bf1 NumHashFunctions/NumBits does not match bf2");
        }
        for (idx = 5; idx < bf1Length; ++idx) {
            int n = bf1Start + idx;
            bf1Bytes[n] = (byte)(bf1Bytes[n] | bf2Bytes[bf2Start + idx]);
        }
    }

    public static void serialize(ByteBuffer out, BloomKFilter bloomFilter) {
        BloomKFilter.serialize(out, out.position(), bloomFilter);
    }

    public static void serialize(ByteBuffer out, int position, BloomKFilter bloomFilter) {
        ByteBuffer view = out.duplicate().order(ByteOrder.BIG_ENDIAN);
        view.position(position);
        view.put((byte)bloomFilter.k);
        view.putInt(bloomFilter.getBitSet().length);
        for (long value : bloomFilter.getBitSet()) {
            view.putLong(value);
        }
    }

    public static BloomKFilter deserialize(ByteBuffer in) throws IOException {
        return BloomKFilter.deserialize(in, in.position());
    }

    public static BloomKFilter deserialize(ByteBuffer in, int position) throws IOException {
        if (in == null) {
            throw new IOException("Input stream is null");
        }
        try {
            ByteBuffer dataBuffer = in.duplicate().order(ByteOrder.BIG_ENDIAN);
            dataBuffer.position(position);
            byte numHashFunc = dataBuffer.get();
            int bitsetArrayLen = dataBuffer.getInt();
            long[] data = new long[bitsetArrayLen];
            for (int i = 0; i < bitsetArrayLen; ++i) {
                data[i] = dataBuffer.getLong();
            }
            return new BloomKFilter(data, numHashFunc);
        }
        catch (RuntimeException e) {
            throw new IOException("Unable to deserialize BloomKFilter", e);
        }
    }

    public static void mergeBloomFilterByteBuffers(ByteBuffer bf1Buffer, int bf1Start, ByteBuffer bf2Buffer, int bf2Start) {
        int idx;
        int bf2Length;
        ByteBuffer view1 = bf1Buffer.duplicate().order(ByteOrder.BIG_ENDIAN);
        ByteBuffer view2 = bf2Buffer.duplicate().order(ByteOrder.BIG_ENDIAN);
        int bf1Length = 5 + view1.getInt(1 + bf1Start) * 8;
        if (bf1Length != (bf2Length = 5 + view2.getInt(1 + bf2Start) * 8)) {
            throw new IllegalArgumentException("bf1Length " + bf1Length + " does not match bf2Length " + bf2Length);
        }
        for (idx = 0; idx < 5; ++idx) {
            if (view1.get(bf1Start + idx) == view2.get(bf2Start + idx)) continue;
            throw new IllegalArgumentException("bf1 NumHashFunctions/NumBits does not match bf2");
        }
        for (idx = 5; idx < bf1Length; ++idx) {
            int pos1 = bf1Start + idx;
            int pos2 = bf2Start + idx;
            view1.put(pos1, (byte)(view1.get(pos1) | view2.get(pos2)));
        }
    }

    public static int getNumSetBits(ByteBuffer bfBuffer, int start) {
        ByteBuffer view = bfBuffer.duplicate().order(ByteOrder.BIG_ENDIAN);
        view.position(start);
        int numLongs = view.getInt(1 + start);
        int setBits = 0;
        int i = 0;
        int pos = 5 + start;
        while (i < numLongs) {
            setBits += Long.bitCount(view.getLong(pos));
            ++i;
            pos += 8;
        }
        return setBits;
    }

    public static int computeSizeBytes(long maxNumEntries) {
        BloomKFilter.checkArgument(maxNumEntries > 0L, "expectedEntries should be > 0");
        long numBits = BloomKFilter.optimalNumOfBits(maxNumEntries, 0.05f);
        int nLongs = (int)Math.ceil((double)numBits / 64.0);
        int padLongs = 8 - nLongs % 8;
        return 5 + (nLongs + padLongs) * 8;
    }

    public static void add(ByteBuffer buffer, byte[] val) {
        BloomKFilter.addBytes(buffer, val);
    }

    public static void addBytes(ByteBuffer buffer, byte[] val, int offset, int length) {
        long hash64 = val == null ? 2862933555777941757L : Murmur3.hash64((byte[])val, (int)offset, (int)length);
        BloomKFilter.addHash(buffer, hash64);
    }

    public static void addBytes(ByteBuffer buffer, byte[] val) {
        BloomKFilter.addBytes(buffer, val, 0, val.length);
    }

    public static void addHash(ByteBuffer buffer, long hash64) {
        int hash1 = (int)hash64;
        int hash2 = (int)(hash64 >>> 32);
        int firstHash = hash1 + hash2;
        if (firstHash < 0) {
            firstHash ^= 0xFFFFFFFF;
        }
        ByteBuffer view = buffer.duplicate().order(ByteOrder.BIG_ENDIAN);
        int startPosition = view.position();
        int numHashFuncs = view.get(startPosition);
        int totalBlockCount = view.getInt(startPosition + 1) / 8;
        int blockIdx = firstHash % totalBlockCount;
        int blockBaseOffset = blockIdx << DEFAULT_BLOCK_SIZE_BITS;
        for (int i = 1; i <= numHashFuncs; ++i) {
            int combinedHash = hash1 + (i + 1) * hash2;
            if (combinedHash < 0) {
                combinedHash ^= 0xFFFFFFFF;
            }
            int absOffset = blockBaseOffset + (combinedHash & 7);
            int bitPos = combinedHash >>> DEFAULT_BLOCK_SIZE_BITS & 0x3F;
            int bufPos = startPosition + 5 + absOffset * 8;
            view.putLong(bufPos, view.getLong(bufPos) | 1L << bitPos);
        }
    }

    public static void addString(ByteBuffer buffer, String val) {
        BloomKFilter.addBytes(buffer, StringUtils.toUtf8((String)val));
    }

    public static void addByte(ByteBuffer buffer, byte val) {
        BloomKFilter.addBytes(buffer, new byte[]{val});
    }

    public static void addInt(ByteBuffer buffer, int val) {
        BloomKFilter.addBytes(buffer, BloomKFilter.intToByteArrayLE(val));
    }

    public static void addLong(ByteBuffer buffer, long val) {
        BloomKFilter.addHash(buffer, Murmur3.hash64((long)val));
    }

    public static void addFloat(ByteBuffer buffer, float val) {
        BloomKFilter.addInt(buffer, Float.floatToIntBits(val));
    }

    public static void addDouble(ByteBuffer buffer, double val) {
        BloomKFilter.addLong(buffer, Double.doubleToLongBits(val));
    }

    public void add(byte[] val) {
        this.addBytes(val);
    }

    public void addBytes(byte[] val, int offset, int length) {
        long hash64 = val == null ? 2862933555777941757L : Murmur3.hash64((byte[])val, (int)offset, (int)length);
        this.addHash(hash64);
    }

    public void addBytes(byte[] val) {
        this.addBytes(val, 0, val.length);
    }

    private void addHash(long hash64) {
        int hash1 = (int)hash64;
        int hash2 = (int)(hash64 >>> 32);
        int firstHash = hash1 + hash2;
        if (firstHash < 0) {
            firstHash ^= 0xFFFFFFFF;
        }
        int blockIdx = firstHash % this.totalBlockCount;
        int blockBaseOffset = blockIdx << DEFAULT_BLOCK_SIZE_BITS;
        for (int i = 1; i <= this.k; ++i) {
            int combinedHash = hash1 + (i + 1) * hash2;
            if (combinedHash < 0) {
                combinedHash ^= 0xFFFFFFFF;
            }
            int absOffset = blockBaseOffset + (combinedHash & 7);
            int bitPos = combinedHash >>> DEFAULT_BLOCK_SIZE_BITS & 0x3F;
            int n = absOffset;
            this.bitSet.data[n] = this.bitSet.data[n] | 1L << bitPos;
        }
    }

    public void addString(String val) {
        this.addBytes(StringUtils.toUtf8((String)val));
    }

    public void addByte(byte val) {
        this.addBytes(new byte[]{val});
    }

    public void addInt(int val) {
        this.addBytes(BloomKFilter.intToByteArrayLE(val));
    }

    public void addLong(long val) {
        this.addHash(Murmur3.hash64((long)val));
    }

    public void addFloat(float val) {
        this.addInt(Float.floatToIntBits(val));
    }

    public void addDouble(double val) {
        this.addLong(Double.doubleToLongBits(val));
    }

    public boolean test(byte[] val) {
        return this.testBytes(val);
    }

    public boolean testBytes(byte[] val) {
        return this.testBytes(val, 0, val.length);
    }

    public boolean testBytes(byte[] val, int offset, int length) {
        long hash64 = val == null ? 2862933555777941757L : Murmur3.hash64((byte[])val, (int)offset, (int)length);
        return this.testHash(hash64);
    }

    private boolean testHash(long hash64) {
        int hash1 = (int)hash64;
        int hash2 = (int)(hash64 >>> 32);
        int firstHash = hash1 + hash2;
        if (firstHash < 0) {
            firstHash ^= 0xFFFFFFFF;
        }
        int blockIdx = firstHash % this.totalBlockCount;
        int blockBaseOffset = blockIdx << DEFAULT_BLOCK_SIZE_BITS;
        for (int i = 1; i <= this.k; ++i) {
            int bitPos;
            int wordOffset;
            int absOffset;
            long bloomWord;
            int combinedHash = hash1 + (i + 1) * hash2;
            if (combinedHash < 0) {
                combinedHash ^= 0xFFFFFFFF;
            }
            if (0L != ((bloomWord = this.bitSet.data[absOffset = blockBaseOffset + (wordOffset = combinedHash & 7)]) & 1L << (bitPos = combinedHash >>> DEFAULT_BLOCK_SIZE_BITS & 0x3F))) continue;
            return false;
        }
        return true;
    }

    public boolean testString(String val) {
        return this.testBytes(StringUtils.toUtf8((String)val));
    }

    public boolean testByte(byte val) {
        return this.testBytes(new byte[]{val});
    }

    public boolean testInt(int val) {
        return this.testBytes(BloomKFilter.intToByteArrayLE(val));
    }

    public boolean testLong(long val) {
        return this.testHash(Murmur3.hash64((long)val));
    }

    public boolean testFloat(float val) {
        return this.testInt(Float.floatToIntBits(val));
    }

    public boolean testDouble(double val) {
        return this.testLong(Double.doubleToLongBits(val));
    }

    private static byte[] intToByteArrayLE(int val) {
        byte[] bytes = BYTE_ARRAY_4.get();
        bytes[0] = (byte)(val >> 0);
        bytes[1] = (byte)(val >> 8);
        bytes[2] = (byte)(val >> 16);
        bytes[3] = (byte)(val >> 24);
        return bytes;
    }

    public long sizeInBytes() {
        return this.getBitSize() / 8;
    }

    public int getBitSize() {
        return this.bitSet.getData().length * 64;
    }

    public int getNumSetBits() {
        int setCount = 0;
        for (long datum : this.bitSet.getData()) {
            setCount += Long.bitCount(datum);
        }
        return setCount;
    }

    public int getNumHashFunctions() {
        return this.k;
    }

    public int getNumBits() {
        return this.m;
    }

    public long[] getBitSet() {
        return this.bitSet.getData();
    }

    public String toString() {
        return "m: " + this.m + " k: " + this.k;
    }

    public void merge(BloomKFilter that) {
        if (this == that || this.m != that.m || this.k != that.k) {
            throw new IllegalArgumentException("BloomKFilters are not compatible for merging. this - " + this + " that - " + that);
        }
        this.bitSet.putAll(that.bitSet);
    }

    public void reset() {
        this.bitSet.clear();
    }

    public static class BitSet {
        private final long[] data;

        public BitSet(long bits) {
            this(new long[(int)Math.ceil((double)bits / 64.0)]);
        }

        public BitSet(long[] data) {
            assert (data.length > 0) : "data length is zero!";
            this.data = data;
        }

        public void set(int index) {
            int n = index >>> 6;
            this.data[n] = this.data[n] | 1L << index;
        }

        public boolean get(int index) {
            return (this.data[index >>> 6] & 1L << index) != 0L;
        }

        public int bitSize() {
            return this.data.length * 64;
        }

        public long[] getData() {
            return this.data;
        }

        public void putAll(BitSet array) {
            assert (this.data.length == array.data.length) : "BitArrays must be of equal length (" + this.data.length + "!= " + array.data.length + ")";
            for (int i = 0; i < this.data.length; ++i) {
                int n = i;
                this.data[n] = this.data[n] | array.data[i];
            }
        }

        public void clear() {
            Arrays.fill(this.data, 0L);
        }
    }
}

