/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.druid.data.input.impl.systemfield;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonValue;
import org.apache.druid.java.util.common.IAE;
import org.apache.druid.segment.column.ColumnType;

/**
 * System fields that can appear when reading data. These are generated by the system itself, rather than actually
 * being present in the data being scanned.
 *
 * Currently these are only used by {@link SystemFieldInputSource}, and are therefore part of the package that
 * contains things related to {@link org.apache.druid.data.input.InputSource}. Perhaps, in the future, system fields
 * may be generated from segment scans as well (for example "__segment_id" returning the
 * {@link org.apache.druid.timeline.SegmentId}). At that point this enum may move to a more central location.
 */
public enum SystemField
{
  URI("__file_uri", ColumnType.STRING),
  BUCKET("__file_bucket", ColumnType.STRING),
  PATH("__file_path", ColumnType.STRING);

  private final String fieldName;
  private final ColumnType columnType;

  SystemField(final String fieldName, final ColumnType columnType)
  {
    this.fieldName = fieldName;
    this.columnType = columnType;
  }

  @JsonCreator
  public static SystemField fromFieldName(final String fieldName)
  {
    for (final SystemField field : values()) {
      if (field.getFieldName().equals(fieldName)) {
        return field;
      }
    }

    throw new IAE("No such system field[%s]", fieldName);
  }

  /**
   * Name of this system field.
   */
  @JsonValue
  public String getFieldName()
  {
    return fieldName;
  }

  /**
   * Type of this system field.
   */
  @SuppressWarnings("unused") // Not used, but still useful for signifying intent
  public ColumnType getColumnType()
  {
    return columnType;
  }

  @Override
  public String toString()
  {
    return fieldName;
  }
}
