<!--
RULES
* Never place a link to an external site like Google Doc. The proposal should be in this issue entirely.
* Use a spelling and grammar checker tools if available for you (there are plenty of free ones).

PROPOSAL HEALTH CHECK
I can read the design document and understand the problem statement and what you plan to change *without* resorting to a couple of hours of code reading just to start having a high level understanding of the change.

IMAGES
If you need diagrams, avoid attaching large files. You can use [MermaidJS]([url](https://mermaid.js.org/)) as a simple language to describe many types of diagrams.

THIS COMMENTS
Please remove them when done.
-->

# PIP-351: Additional options for Pulsar-Test client to support KeyStore based TLS

# Background knowledge

<!--
Describes all the knowledge you need to know in order to understand all the other sections in this PIP

* Give a high level explanation on all concepts you will be using throughout this document. For example, if you want to talk about Persistent Subscriptions, explain briefly (1 paragraph) what this is. If you're going to talk about Transaction Buffer, explain briefly what this is. 
  If you're going to change something specific, then go into more detail about it and how it works. 
* Provide links where possible if a person wants to dig deeper into the background information. 

DON'T
* Do not include links *instead* explanation. Do provide links for further explanation.

EXAMPLES
* See [PIP-248](https://github.com/apache/pulsar/issues/19601), Background section to get an understanding on how you add the background knowledge needed.
  (They also included the motivation there, but ignore it as we place that in Motivation section explicitly).
-->

In both Pulsar Client and Pulsar Admin, we support the use of KeyStores. This feature is provided by means of the boolean
"useKeyStoreTls". The boolean is also the only way authentication mechanisms such as AuthenticationKeyStoreTls can be utilised
properly, as the logic to use keystores for SSL Connections, from either ClientConfigurationData stored in Pulsar Admin/Client
or AuthData hinges on the "useKeyStoreTls" boolean as can be seen below:

<b>AsyncHttpConnector.java</b>
```java
if (conf.isUseKeyStoreTls()) {
    KeyStoreParams params = authData.hasDataForTls() ? authData.getTlsKeyStoreParams() :
            new KeyStoreParams(conf.getTlsKeyStoreType(), conf.getTlsKeyStorePath(),
                    conf.getTlsKeyStorePassword());

    final SSLContext sslCtx = KeyStoreSSLContext.createClientSslContext(
            conf.getSslProvider(),
            params.getKeyStoreType(),
            params.getKeyStorePath(),
            params.getKeyStorePassword(),
            conf.isTlsAllowInsecureConnection(),
            conf.getTlsTrustStoreType(),
            conf.getTlsTrustStorePath(),
            conf.getTlsTrustStorePassword(),
            conf.getTlsCiphers(),
            conf.getTlsProtocols());

    JsseSslEngineFactory sslEngineFactory = new JsseSslEngineFactory(sslCtx);
    confBuilder.setSslEngineFactory(sslEngineFactory);
}
```

None of these options can be currently configured when using Pulsar Test client.

# Motivation

<!--
Describe the problem this proposal is trying to solve.

* Explain what is the problem you're trying to solve - current situation.
* This section is the "Why" of your proposal.
-->

As we already let users both extend authentication and use just the keystore and truststore properties to set up mTLS
connections, without using any authentication plugin class, a lot of them might want to use this method of authentication
during Performance Testing as well.

I understand that currently mTLS (for testing purposes) can be achieved by using trust and client certificates.
However, the issue of users extending authentication plugin classes and utilizing keystores is still not covered 
with the current options. Therefore, I propose we make these already existing options be configured in test clients,
increasing its usability.

# Goals

## In Scope

Create new Arguments for the following properties, in PerformanceBaseArguments.java :
1. useKeyStoreTls
2. trustStoreType
3. trustStorePath
4. trustStorePass
5. keyStoreType
6. keyStorePath
7. keyStorePass

Update the code to change between TrustCerts and TrustStore based on useKeyStoreTls.

<!--
What this PIP intend to achieve once It's integrated into Pulsar.
Why does it benefit Pulsar.
-->

[//]: # (## Out of Scope)

<!--
Describe what you have decided to keep out of scope, perhaps left for a different PIP/s.
-->


[//]: # (# High Level Design)

<!--
Describe the design of your solution in *high level*.
Describe the solution end to end, from a birds-eye view.
Don't go into implementation details in this section.

I should be able to finish reading from beginning of the PIP to here (including) and understand the feature and 
how you intend to solve it, end to end.

DON'T
* Avoid code snippets, unless it's essential to explain your intent.
-->

# Detailed Design

## Design & Implementation Details

<!--
This is the section where you dive into the details. It can be:
* Concrete class names and their roles and responsibility, including methods.
* Code snippets of existing code.
* Interface names and its methods.
* ...
-->

Add the options for utilizing keystores as part of performance base arguments, along with forwarding their values
to the client/admin builders.

## Public-facing Changes

<!--
Describe the additions you plan to make for each public facing component. 
Remove the sections you are not changing.
Clearly mark any changes which are BREAKING backward compatability.
-->

### CLI

All places we utilize Pulsar Test client, for example Pulsar-Perf will have the following new options:

1. --use-keystore-tls &rarr; Default value = false
2. --truststore-type &rarr; Default value = JKS, Possible values = JKS, PKCS12
3. --truststore-path &rarr; Default value = ""
4. --truststore-pass &rarr; Default value = ""
5. --keystore-type &rarr; Default value = JKS, Possible values = JKS, PKCS12
6. --keystore-path &rarr; Default value = ""
7. --keystore-pass &rarr; Default value = ""



# Backward & Forward Compatibility

The change will not affect any previous releases. The options can also be brought to previous versions, however, I have
noticed that Pulsar has moved away from JCommander in Version 3.2.x to Picocli (currently in master)
Therefore, to add these options to previous versions, the code has to be replicated to those versions.
