#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/stat.h>
#include <iostream>
#include <fstream>
#include <string>
#include <list>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <openssl/x509.h>

#include "regexpression.h"
#include "namespacespolicy.h"


namespace AuthN {

  const std::string issuername = "TO Issuer";
  const std::string permit  = "PERMIT";
  const std::string deny    = "DENY";
  const std::string subjectname = "Subject";
  const std::string selfissuer = "SELF";
  const char policy_suffix[]   = ".namespaces";

  static void get_line(std::istream& in,std::string& s) {
    for(;;) {
      s.resize(0);
      if(in.fail() || in.eof()) return;
      getline(in,s);
      std::string::size_type p;
      p=s.find_first_not_of(" \t");
      if(p != std::string::npos) s=s.substr(p);
      if((!s.empty()) && (s[0] != '#')) break;
    };
    return;
  }

  static void get_one_rule(std::istream& in,std::string& s) {
    //Combine a statment that is across multiple lines into one string
    s.resize(0);
    std::string s1;
    do {
      s1.resize(0);
      get_line(in,s1);
      std::string::size_type pos;
      pos = s1.find("\\");
      if(pos != std::string::npos) {
        s1 = s1.substr(0, pos); 
        s.append(s1).append(" ");
        continue;
      }
      else { if(!s1.empty())s.append(s1); break; }
    } while(1);
    return;
  }

  static int memicmp(const void *s1, const void *s2, size_t n) {
    const unsigned char *p1 = (const unsigned char *) s1;
    const unsigned char *p2 = (const unsigned char *) s2;
    for (; n-- > 0; p1++, p2++) {
      int c;
      if (*p1 != *p2 && (c = (toupper(*p1) - toupper(*p2)))) return c;
    }
    return 0;
  }

  static std::string::size_type find(const char* s, int s_length, const char* to_find, int to_find_length) {
    int n = s_length - to_find_length;
    std::string::size_type pos = 0;
    while(n-- >= 0) {
      if(memicmp(s, to_find, to_find_length) == 0) return pos;
      ++s; ++pos;
    }
    return std::string::npos;
  }

  static void get_word(std::string& s,std::string& word) {
    std::string::size_type w_s;
    std::string::size_type w_e;
    word.resize(0);
    w_s=s.find_first_not_of(" \t");
    if(w_s == std::string::npos) { s.resize(0); return; };
    if(s[w_s] == '\'') {
      w_e=s.find('\'',++w_s);
    } else if(s[w_s] == '"') {
      w_e=s.find('"',++w_s);
    } else {
      w_e=s.find_first_of(" \t",w_s);
    };
    if(w_e == std::string::npos) w_e=s.length();
    word=s.substr(w_s,w_e-w_s);
    if((s[w_e] == '\'') || (s[w_e] == '"')) ++w_e;
    w_s=s.find_first_not_of(" \t",w_e);
    if(w_s == std::string::npos) w_s=w_e;
    s=s.substr(w_s);
    return;
  }


  // TODO: the solution below is not very robust regarding wrong order of tokens.
  // It must be redone in more robust way. For example by tokenizing strings 
  // and then going sequentially through obtained tokens.

  static bool get_issuer(std::string& s, std::string& issuer, std::string& error) {
    issuer.resize(0);
    std::string::size_type f;
    std::string str;
    f = find(s.c_str(), s.length(), issuername.c_str(), issuername.length());
    if(f == std::string::npos) {
      error += "Missing issuer token in namespaces policy\n";
      return false;
    }
    str = s.substr(f + issuername.length());
    get_word(str, issuer);
    if(issuer.empty()) {
      error += "Missing issuer in namespaces policy\n";
      return false;
    }
    return true; 
  }

  static bool get_right(std::string& s, std::string& right, std::string& error) {
    right.resize(0);
    std::string::size_type f;
    std::string str;
    f = find(s.c_str(), s.length(), permit.c_str(), permit.length());
    if(f != std::string::npos) {
      right = "permit";
      return true;
    }
    f = find(s.c_str(), s.length(), deny.c_str(), deny.length());
    if(f != std::string::npos) {
      right = "deny";
      return true;
    }
    error += "Missing right in namespaces policy\n";
    return false;
  }

  static bool get_subject(std::string& s, std::list<std::string>& patterns, std::string& error) {
    patterns.resize(0);
    std::string::size_type f;
    std::string str;
    f = find(s.c_str(), s.length(), subjectname.c_str(), subjectname.length());
    if(f == std::string::npos) {
      error += "Missing subjects token in namespaces policy\n";
      return false;
    }
    str = s.substr(f + subjectname.length());
    std::string subject;
    get_word(str,subject);
    if(subject.empty()) {
      error += "Missing subjects in namespaces policy\n";
      return false;
    }
    patterns.push_back(subject);
    return true;
  }

  static bool match_all(const std::string& issuer_subject,const std::string& subject,const std::string& policy_ca_subject,std::list<std::string> policy_patterns) {
    if(issuer_subject == policy_ca_subject) {
      std::list<std::string>::iterator pattern = policy_patterns.begin();
      for(;pattern!=policy_patterns.end();++pattern) {
        std::string::size_type p = 0;
        for(;;) {
          p=pattern->find('*',p);
          if(p == std::string::npos) break;
          pattern->insert(p,"."); p+=2;
        };
        (*pattern)="^"+(*pattern)+"$";
        AuthN::Utils::RegularExpression re(*pattern);
        bool r = re.match(subject);
        if(r) return true;
      };
    };
    return false;
  }

  static void X509_NAME_to_string(std::string& str,const X509_NAME* name) {
    str.resize(0);
    if(name == NULL) return;
    char* s = X509_NAME_oneline((X509_NAME*)name,NULL,0);
    if(s) {
      str=s;
      OPENSSL_free(s);
    };
    return;
  }

  static void string_to_X509_NAME(X509_NAME** name, const std::string& str) {
    if(name == NULL) return;
    if(*name != NULL) { X509_NAME_free(*name); *name = NULL; }
    *name = X509_NAME_new();
    if(*name == NULL) return;

    // The input string is supposed to be /O=Grid/OU=EMI/CN=UserTest
    std::string::size_type pos1 = 0, pos2;
    pos1 = str.find("/", 0);
    while(pos1 != std::string::npos) {
      std::string entry_name, entry_value;
      pos2 = str.find("=", pos1);
      if(pos2 == std::string::npos) break;
      entry_name = str.substr(pos1+1, pos2-pos1-1);
      pos1 = str.find("/", pos2);
      if(pos1 == std::string::npos) entry_value = str.substr(pos2+1);
      else entry_value = str.substr(pos2+1, pos1-pos2-1);

#ifdef HAVE_OPENSSL_OLDRSA 
      int r = X509_NAME_add_entry_by_txt(*name, (char*)(entry_name.c_str()),
        MBSTRING_ASC, (unsigned char*)(entry_value.c_str()), -1, -1, 0);
#else
      int r = X509_NAME_add_entry_by_txt(*name, entry_name.c_str(),
        MBSTRING_ASC, (const unsigned char*)(entry_value.c_str()), -1, -1, 0);
#endif
      if(!r) { /* TODO: Error handling should go here */ }
    };
    return;
  }

  bool NamespacesPolicy::Match(const std::string& subject_str) {
    X509_NAME* name = NULL;
    string_to_X509_NAME(&name, subject_str);
    if(!name) return false;
    return Match(name);
  }

  bool NamespacesPolicy::Match(const X509_NAME* subject) {
    error_.resize(0);
    if(!file_) return false;
    std::string subject_str;
    std::string s;
    std::string policy_ca_subject;
    std::string policy_right;
    std::list<std::string> policy_patterns;
    bool right = false;
    bool failure = false;
    X509_NAME_to_string(subject_str, subject);
    bool permit = false;
    for(;;) {
      right = false;
      failure = false;
      get_one_rule(*file_, s);
      if(s.empty()) break;
      //logger.msg(INFO, "Get rule from namespaces file: %s", s.c_str());

      policy_ca_subject.resize(0);
      if(!get_issuer(s, policy_ca_subject, error_)) failure = true;
      //If the "SELF" token is contained.
      //According to the specification: 
      //The (quoted) issuerDirectoryName may be replaced by the single 
      //token “SELF”, in which case this policy applies to the issuer 
      //whose hash corresponds to the hash contained in the namespaces file name
      if(memicmp(policy_ca_subject.c_str(), selfissuer.c_str(), policy_ca_subject.length()) == 0) {
        policy_ca_subject.resize(0);
        policy_ca_subject = issuer_;
      }

      policy_right.resize(0);
      if(!get_right(s, policy_right, error_)) failure = true;
      else { right = (policy_right == "permit"); }

      policy_patterns.resize(0);
      if(!get_subject(s, policy_patterns, error_)) failure = true;

      if((!policy_ca_subject.empty()) && (!failure)) {
        //According to eugridpma's namespaces format specification: 
        //The policy can either permit or deny the issuer the right 
        //to issue subjects with specific names. A denial overrides 
        //any permissive statements in the same file.
        bool r = match_all(issuer_, subject_str, policy_ca_subject, policy_patterns);
        if(r && right) {
          //logger.msg(INFO, "The issuer: %s is permitted to sign the subject: %s", issuer_.c_str(), subject_str.c_str());
          permit = true;
        }
        else if(r && !right) { //Denial overrides
          //logger.msg(INFO, "The issuer: %s is explicitly denied to sign the subject: %s", issuer_.c_str(), subject_str.c_str());
          return false;
        }
        else {
          //logger.msg(INFO, "The subject: %s and issuer: %s in the verified certificate does not match any namespaces policies files", subject_str.c_str(), issuer_.c_str());
        }
      };
    };
    return permit;
  }

  NamespacesPolicy::NamespacesPolicy(const std::string& issuer_subject_str, const std::string& ca_path): issuer_(issuer_subject_str), file_(NULL) {
    X509_NAME* name = NULL;
    if(issuer_.empty()) {
      error_ = "Issuer for namespaces location is not defined";
      return;    
    }
    string_to_X509_NAME(&name, issuer_);
    if(!name) {
      error_ = "Failed to parsed X509_NAME from string";
      return;
    }
    Init(name, ca_path);
    X509_NAME_free(name);
  }

  NamespacesPolicy::NamespacesPolicy(const X509_NAME* issuer_subject, const std::string& ca_path): file_(NULL) {
    Init(issuer_subject, ca_path);
  }

  void NamespacesPolicy::Init(const X509_NAME* issuer_subject, const std::string& ca_path) {
    if(!issuer_subject) {
      error_ = "Issuer for namespaces location is not defined";
      return;
    }
    X509_NAME_to_string(issuer_,issuer_subject);
    unsigned long hash = X509_NAME_hash((X509_NAME*)issuer_subject);
    char hash_str[32];
    snprintf(hash_str,sizeof(hash_str)-1,"%08lx",hash);
    hash_str[sizeof(hash_str)-1]=0;
    std::string fname = ca_path+"/"+hash_str+policy_suffix;
    struct stat st;
    if(stat(fname.c_str(),&st) != 0) {
      error_ = "The namespaces location "+fname+" for "+issuer_+" was not found";
      return;
    }
    if(!S_ISREG(st.st_mode)) {
      error_ = "The namespaces location "+fname+" for "+issuer_+" is not a regular file";
      return;
    }
    file_ = new std::ifstream(fname.c_str());
    if(!(*file_)) { delete file_; file_ = NULL; };
  }

  NamespacesPolicy::~NamespacesPolicy() {
    if(file_) delete file_;
  }

}

