#ifndef COMMAND_HANDLERS_H_
#define COMMAND_HANDLERS_H_
#include "commandhandlersinclude.h"
#include "shareitemmodel.h"
#include "searchentry.h"
#include "queueitem.h"
#include "filetransfer.h"
#include "exceptions.h"
#include "userfilemodel.h"

#include <QCoreApplication>
#include <QMessageBox>

class User;
class FinishedItem;

namespace ui_cmd_handlers
{

class RpcCommandEvent : public QEvent
{
public:
    RpcCommandEvent(const anylist& params) : QEvent(QEvent::User), parameters(params)
    {}
    ;
    const anylist& getParams() const
    {
        return parameters;
    }
private:
    const anylist parameters;
};

class GuiCommandHandlerBase : public QObject, public rpc::RpcCommandHandler
{
    Q_OBJECT
public:
	GuiCommandHandlerBase(const char* name,
						  boost::function<void( int, const rpc::CmdPtr&)> sender,
						  int numParams ) :
		rpc::RpcCommandHandler(name, sender,numParams ),dialogFence(false) {}

	//! Called by the RpcDriver.    
	virtual void handleCommand(int /*clientId*/, const anylist& params)
    {
        RpcCommandEvent* event = new RpcCommandEvent(params);	// deleted by qt
        QCoreApplication::postEvent(this,event);
    }

protected:
    virtual void customEvent(QEvent* e)
    {
        RpcCommandEvent* ce = static_cast<RpcCommandEvent *>(e);
		try {
			handleCommand(ce->getParams());
		} catch(const boost::bad_any_cast& e) {
			if(!dialogFence) {
				dialogFence = true;
			QMessageBox::critical(0,tr("Backend Communication Error"), tr("We encountered an error while communicating with the backend! Perhaps versions mismatch?"));
			dialogFence = false;
			}
		} catch(const RpcHandlerException& re) {
			if(!dialogFence) {
				dialogFence = true;
				QMessageBox::critical(0,tr("Backend Communication Error"), QString("Exception: ") + re.getMethodName() + " " + re.getMessage());
				dialogFence = false;
			}
		}
    }
    virtual void handleCommand(const anylist& params) = 0;
	private:
		bool dialogFence;
};



class HubConnectedHandler : public GuiCommandHandlerBase
{
    Q_OBJECT
public:
    HubConnectedHandler(boost::function<void( int, const rpc::CmdPtr&)> sender ) :
            GuiCommandHandlerBase("hubConnected", sender,NUM_PARAMS )
    {}
protected:
	//! Implements the actual command parsing and signal emission.
    virtual void handleCommand(const anylist& params);
private:
    static const short NUM_PARAMS = 1;
signals:
    void hubConnected(int);
};

class HubUpdatedHandler : public GuiCommandHandlerBase
{
    Q_OBJECT
public:
    HubUpdatedHandler(boost::function<void( int, const rpc::CmdPtr&)> sender ) :
            GuiCommandHandlerBase("hubUpdated", sender,NUM_PARAMS )
    {}
protected:
	//! Implements the actual command parsing and signal emission.
    virtual void handleCommand(const anylist& params);
private:
    static const short NUM_PARAMS = 2;
signals:
    void hubUpdated(int,const QString&);
};

class HubStatsHandler : public GuiCommandHandlerBase
{
	Q_OBJECT
	public:
    HubStatsHandler(boost::function<void( int, const rpc::CmdPtr&)> sender ) :
		GuiCommandHandlerBase("hubStats", sender,NUM_PARAMS )
		{}
	protected:
	//! Implements the actual command parsing and signal emission.
		virtual void handleCommand(const anylist& params);
	private:
		static const short NUM_PARAMS = 2;
	signals:
		void hubStats(int,qint64);
};

class ConnectionFailedHandler : public GuiCommandHandlerBase
{
    Q_OBJECT
public:
    ConnectionFailedHandler(boost::function<void( int, const rpc::CmdPtr&)> sender ) :
            GuiCommandHandlerBase("connectionFailed", sender,NUM_PARAMS )
    {}
protected:
	//! Implements the actual command parsing and signal emission.
    virtual void handleCommand(const anylist& params);
private:
    static const short NUM_PARAMS = 2;
signals:
    void connectionFailed(int,const QString&);
};

class PrivateChatHandler : public GuiCommandHandlerBase
{
   Q_OBJECT
public:
	PrivateChatHandler(boost::function<void( int, const rpc::CmdPtr&)> sender ) : 
		GuiCommandHandlerBase("privateChatMessage", sender,NUM_PARAMS) {}
protected:
	virtual void handleCommand(const anylist& params);
private:
	static const short NUM_PARAMS = 3;
signals:
	void privateChat(int session,const QString& user,const QString& msg);

};

class ChatMessageHandler : public GuiCommandHandlerBase
{
    Q_OBJECT
public:
    ChatMessageHandler( boost::function<void( int, const rpc::CmdPtr&)> sender ) :
            GuiCommandHandlerBase("chatMessage",sender,NUM_PARAMS ) {}
protected:
    virtual void handleCommand(const anylist& params);
private:
    static const short NUM_PARAMS = 2;
signals:
    void chatMessage(int session,const QString&);
};

class UsersUpdatedHandler : public GuiCommandHandlerBase
{
    Q_OBJECT
public:
    UsersUpdatedHandler( boost::function<void( int, const rpc::CmdPtr&)> sender ) :
            GuiCommandHandlerBase("usersUpdated",sender,NUM_PARAMS ) {}
protected:
    virtual void handleCommand(const anylist& params);
private:
    static const short NUM_PARAMS = 2;
signals:
    void usersUpdated(int session,QList<User*>);
};


class settingsInfoHandler : public GuiCommandHandlerBase
{
    Q_OBJECT
public:
    settingsInfoHandler(boost::function<void( int, const rpc::CmdPtr&)> sender ) :
            GuiCommandHandlerBase("settingsInfo", sender,NUM_PARAMS )
    {}
protected:
    virtual void handleCommand(const anylist& params);
private:
    static const short NUM_PARAMS = 2;
signals:
    void settingsInfo(const QList<QString>&,const QList<QVariant>&);
};


class PublicHubListHandler : public GuiCommandHandlerBase
{
	Q_OBJECT
public:
	PublicHubListHandler( boost::function<void( int, const rpc::CmdPtr& )> sender ) :
	GuiCommandHandlerBase( "newHubList", sender, NUM_PARAMS )
	{}
protected:
	virtual void handleCommand( const anylist& params );
private:
	static const short NUM_PARAMS = 1;
signals:
	void hubList( const QList<rpc_types::HubEntry>& );
};


class FavouriteHubAddedHandler : public GuiCommandHandlerBase
{
	Q_OBJECT
public:
	FavouriteHubAddedHandler( boost::function<void( int, const rpc::CmdPtr& )> sender ) :
	GuiCommandHandlerBase( "favouriteHubAdded", sender, NUM_PARAMS )
	{}
protected:
	virtual void handleCommand( const anylist& params );
private:
	static const short NUM_PARAMS = 1;
signals:
	void favHubAdded( const rpc_types::FavouriteHub& );
};


class FavouriteHubRemovedHandler : public GuiCommandHandlerBase
{
	Q_OBJECT
public:
	FavouriteHubRemovedHandler( boost::function<void( int, const rpc::CmdPtr& )> sender ) :
	GuiCommandHandlerBase( "favouriteHubRemoved", sender, NUM_PARAMS )
	{}
protected:
	virtual void handleCommand( const anylist& params );
private:
	static const short NUM_PARAMS = 1;
signals:
	void favHubRemoved( const string& );
};


class FavouriteHubListHandler : public GuiCommandHandlerBase
{
	Q_OBJECT
public:
	FavouriteHubListHandler( boost::function<void( int, const rpc::CmdPtr& )> sender ) :
	GuiCommandHandlerBase( "favouriteHubList", sender, NUM_PARAMS )
	{}
protected:
	virtual void handleCommand( const anylist& params );
private:
	static const short NUM_PARAMS = 1;
signals:
	void hubList( const QList<rpc_types::FavouriteHub>& );
};


class RunningSessionsHandler : public GuiCommandHandlerBase
{
	Q_OBJECT
	public:
    RunningSessionsHandler(boost::function<void( int, const rpc::CmdPtr&)> sender ) :
		GuiCommandHandlerBase("runningSessions", sender,NUM_PARAMS )
		{}
	protected:
		virtual void handleCommand(const anylist& params);
	private:
		static const short NUM_PARAMS = 2;
	signals:
		void sessionInfo(int id,const QString& hubname,const QString& url,const QList<User*>);
		void queueList(const QList<QueueItem>&);
};


class SessionCreatedHandler : public GuiCommandHandlerBase
{
Q_OBJECT
public:
	SessionCreatedHandler(boost::function<void( int, const rpc::CmdPtr&)> sender ) : GuiCommandHandlerBase("sessionCreated", sender,NUM_PARAMS ) {}
protected:
	virtual void handleCommand(const anylist& params);
private:
 static const short NUM_PARAMS = 1;
signals:
	void sessionCreated(int);


};


//! Handles the sharedDirs remote procedure call.
class SharedDirsHandler : public GuiCommandHandlerBase
{
	Q_OBJECT
	public:
		SharedDirsHandler(boost::function<void( int, const rpc::CmdPtr&)> sender ) : GuiCommandHandlerBase("sharedDirs", sender,NUM_PARAMS ) {}
	protected:
		virtual void handleCommand(const anylist& params);
	private:
		static const short NUM_PARAMS = 1;
	signals:
		void sharedDirs(const QList<ShareItem>);
};

//! Handles the searchResults remote procedure call.
class SearchResultsHandler : public GuiCommandHandlerBase
{
	Q_OBJECT
	public:
		SearchResultsHandler(boost::function<void( int, const rpc::CmdPtr&)> sender ) : GuiCommandHandlerBase("searchResults", sender,NUM_PARAMS ) {}
	protected:
		virtual void handleCommand(const anylist& params);
	private:
		static const short NUM_PARAMS = 2;
	signals:
		void searchResults(int,QList<SearchEntry>&);
};

//! Handles queueEvent
class QueueEventHandler : public GuiCommandHandlerBase
{
	Q_OBJECT
	public:
		QueueEventHandler(boost::function<void( int, const rpc::CmdPtr&)> sender ) : GuiCommandHandlerBase("queueEvent", sender,NUM_PARAMS ) {}
	protected:
		virtual void handleCommand(const anylist& params);
	private:
		static const short NUM_PARAMS = 1;
	signals:
		void queueItemAdded(const QueueItem&);
		void queueItemRemoved(int id);
		void queueItemFinshed(int id);
		void sourcesUpdated(int id, QList<int> sources);
		void statusUpdated(int id, int status, int currentSource, int priority);
};

//! Handles finishedEvent
class FinishedEventHandler : public GuiCommandHandlerBase
{
	Q_OBJECT
	public:
		FinishedEventHandler(boost::function<void( int, const rpc::CmdPtr&)> sender ) : GuiCommandHandlerBase("finishedEvent", sender,NUM_PARAMS ) {}
	protected:
		virtual void handleCommand(const anylist& params);
	private:
		static const short NUM_PARAMS = 2;
	signals:
		void itemAdded(const FinishedItem&);
		void itemRemoved(int);
		void allRemoved(int);
};

//! Handles transferEvent
class TransferEventHandler : public GuiCommandHandlerBase
{
	Q_OBJECT
	public:
		TransferEventHandler(boost::function<void( int, const rpc::CmdPtr&)> sender ) : GuiCommandHandlerBase("transferEvent", sender,NUM_PARAMS ) {}
	protected:
		virtual void handleCommand(const anylist& params);
	private:
		static const short NUM_PARAMS = 2;
		FileTransfer createTransfer(FileTransfer::Type type,const anylist& ft);
	signals:
		void transferStarting(const FileTransfer&);
		void transferTick( const QList<FileTransfer>& );
		void transferComplete( const FileTransfer& );
		void transferFailed( const FileTransfer&,const QString& );
};

class UserFileListHandler : public GuiCommandHandlerBase
{
	Q_OBJECT
public:
	UserFileListHandler(boost::function<void( int, const rpc::CmdPtr&)> sender ) : GuiCommandHandlerBase("userFileListing", sender,NUM_PARAMS ) {}
protected:
	virtual void handleCommand(const anylist& params);
private:
	static const short NUM_PARAMS = 2;
signals:
	void fileListing( const UserFileModelPtr& model );
};

//! Handles the userRemoved command.
class UserRemovedHandler : public GuiCommandHandlerBase
{
	Q_OBJECT
	public:
		UserRemovedHandler(boost::function<void( int, const rpc::CmdPtr&)> sender ) : GuiCommandHandlerBase("userRemoved", sender,NUM_PARAMS ) {}
	protected:
		virtual void handleCommand(const anylist& params);
	private:
		static const short NUM_PARAMS = 2;
	signals:
		void userRemoved(int session, int id);
};

class HashInfoHandler : public GuiCommandHandlerBase
{
	Q_OBJECT
	public:
		HashInfoHandler(boost::function<void( int, const rpc::CmdPtr&)> sender ) : GuiCommandHandlerBase("hashInfo", sender,NUM_PARAMS ) {}
	protected:
		virtual void handleCommand(const anylist& params);
	private:
		static const short NUM_PARAMS = 4;
	signals:
		void userRemoved(int session, int id);
};


}
#endif
