package latexDraw.parsers.pstricks;

import static java.lang.Math.cos;
import static java.lang.Math.sin;
import static java.lang.Math.toRadians;

import java.awt.Color;
import java.awt.geom.Point2D;
import java.util.Vector;

import latexDraw.figures.ArrowHead;
import latexDraw.parsers.IgnoreCommandException;
import latexDraw.parsers.InvalidFormatCommandException;
import latexDraw.parsers.NotFullyManagedException;
import latexDraw.parsers.pstricks.PSTricksParameters.PositionParameters;
import latexDraw.psTricks.DviPsColors;
import latexDraw.psTricks.PSTricksConstants;

/** 
 * This class defines actions to do on a given command: it is an
 * abstract class, because the actions to do depend on want the
 * developer want; so you must create a sub class to implement
 * actions to do as in the class {@link LaTeXDrawPSTricksParserActions}.<br>
 * <br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 09/06/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.5<br>
 */
public abstract class PSTricksParserActionsManager
{
	/** When the parsing of a command goes bad, this value is returned. */
	public static final short ERR_END_CMD_INDEX = -1;
	
	
	
	/**
	 * Action to do when a text is parsed.
	 */
	public abstract void actionsOnText(Vector<PSTricksParameters> p) throws InvalidFormatCommandException;
	
	
	
	/**
	 * Action to do when a frame or an ellipse is parsed.
	 */
	public abstract int actionsOnFrameEllipse(int line, boolean hasStar, Vector<PSTricksParameters> p, 
			String params, boolean isFrame) throws InvalidFormatCommandException;
	
	
	
	/**
	 * Action to do when a frame is parsed.
	 */
	public abstract int actionsOnCircle(int line, boolean hasStar, Vector<PSTricksParameters> ps, 
			String params) throws InvalidFormatCommandException;
	
	
	
	/**
	 * Action to do when a dot is parsed.
	 */
	public abstract int actionsOnDot(int line, Vector<PSTricksParameters> ps, String params,
			boolean noCoordinatePossible) throws InvalidFormatCommandException;
	
	
	
	/**
	 * Action to do when a picture is parsed.
	 */
	public abstract int actionsOnPicture(int line, Vector<PSTricksParameters> ps, String params) 
			throws InvalidFormatCommandException;
	
	


	/**
	 * Allows to parse the parameters of the commands psdot and psdots
	 * @param line The number of the line
	 * @param params The parameters
	 * @param center Will contains all points read
	 * @param p The parameters to be set
	 * @return The position of the end of the parameters in the string params
	 * @throws InvalidFormatCommandException If the command is invalid
	 */
	public int parseParametersDot(int line, String params, Vector<Point2D.Double> center, 
			PSTricksParameters p, boolean noCoordinatePossible) throws InvalidFormatCommandException
	{
		int i=0, j;
		boolean stop = false;
	
		while(i<params.length() && params.charAt(i)!='[' && params.charAt(i)!='(' && !stop)
		{
			if(params.charAt(i)=='\\') 
			{
				if(noCoordinatePossible)
					 stop = true;
				else throw new InvalidFormatCommandException(line);
			}
			i++;
		}
		
		if(stop)
		{
			center.addElement(new Point2D.Double(p.origin.x, p.origin.y));
			return 0;
		}
		
		if(i>=params.length()) throw new InvalidFormatCommandException(line);
		j=i;
		
		// We parse the options [..]
		if(params.charAt(i)=='[') // Analyse of the options
		{
			j = params.indexOf("]", j);//$NON-NLS-1$
			if(j==-1) throw new InvalidFormatCommandException(line);
			
			parseOptionsFigure(params.substring(i+1, j), p, !p.psCustomP.fromPsCustom);
			j++;
		}
		
		if(center==null) return ERR_END_CMD_INDEX;
		
		int[] tmp = {j};
		try
		{
			goToNextChar(params, tmp);
			if(params.charAt(tmp[0])!='(') 
			{
				if(noCoordinatePossible)
				{
					center.addElement(new Point2D.Double(p.origin.x, p.origin.y));
					return j;
				}
				throw new InvalidFormatCommandException(line);
			}

			j = params.indexOf(')', tmp[0]);
			if(j==-1) throw new InvalidFormatCommandException(line);
			
			//	The first position must be here
			Point2D.Double pt = readOneCoordinate(params.substring(tmp[0], j+1), 0, line, true); 
			if(pt==null) throw new InvalidFormatCommandException(line);
			if(Double.isNaN(pt.x)) pt.x = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
			if(Double.isNaN(pt.y)) pt.y = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
			pt.setLocation(pt.x+p.origin.x, pt.y+p.origin.y);
			center.addElement(pt);
			tmp[0] = j+1;

		}catch(IgnoreCommandException e1) { throw new InvalidFormatCommandException(line); }
		
		stop=false;
		
		if(!noCoordinatePossible)
			while(!stop)
			{
				try
				{
					goToNextChar(params, tmp);
					if(params.charAt(tmp[0])!='(')
						throw new InvalidFormatCommandException(line);
	
					j = params.indexOf(')', tmp[0]);
					if(j==-1) throw new InvalidFormatCommandException(line);
					
					Point2D.Double pt = readOneCoordinate(params.substring(tmp[0], j+1), 0, line, true); 
					if(pt==null) throw new InvalidFormatCommandException(line);
					if(Double.isNaN(pt.x)) pt.x = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
					if(Double.isNaN(pt.y)) pt.y = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
					pt.setLocation(pt.x+p.origin.x, pt.y+p.origin.y);
					center.addElement(pt);
					tmp[0] = j+1;
					
				}catch(Exception e){ stop = true; }
			} // while(!stop)
		
		return tmp[0];
	}
	
	
	
	
	
	/**
	 * Allows to parse the parameters of a include graphics command.<br>
	 * All the parameters will be in cm.
	 */
	public int parseParametersPicture(int nLine, String params, String[] path) throws InvalidFormatCommandException
	{
		try
		{
			int[] i={0};
			
			while(i[0]<params.length() && params.charAt(i[0])!='{')
			{
				if(params.charAt(i[0])=='\\') throw new InvalidFormatCommandException(nLine);
				i[0]++;
			}
			
			i[0]++;
			if(i[0]>=params.length()) throw new InvalidFormatCommandException(nLine);
			
			goToNextChar(params, i);
			
			int j = params.indexOf("}", i[0]), k; //$NON-NLS-1$
			
			if(j==-1) throw new InvalidFormatCommandException(nLine);
			
			String path2 = params.substring(i[0], j);
			k = path2.length()-1;
			boolean stop = false;
			
			while(!stop && k>i[0])
			{
				if(isCharToIgnore(path2.charAt(k)))
					k--;
				else stop = true;
			}
			
			if(k<=i[0]) throw new InvalidFormatCommandException(nLine);
			path2 = path2.substring(0, k+1);
			
			path[0] = path2;
			
			return j+1;
			
		}catch(Exception e)
		{ throw new InvalidFormatCommandException("includegraphics", nLine); } //$NON-NLS-1$
	}
	
	
	
	
	
	/**
	 * Allows to parse the parameters of a frame or a ellipse command.<br>
	 * All the parameters will be in cm.
	 */
	public int parseParametersFrameEllipse(int nLine, String params, Point2D.Double p1, Point2D.Double p2, 
			PSTricksParameters p) throws InvalidFormatCommandException
	{
			int i=0, j;
			while(i<params.length() && params.charAt(i)!='[' && params.charAt(i)!='(' && params.charAt(i)!='{')
			{
				if(params.charAt(i)=='\\') throw new InvalidFormatCommandException(nLine);
				i++;
			}
			
			if(i>=params.length()) throw new InvalidFormatCommandException(nLine);
			j=i;
			
			// We parse the options [..]
			if(params.charAt(i)=='[') // Analyse of the options
			{
				while(j<params.length() && params.charAt(j)!=']') j++;
				
				if(j>=params.length()) throw new InvalidFormatCommandException(nLine);
				
				parseOptionsFigure(params.substring(i+1, j), p, !p.psCustomP.fromPsCustom);
				j++;
			}
			
			if(p1==null || p2==null || p.origin==null) return j;
			
			// We parse the coordinates (..)
			int[] tmp = {j};
			Point2D.Double pt1, pt2;
			
			try
			{
				goToNextChar(params, tmp);
				if(params.charAt(tmp[0])!='(') throw new InvalidFormatCommandException(nLine);
				j = params.indexOf(')', tmp[0]);
				if(j==-1) throw new InvalidFormatCommandException(nLine);
				
				//	The first position must be here
				pt1 = readOneCoordinate(params.substring(tmp[0], j+1), 0, nLine, true); 
				if(pt1==null) throw new InvalidFormatCommandException(nLine);
				tmp[0] = j+1;

			}catch(IgnoreCommandException e1) { throw new InvalidFormatCommandException(nLine); }
			
			try
			{
				goToNextChar(params, tmp);
				if(params.charAt(tmp[0])!='(') throw new InvalidFormatCommandException(nLine);
				j = params.indexOf(')', tmp[0]);
				if(j==-1) throw new InvalidFormatCommandException(nLine);
				pt2 = readOneCoordinate(params.substring(tmp[0], j+1), 0, nLine, true); 
				tmp[0] = j+1;
			}catch(Exception e){ pt2 = null; }

			if(pt2==null)
			{
				p1.setLocation((Point2D.Double)p.origin.clone());
				p2.setLocation(pt1);
			}
			else
			{
				p1.setLocation(pt1);
				p2.setLocation(pt2);
			}
			if(Double.isNaN(p1.x)) p1.x = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
			if(Double.isNaN(p1.y)) p1.y = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
			if(Double.isNaN(p2.x)) p2.x = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
			if(Double.isNaN(p2.y)) p2.y = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
			
			return tmp[0];
	}



	
	
	
	
	/**
	 * Allows to parse the parameters of a frame or a ellipse command.<br>
	 * All the parameters will be in cm
	 * @throws InvalidFormatCommandException If the format of the command is invalid.
	 */
	public int parseParametersCircle(int line, String params, Point2D.Double center, double[] radius, 
			PSTricksParameters p) throws InvalidFormatCommandException
	{
		int i=0, j;
		while(i<params.length() && params.charAt(i)!='[' && params.charAt(i)!='(' && params.charAt(i)!='{')
		{
			if(params.charAt(i)=='\\') throw new InvalidFormatCommandException(line);
			i++;
		}
		
		if(i>=params.length()) throw new InvalidFormatCommandException(line);
		j=i;
		
		// We parse the options [..]
		if(params.charAt(i)=='[') // Analyse of the options
		{
			while(j<params.length() && params.charAt(j)!=']') j++;
			
			if(j>=params.length()) throw new InvalidFormatCommandException(line);
			
			parseOptionsFigure(params.substring(i+1, j), p, !p.psCustomP.fromPsCustom);
			j++;
		}
		
		if(center==null || p.origin==null) return j;
		
		int[] tmp = {j}, temp = {0};
		try
		{//	The centre is optional
			goToNextChar(params, tmp);
			if(params.charAt(tmp[0])!='(') throw new InvalidFormatCommandException(line);
			j = params.indexOf(')', tmp[0]);
			if(j==-1) throw new InvalidFormatCommandException(line);

			center.setLocation(readOneCoordinate(params.substring(tmp[0], j+1), 0, line, true)); 
			tmp[0] = j+1;
			
		}catch(Exception e){ center.setLocation((Point2D.Double)p.origin.clone()); }
		
		if(Double.isNaN(center.x)) center.x = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
		if(Double.isNaN(center.y)) center.y = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
		
		try
		{
			// We parse the radius {Real}
			goToNextChar(params, tmp);
			if(params.charAt(tmp[0])!='{') throw new InvalidFormatCommandException(line);
			j = params.indexOf('}', tmp[0]);	
			if(j==-1) throw new InvalidFormatCommandException(line);
			
			radius[0] = readReal(params.substring(tmp[0]+1, j), temp, true);
		}catch(Exception e){throw new InvalidFormatCommandException(line);}
		
		return j+1;
	}
	

	
	
	

	/**
	 * Allows to parse the parameters of a line command.<br>
	 * All the parameters will be in cm
	 * @throws InvalidFormatCommandException If the format of the command is invalid
	 */
	public int parseParametersLine(int line, String params, Vector<Point2D.Double> pts, 
			PSTricksParameters p) throws InvalidFormatCommandException
	{
		int i=0, j;
		while(i<params.length() && params.charAt(i)!='[' && params.charAt(i)!='(' && params.charAt(i)!='{')
		{
			if(params.charAt(i)=='\\') throw new InvalidFormatCommandException(line);
			i++;
		}
		
		if(i>=params.length()) throw new InvalidFormatCommandException(line);
		j=i;
		
		// We parse the options [..]
		if(params.charAt(i)=='[') // Analyse of the options
		{
			while(j<params.length() && params.charAt(j)!=']') j++;
			
			if(j>=params.length()) throw new InvalidFormatCommandException(line);

			parseOptionsFigure(params.substring(i+1, j), p, !p.psCustomP.fromPsCustom);
			j++;
		}
		
		if(pts==null || p.origin==null) return ERR_END_CMD_INDEX;
		
		int[] tmp = {j};
		try
		{
			goToNextChar(params, tmp);
			
			if(params.charAt(tmp[0])=='{')// We read the arrows
			{// {arrow1-arrow2}
				j = params.indexOf("}", tmp[0]);//$NON-NLS-1$
				if(j==-1) throw new InvalidFormatCommandException(line);
				
				String arr = params.substring(tmp[0]+1, j);
				tmp[0] = j+1;
				
				if(arr.length()>0)
				{
					int k = arr.indexOf("-");//$NON-NLS-1$
					
					if(k==-1)// One arrow
					{
						if(PSTricksConstants.isValidArrowStyle(arr))
							p.arrowStyle[0] = ArrowHead.invertArrowStyle(arr);
					}
					else
					{
						if(k!=0)
							if(PSTricksConstants.isValidArrowStyle(arr.substring(0, k)))
								p.arrowStyle[0] = arr.substring(0, k);
						
						if(k!=arr.length())
							if(PSTricksConstants.isValidArrowStyle(arr.substring(k+1)))
								p.arrowStyle[1] = arr.substring(k+1);
					}
				}
			}
			
			if(params.charAt(tmp[0])!='(') throw new InvalidFormatCommandException(line);
			j = params.indexOf(')', tmp[0]);
			if(j==-1) throw new InvalidFormatCommandException(line);
			
			//	The first position must be here
			Point2D.Double pt = readOneCoordinate(params.substring(tmp[0], j+1), 0, line, true); 
			if(pt==null) throw new InvalidFormatCommandException(line);
			if(Double.isNaN(pt.x)) pt.x = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
			if(Double.isNaN(pt.y)) pt.y = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
			pt.setLocation(pt.x+p.origin.x, pt.y+p.origin.y);
			pts.addElement(pt);
			tmp[0] = j+1;

		}catch(IgnoreCommandException e1) { throw new InvalidFormatCommandException(line); }
		
		boolean stop=false;
		while(!stop)
		{
			try
			{
				goToNextChar(params, tmp);
				if(params.charAt(tmp[0])!='(') throw new InvalidFormatCommandException(line);
				j = params.indexOf(')', tmp[0]);
				if(j==-1) throw new InvalidFormatCommandException(line);
				
				Point2D.Double pt = readOneCoordinate(params.substring(tmp[0], j+1), 0, line, true); 
				if(pt==null) throw new InvalidFormatCommandException(line);
				if(Double.isNaN(pt.x)) pt.x = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
				if(Double.isNaN(pt.y)) pt.y = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
				pt.setLocation(pt.x+p.origin.x, pt.y+p.origin.y);
				pts.addElement(pt);
				tmp[0] = j+1;
				
			}catch(Exception e){ stop = true; }
		} // while(!stop)
		
		if(pts.size()==1)
			pts.add(0,new Point2D.Double(p.origin.x, p.origin.y));
		
		return tmp[0];
	}

	
	
	

	/**
	 * Allows to parse the parameters of a polygon command.
	 * @param line The line of the command.
	 * @param params The parameters.
	 * @param pts The futures points of the polygon.
	 * @param p The current generals parameters.
	 * @return The index of the end of the parameters in params.
	 * @throws InvalidFormatCommandException If the format of the command is invalid.
	 */
	public int parseParametersPolygon(int line, String params, Vector<Point2D.Double> pts, 
			PSTricksParameters p) throws InvalidFormatCommandException
	{
		int i=0, j;
		while(i<params.length() && params.charAt(i)!='[' && params.charAt(i)!='(' && params.charAt(i)!='{')
		{
			if(params.charAt(i)=='\\') throw new InvalidFormatCommandException(line);
			i++;
		}
		
		if(i>=params.length()) throw new InvalidFormatCommandException(line);
		j=i;
		
		// We parse the options [..]
		if(params.charAt(i)=='[') // Analyse of the options
		{
			while(j<params.length() && params.charAt(j)!=']') j++;
			
			if(j>=params.length()) throw new InvalidFormatCommandException(line);
			
			parseOptionsFigure(params.substring(i+1, j), p, !p.psCustomP.fromPsCustom);
			j++;
		}
		
		if(pts==null || p.origin==null) return j;
		
		boolean stop=false;
		int[] tmp = {j};
		while(!stop)
		{
			try
			{
				goToNextChar(params, tmp);
				if(params.charAt(tmp[0])!='(') throw new InvalidFormatCommandException(line);
				j = params.indexOf(')', tmp[0]);
				if(j==-1) throw new InvalidFormatCommandException(line);
				
				Point2D.Double pt = readOneCoordinate(params.substring(tmp[0], j+1), 0, line, true); 
				if(pt==null) throw new InvalidFormatCommandException(line);
				if(Double.isNaN(pt.x)) pt.x = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
				if(Double.isNaN(pt.y)) pt.y = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
				pt.setLocation(pt.x+p.origin.x, pt.y+p.origin.y);
				pts.addElement(pt);
				tmp[0] = j+1;
				
			}catch(Exception e){ stop = true; }
		} // while(!stop)
		
		if(pts.isEmpty())
			throw new InvalidFormatCommandException(line);
		
		return tmp[0];
	}


	
	


	/**
	 * Allows to parse the parameters of a Bézier curve command.
	 * @param line The line of the command.
	 * @param params The parameters.
	 * @param pts The futures points of the Bézier curve.
	 * @param p The current generals parameters.
	 * @return The index of the end of the parameters in params.
	 * @throws InvalidFormatCommandException If the format of the command is invalid.
	 */
	public int parseParametersBezierCurve(int line, String params, Vector<Point2D.Double> pts, 
			PSTricksParameters p) throws InvalidFormatCommandException
	{
		int i=0, j;
		while(i<params.length() && params.charAt(i)!='[' && params.charAt(i)!='(')
		{
			if(params.charAt(i)=='\\') throw new InvalidFormatCommandException(line);
			i++;
		}
		
		if(i>=params.length()) throw new InvalidFormatCommandException(line);
		j=i;
		
		// We parse the options [..]
		if(params.charAt(i)=='[') // Analyse of the options
		{
			while(j<params.length() && params.charAt(j)!=']') j++;
			
			if(j>=params.length()) throw new InvalidFormatCommandException(line);
			
			parseOptionsFigure(params.substring(i+1, j), p, !p.psCustomP.fromPsCustom);
			j++;
		}
		
		if(pts==null) return ERR_END_CMD_INDEX;
	
		int[] tmp = {j};
		try
		{
			goToNextChar(params, tmp);
			
			if(params.charAt(tmp[0])=='{')// We read the arrows
			{// {arrow1-arrow2}
				j = params.indexOf("}", tmp[0]);//$NON-NLS-1$
				if(j==-1) throw new InvalidFormatCommandException(line);
				
				String arr = params.substring(tmp[0]+1, j);
				tmp[0] = j+1;
				
				if(arr.length()>0)
				{
					int k = arr.indexOf("-");//$NON-NLS-1$
					
					if(k==-1)// One arrow
					{
						if(PSTricksConstants.isValidArrowStyle(arr))
							p.arrowStyle[0] = ArrowHead.invertArrowStyle(arr);
					}
					else
					{
						if(k!=0)
							if(PSTricksConstants.isValidArrowStyle(arr.substring(0, k)))
								p.arrowStyle[0] = arr.substring(0, k);
						
						if(k!=arr.length())
							if(PSTricksConstants.isValidArrowStyle(arr.substring(k+1)))
								p.arrowStyle[1] = arr.substring(k+1);
					}
				}
			}
			
			if(params.charAt(tmp[0])!='(') throw new InvalidFormatCommandException(line);
			j = params.indexOf(')', tmp[0]);
			if(j==-1) throw new InvalidFormatCommandException(line);
			
			//	The first position must be here
			Point2D.Double pt = readOneCoordinate(params.substring(tmp[0], j+1), 0, line, true); 
			if(pt==null) throw new InvalidFormatCommandException(line);
			if(Double.isNaN(pt.x)) pt.x = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
			if(Double.isNaN(pt.y)) pt.y = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
			pt.setLocation(pt.x+p.origin.x, pt.y+p.origin.y);
			pts.addElement(pt);
			tmp[0] = j+1;

		}catch(IgnoreCommandException e1) { throw new InvalidFormatCommandException(line); }
		
		boolean stop=false;
		
		while(!stop)
		{
			try
			{
				goToNextChar(params, tmp);
				if(params.charAt(tmp[0])!='(') throw new InvalidFormatCommandException(line);
				j = params.indexOf(')', tmp[0]);
				if(j==-1) throw new InvalidFormatCommandException(line);
				
				Point2D.Double pt = readOneCoordinate(params.substring(tmp[0], j+1), 0, line, true); 
				if(pt==null) throw new InvalidFormatCommandException(line);
				if(Double.isNaN(pt.x)) pt.x = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
				if(Double.isNaN(pt.y)) pt.y = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
				pt.setLocation(pt.x+p.origin.x, pt.y+p.origin.y);
				pts.addElement(pt);
				tmp[0] = j+1;
			}catch(Exception e){ stop = true; }
		} // while(!stop)
		
		return tmp[0];
	}

	
	
	
	

	/**
	 * Allows to parse the parameters of a Bézier curve command.
	 * @param line The line of the command.
	 * @param params The parameters.
	 * @param radius The futures radius of the arc.
	 * @param p The current generals parameters.
	 * @param center The future centre of the arc.
	 * @param angle1 The future first angle of the arc.
	 * @param angle2 The future last angle of the arc.
	 * @return The index of the end of the parameters in params.
	 * @throws InvalidFormatCommandException If the format of the command is invalid.
	 */
	public int parseParametersArcWedge(int line, String params, double[] radius, Point2D.Double center, 
			double[] angle1, double[] angle2, PSTricksParameters p, boolean isArc) throws InvalidFormatCommandException
		{
			int i=0, j;
			while(i<params.length() && params.charAt(i)!='[' && params.charAt(i)!='('  && params.charAt(i)!='{')
			{
				if(params.charAt(i)=='\\') throw new InvalidFormatCommandException(line);
				i++;
			}
				
			
			if(i>=params.length()) throw new InvalidFormatCommandException(line);
			j=i;
			
			// We parse the options [..]
			if(params.charAt(i)=='[') // Analyse of the options
			{
				while(j<params.length() && params.charAt(j)!=']') j++;
				
				if(j>=params.length()) throw new InvalidFormatCommandException(line);
				
				parseOptionsFigure(params.substring(i+1, j), p, !p.psCustomP.fromPsCustom);
				j++;
			}
			
			if(center==null || p.origin==null) return ERR_END_CMD_INDEX;
		
			int[] tmp = {j}, temp = {0};
			try
			{
				goToNextChar(params, tmp);
				
				if(isArc)
					if(params.charAt(tmp[0])=='{')// We read the arrows
					{// {arrow1-arrow2}
						j = params.indexOf("}", tmp[0]);//$NON-NLS-1$
						if(j==-1) throw new InvalidFormatCommandException(line);
						
						String arr = params.substring(tmp[0]+1, j);
						tmp[0] = j+1;
						
						if(arr.length()>0)
						{
							int k = arr.indexOf("-");//$NON-NLS-1$
							
							if(k==-1)// One arrow
							{
								if(PSTricksConstants.isValidArrowStyle(arr))
									p.arrowStyle[0] = ArrowHead.invertArrowStyle(arr);
							}
							else
							{
								if(k!=0)
									if(PSTricksConstants.isValidArrowStyle(arr.substring(0, k)))
										p.arrowStyle[0] = arr.substring(0, k);
								
								if(k!=arr.length())
									if(PSTricksConstants.isValidArrowStyle(arr.substring(k+1)))
										p.arrowStyle[1] = arr.substring(k+1);
							}
						}
					}
				
				try
				{// The centre is optional
					goToNextChar(params, tmp);
					if(params.charAt(tmp[0])!='(') throw new InvalidFormatCommandException(line);
					j = params.indexOf(')', tmp[0]);
					if(j==-1) throw new InvalidFormatCommandException(line);
					
					Point2D.Double pt = readOneCoordinate(params.substring(tmp[0], j+1), 0, line, true); 
					if(pt==null) throw new InvalidFormatCommandException(line);
					pt.setLocation(pt.x+p.origin.x, pt.y+p.origin.y);
					center.setLocation(pt);
					tmp[0] = j+1;
					
				}catch(Exception e){ center.setLocation((Point2D.Double)p.origin.clone()); }

				// We parse the radius {Real}
				goToNextChar(params, tmp);
				if(params.charAt(tmp[0])!='{') throw new InvalidFormatCommandException(line);
				j = params.indexOf('}', tmp[0]);	
				if(j==-1) throw new InvalidFormatCommandException(line);
				
				radius[0] = readRealUnit(temp, params.substring(tmp[0]+1, j), true);
				
				tmp[0] = j+1;
//					 We parse the first angle {Real}
				goToNextChar(params, tmp);
				if(params.charAt(tmp[0])!='{') throw new InvalidFormatCommandException(line);
				j = params.indexOf('}', tmp[0]);	
				if(j==-1) throw new InvalidFormatCommandException(line);
				temp[0] = 0;
				angle1[0] = readReal(params.substring(tmp[0]+1, j), temp, true);
				
				tmp[0] = j+1;
//					 We parse the second angle radius {Real}
				goToNextChar(params, tmp);
				if(params.charAt(tmp[0])!='{') throw new InvalidFormatCommandException(line);
				j = params.indexOf('}', tmp[0]);	
				if(j==-1) throw new InvalidFormatCommandException(line);
				temp[0] = 0;
				angle2[0] = readReal(params.substring(tmp[0]+1, j), temp, true);
				
				return j+1;
				
			}catch(IgnoreCommandException e1) { throw new InvalidFormatCommandException(line); }
		}

	
	
	

	/**
	 * Allows to parse the parameters of a Bézier curve command.
	 * @param line The line of the command.
	 * @param params The parameters.
	 * @param p The current generals parameters.
	 * @param pts The future points of the grids.
	 * @return The index of the end of the parameters in params.
	 * @throws InvalidFormatCommandException If the format of the command is invalid.
	 */
	public int parseParametersGridAxes(int line, String params, Vector<Point2D.Double> pts, 
			PSTricksParameters p) throws InvalidFormatCommandException
		{
			int i=0, j;
			int[] k={0};
			
			try { goToNextChar(params, k); } 
			catch(IgnoreCommandException e1) { throw new InvalidFormatCommandException(line); }
			
			if(params.charAt(k[0])!='[' && params.charAt(k[0])!='(' && params.charAt(k[0])!='{')
				return k[0];

			j=k[0];
			
			// We parse the options [..]
			if(params.charAt(i)=='[') // Analyse of the options
			{
				while(j<params.length() && params.charAt(j)!=']') j++;
				
				if(j>=params.length()) throw new InvalidFormatCommandException(line);
				
				parseOptionsFigure(params.substring(i+1, j), p, !p.psCustomP.fromPsCustom);
				j++;
			}
			
			if(pts==null) return ERR_END_CMD_INDEX;
		
			int[] tmp = {j};
			Point2D.Double p1, p2, p3;

			try
			{// These parameters are optional
				goToNextChar(params, tmp);
				
				if(params.charAt(tmp[0])=='{')// We read the arrows
				{// {arrow1-arrow2}
					j = params.indexOf("}", tmp[0]);//$NON-NLS-1$
					if(j==-1) throw new InvalidFormatCommandException(line);
					
					String arr = params.substring(tmp[0]+1, j);
					tmp[0] = j+1;
					
					if(arr.length()>0)
					{
						int m = arr.indexOf("-");//$NON-NLS-1$
						
						if(m==-1)// One arrow
						{
							if(PSTricksConstants.isValidArrowStyle(arr))
								p.arrowStyle[0] = ArrowHead.invertArrowStyle(arr);
						}
						else
						{
							if(m!=0)
								if(PSTricksConstants.isValidArrowStyle(arr.substring(0, m)))
									p.arrowStyle[0] = arr.substring(0, m);
							
							if(m!=arr.length())
								if(PSTricksConstants.isValidArrowStyle(arr.substring(m+1)))
									p.arrowStyle[1] = arr.substring(m+1);
						}
					}
				}
				
				goToNextChar(params, tmp);
				if(params.charAt(tmp[0])!='(') throw new InvalidFormatCommandException(line);
				j = params.indexOf(')', tmp[0]);
				if(j==-1) throw new InvalidFormatCommandException(line);
				
				p1 = readOneCoordinate(params.substring(tmp[0], j+1), 0, line, true); 
				if(p1==null) throw new InvalidFormatCommandException(line);
				if(Double.isNaN(p1.x)) p1.x = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
				if(Double.isNaN(p1.y)) p1.y = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
				pts.add(p1);
				tmp[0] = j+1;
				
			}catch(Exception e) { return tmp[0]; }

			try
			{// These parameters are optional
				goToNextChar(params, tmp);
				if(params.charAt(tmp[0])!='(') throw new InvalidFormatCommandException(line);
				j = params.indexOf(')', tmp[0]);
				if(j==-1) throw new InvalidFormatCommandException(line);
				
				p2 = readOneCoordinate(params.substring(tmp[0], j+1), 0, line, true); 
				if(p2==null) throw new InvalidFormatCommandException(line);
				if(Double.isNaN(p2.x)) p2.x = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
				if(Double.isNaN(p2.y)) p2.y = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
				pts.add(p2);
				tmp[0] = j+1;
				
			}catch(Exception e) { return tmp[0]; }
			
			try
			{// These parameters are optional
				goToNextChar(params, tmp);
				if(params.charAt(tmp[0])!='(') throw new InvalidFormatCommandException(line);
				j = params.indexOf(')', tmp[0]);
				if(j==-1) throw new InvalidFormatCommandException(line);
		
				p3 = readOneCoordinate(params.substring(tmp[0], j+1), 0, line, true); 
				if(p3==null) throw new InvalidFormatCommandException(line);
				if(Double.isNaN(p3.x)) p3.x = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
				if(Double.isNaN(p3.y)) p3.y = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
				pts.add(p3);
				tmp[0] = j+1;
				
				return tmp[0];
				
			}catch(Exception e) { return tmp[0]; }
		}

	
	
	
	
	/**
	 * Allows to parse the options of any PSTricks figure (excepts psplot).<br>
	 * All the parameters will be in cm.
	 * @param params The code to parse.
	 * @param pstp The pstricks parameters.
	 * @param applyParams If <code>true</code> the parsed parameters will be applied to the <code>pstp</code>, 
	 * else the parameters will be parsed but ignored.
	 */
	public static void parseOptionsFigure(String params, PSTricksParameters pstp, boolean applyParams)
	{
		int size = params.length(); 
		int id[] = {0};
		PSTricksParameters p = applyParams ? pstp : new PSTricksParameters(pstp);
	
		while(id[0]<params.length())
		{
			try
			{
				switch(params.charAt(id[0]))
				{
					case 's' : 
						if(++id[0]>size) throw new IgnoreCommandException(-1);
						switch(params.charAt(id[0]))
						{
							case 'h' : //sh
								if(++id[0]>size) throw new IgnoreCommandException(-1);
								switch(params.charAt(id[0]))
								{
									case 'a' : // sha
										if(id[0]+3>size) throw new IgnoreCommandException(-1);
										
										if(params.charAt(++id[0])=='d' && params.charAt(++id[0])=='o' && 
											params.charAt(++id[0])=='w') // shadow
										{
											if(++id[0]>size) throw new IgnoreCommandException(-1);
											switch(params.charAt(id[0]))
											{
												case 's' : // shadowsize = Real unit
													if(id[0]+4>size) throw new IgnoreCommandException(-1);
													
													if(params.charAt(++id[0])=='i' && params.charAt(++id[0])=='z' && 
														params.charAt(++id[0])=='e' && 
														(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
													{
														p.shadowSize = readEqualRealUnit(id, params);
														if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
														
													}else throw new IgnoreCommandException(-1);
													break;
													
												case 'a' : // shadowangle = Real
													if(id[0]+5>size) throw new IgnoreCommandException(-1);
													
													if(params.charAt(++id[0])=='n' && params.charAt(++id[0])=='g' && 
														params.charAt(++id[0])=='l' && params.charAt(++id[0])=='e' && 
														(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
													{
														p.shadowAngle = readEqualReal(id, params);
														goToNextChar(params, id); // We go to the next token : ','
														
													}else throw new IgnoreCommandException(-1);
													break;
													
												case 'c' : // shadowcolor = String
													if(id[0]+5>size) throw new IgnoreCommandException(-1);
													
													if(params.charAt(++id[0])=='o' && params.charAt(++id[0])=='l' && 
														params.charAt(++id[0])=='o' && params.charAt(++id[0])=='r' && 
														(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
													{
														String s2 = readEqualString(id, params, true);
														Color c = DviPsColors.getColour(s2); // We search the colour
														if(c!=null) p.shadowCol = c;
														if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
														
													}else throw new IgnoreCommandException(-1);
													break;
												
												default : 
													if((params.charAt(id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
													{// shadow
														String s = readEqualString(id, params, false);
														if(s.equals("true")) 		 p.isShadow = true;//$NON-NLS-1$
														else if(s.equals("false")) p.isShadow = false;//$NON-NLS-1$
															
														if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
													}
													else
														throw new IgnoreCommandException(-1);
											}
										}else throw new IgnoreCommandException(-1);
										break;
										
									case 'o' : // sho
										if(id[0]+1>size) throw new IgnoreCommandException(-1);
										
										if(params.charAt(++id[0])=='w')//show
										{
											if(id[0]+1>size) throw new IgnoreCommandException(-1);
											
											switch(params.charAt(++id[0]))
											{
												case 'p'://showp
													if(id[0]+6>size) throw new IgnoreCommandException(-1);
													if(params.charAt(++id[0])=='o' && params.charAt(++id[0])=='i' && 
														params.charAt(++id[0])=='n' && params.charAt(++id[0])=='t' && 
														params.charAt(++id[0])=='s' && 
														(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
													{//showpoints = true|false
														String s = readEqualString(id, params, false);
														if(s.equals("true")) 		 p.showPoints = true;//$NON-NLS-1$
														else if(s.equals("false")) p.showPoints = false;//$NON-NLS-1$
															
														if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
													}
													else throw new IgnoreCommandException(-1);
													break;
													
												case 'o'://showo
													if(id[0]+6>size) throw new IgnoreCommandException(-1);
													if(params.charAt(++id[0])=='r' && params.charAt(++id[0])=='i' && 
														params.charAt(++id[0])=='g' && params.charAt(++id[0])=='i' && 
														params.charAt(++id[0])=='n' && 
														(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
													{//showorigin = true|false
														String s = readEqualString(id, params, false);
														if(s.equals("true")) 		 p.showOrigin = true;//$NON-NLS-1$
														else if(s.equals("false")) p.showOrigin = false;//$NON-NLS-1$
															
														if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
													}
													else throw new IgnoreCommandException(-1);
													break;
													
												default: throw new IgnoreCommandException(-1);
											}
											
										}else throw new IgnoreCommandException(-1);
										break;
										
									default : throw new IgnoreCommandException(-1);
								}
								break;
								
							case 'u' : // su
								if(id[0]+5>size) throw new IgnoreCommandException(-1);
								
								if(params.charAt(++id[0])=='b' && params.charAt(++id[0])=='g' && 
									params.charAt(++id[0])=='r' && params.charAt(++id[0])=='i' && 
									params.charAt(++id[0])=='d') // subgrid
								{
									if(++id[0]>size) throw new IgnoreCommandException(-1);
									switch(params.charAt(id[0]))
									{
										case 'd' : // subgridd
											if(++id[0]>size) throw new IgnoreCommandException(-1);
											switch(params.charAt(id[0]))
											{
												case 'o' : // subgriddots = Integer
													if(id[0]+3>size) throw new IgnoreCommandException(-1);
													
													if(params.charAt(++id[0])=='t' && params.charAt(++id[0])=='s' && 
														(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
													{
														p.subGridDots = readEqualInteger(id, params);
														if(id[0]<size) goToNextChar(params, id);
														
													}else throw new IgnoreCommandException(-1);
													break;
													
												case 'i' : // subgriddiv = Integer
													if(id[0]+2>size) throw new IgnoreCommandException(-1);
													
													if(params.charAt(++id[0])=='v' && 
														(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
													{
														p.subGridDiv = readEqualInteger(id, params);
														if(id[0]<size) goToNextChar(params, id);
														
													}else throw new IgnoreCommandException(-1);
													break;
												
												default : throw new IgnoreCommandException(-1);
											}
											break;
											
										case 'w' : // subgridwidth = Real unit
											if(id[0]+5>size) throw new IgnoreCommandException(-1);
											
											if(params.charAt(++id[0])=='i' && params.charAt(++id[0])=='d' && 
												params.charAt(++id[0])=='t' && params.charAt(++id[0])=='h' && 
												(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
											{
												p.subGridWidth = readEqualRealUnit(id, params);
												if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
												
											}else throw new IgnoreCommandException(-1);
											break;
											
										case 'c': // subgridcolor = String
											if(id[0]+5>size) throw new IgnoreCommandException(-1);
											
											if(params.charAt(++id[0])=='o' && params.charAt(++id[0])=='l' && 
												params.charAt(++id[0])=='o' && params.charAt(++id[0])=='r' && 
												(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
											{
												String s2 = readEqualString(id, params, true);
												Color c = DviPsColors.getColour(s2); // We search the colour
												if(c!=null)	p.subGridCol = c;
												if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
												
											}else throw new IgnoreCommandException(-1);
											break;
											
										default : throw new IgnoreCommandException(-1);
									}
								}else throw new IgnoreCommandException(-1);
								break;
								
							case 'w' : // swapaxes = true|false
								if(id[0]+7>size) throw new IgnoreCommandException(-1);
								
								if(params.charAt(++id[0])=='a' && params.charAt(++id[0])=='p' && 
									params.charAt(++id[0])=='a' && params.charAt(++id[0])=='x' && 
									params.charAt(++id[0])=='e' && params.charAt(++id[0])=='s' && 
									(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
								{
									String s = readEqualString(id, params, false);
									if(s.equals("true")) 		 p.swapAxes = true;//$NON-NLS-1$
									else if(s.equals("false")) p.swapAxes = false;//$NON-NLS-1$
										
									if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
									
								}else throw new IgnoreCommandException(-1);
								break;
						
							default : throw new IgnoreCommandException(-1);
						}
						break;
						
					case 'd' : 
						if(++id[0]>size) throw new IgnoreCommandException(-1);
						switch(params.charAt(id[0]))
						{
							case 'o' : // do
								if(++id[0]>size) throw new IgnoreCommandException(-1);
								switch(params.charAt(id[0]))
								{
									case 't' : // dot
										if(++id[0]>size) throw new IgnoreCommandException(-1);
										switch(params.charAt(id[0]))
										{
											case 's' : // dots
												if(++id[0]>size) throw new IgnoreCommandException(-1);
												switch(params.charAt(id[0]))
												{
													case 'i' : //dotsize = Real unit Real
														if(id[0]+3>size) throw new IgnoreCommandException(-1);
														
														if(params.charAt(++id[0])=='z' && params.charAt(++id[0])=='e' && 
															(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
														{
															p.arrowDotSD = readEqualRealUnit(id, params);
															if(id[0]<size) goToNextChar(params, id); // We go to the next token

															if(id[0]<size && (Character.isDigit(
																			params.charAt(id[0])) || params.charAt(id[0])=='.')) 
																p.arrowDotSN = readReal(params, id, true);// If there is the x, we read it
															else p.arrowDotSN = 0;

															if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
															
														}else throw new IgnoreCommandException(-1);
														break;
														
													case 't' : // dotstyle = String
														if(id[0]+4>size) throw new IgnoreCommandException(-1);
														
														if(params.charAt(++id[0])=='y' && params.charAt(++id[0])=='l' && 
															params.charAt(++id[0])=='e' && 
															(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
														{
															while(id[0]<size && params.charAt(id[0])!='=') id[0]++; // We search the '='
															if(id[0]>=size) throw new IgnoreCommandException(-1);
															id[0]++;
															
															goToNextChar(params, id); // We go to the next token
															
															String s;
															
															if(params.charAt(id[0])=='|' || params.charAt(id[0])=='+')
																 s = String.valueOf(params.charAt(id[0]++));
															else
															{
																s = readString(params, id, false);
																if(id[0]<size && params.charAt(id[0])=='*')
																	s+=String.valueOf(params.charAt(id[0]++));
															}
															
															if(!PSTricksConstants.isValidDotStyle(s))
																throw new IgnoreCommandException(-1);
																
															p.dotStyle = s;
															if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','

														}else throw new IgnoreCommandException(-1);
														break;
														
													case 'c' : // dotscale = Real Real
														if(id[0]+4>size) throw new IgnoreCommandException(-1);
														
														if(params.charAt(++id[0])=='a' && params.charAt(++id[0])=='l' && 
															params.charAt(++id[0])=='e' && 
															(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
														{
															p.dotScale1 = readEqualReal(id, params); // We read the first parameter
															
															goToNextChar(params, id); // We go to the next token
															if(id[0]<size && (Character.isDigit(
																			params.charAt(id[0])) || params.charAt(id[0])=='.')) 
																p.dotScale2 = readReal(params, id, true);// If there is the x, we read it
															else p.dotScale2 = 1;
															
															if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
															
														}else throw new IgnoreCommandException(-1);
														break;
														
													case 'e' : // dotsep = Real unit
														if(id[0]+2>size) throw new IgnoreCommandException(-1);

														if(params.charAt(++id[0])=='p' && 
															(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
														{
															p.dotStep = readEqualRealUnit(id, params);
															if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
														} else throw new IgnoreCommandException(-1);
														break;
												
													default : throw new IgnoreCommandException(-1);
												}
												break;
												
											case 'a' : // dotangle = Real
												
												if(id[0]+5>size) throw new IgnoreCommandException(-1);

												if(params.charAt(++id[0])=='n' && params.charAt(++id[0])=='g' &&
													params.charAt(++id[0])=='l' && params.charAt(++id[0])=='e' &&
													(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
												{
													p.dotAngle = readEqualReal(id, params);
											
													if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
												}else throw new IgnoreCommandException(-1);
												break;
										
											default : throw new IgnoreCommandException(-1);
										}
										break;
										
									case 'u' : // dou
										if(id[0]+3>size) throw new IgnoreCommandException(-1);
										
										if(params.charAt(++id[0])=='b' && params.charAt(++id[0])=='l' && 
											params.charAt(++id[0])=='e') // double
										{
											if(++id[0]>size) throw new IgnoreCommandException(-1);
											switch(params.charAt(id[0]))
											{
												case 'l' : // doubleline = true/false
													if(id[0]+4>size) throw new IgnoreCommandException(-1);
													
													if(params.charAt(++id[0])=='i' && params.charAt(++id[0])=='n' &&
														params.charAt(++id[0])=='e' && 
														(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
													{
														String s = readEqualString(id, params, false);
														if(s.equals("true")) 		 p.dbleLine = true;//$NON-NLS-1$
														else if(s.equals("false")) p.dbleLine = false;//$NON-NLS-1$
														
														if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
														
													}else throw new IgnoreCommandException(-1);
													break;
													
												case 's' : // doublesep = Real unit
													if(id[0]+3>size) throw new IgnoreCommandException(-1);
													
													if(params.charAt(++id[0])=='e' && params.charAt(++id[0])=='p' &&
														(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
													{
														p.dbleSep = readEqualRealUnit(id, params);
														if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
														
													}else throw new IgnoreCommandException(-1);
													break;
													
												case 'c' : // doubleColor = String
													if(id[0]+5>size) throw new IgnoreCommandException(-1);
													
													if(params.charAt(++id[0])=='o' && params.charAt(++id[0])=='l' &&
														params.charAt(++id[0])=='o' && params.charAt(++id[0])=='r' &&
														(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
													{
														String s2 = readEqualString(id, params, true);
														Color c = DviPsColors.getColour(s2); // We search the colour
														if(c!=null) p.dbleColor = c;
														if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
														
													}else throw new IgnoreCommandException(-1);
													break;
												
												default : throw new IgnoreCommandException(-1);
											}
										}else throw new IgnoreCommandException(-1);
										break;
									
									default : throw new IgnoreCommandException(-1);
								}
								break;
								
							case 'i' : // dimen = outer|inner|middle
								if(id[0]+4>size) throw new IgnoreCommandException(-1);
								
								if(params.charAt(++id[0])=='m' && params.charAt(++id[0])=='e' && 
									params.charAt(++id[0])=='n' && 
									(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
								{
									String s = readEqualString(id, params, false);
									
									if(s.equals(PSTricksConstants.BORDERS_INSIDE) ||
										s.equals(PSTricksConstants.BORDERS_MIDDLE) ||
										s.equals(PSTricksConstants.BORDERS_OUTSIDE))
									{
										p.borderPos = s;
										if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
										
									}else throw new IgnoreCommandException(-1);
								}else throw new IgnoreCommandException(-1);
								break;
								
							case 'a' : // dash = Real unit Real unit
								if(id[0]+3>size) throw new IgnoreCommandException(-1);
								
								if(params.charAt(++id[0])=='s' && params.charAt(++id[0])=='h' && 
									(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
								{
									
									double r1 = readEqualRealUnit(id, params);
									double r2 = readRealUnit(id, params, true); // We read the second parameter
									
									p.dashBlack = r1;
									p.dashWhite = r2;
									
									if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
									
								}else throw new IgnoreCommandException(-1);
								break;
								
							case 'x'://dx = real unit
								if(id[0]+1>size) throw new IgnoreCommandException(-1);
								
								if((params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
								{
									p.dxLabelDist = readEqualRealUnit(id, params);
									if(id[0]<size) goToNextChar(params, id);
								}
								else throw new IgnoreCommandException(-1);
								break;
								
							case 'y'://dy = real unit
								if(id[0]+1>size) throw new IgnoreCommandException(-1);
								
								if((params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
								{
									p.dyLabelDist = readEqualRealUnit(id, params);
									if(id[0]<size) goToNextChar(params, id);
								}
								else throw new IgnoreCommandException(-1);
								break;
						
							default : throw new IgnoreCommandException(-1);
						}
						break;
						
					case 'a' : 
						if(++id[0]>size) throw new IgnoreCommandException(-1);
						
						switch(params.charAt(id[0]))
						{
							case 'r':
								if(++id[0]>size) throw new IgnoreCommandException(-1);
								switch(params.charAt(id[0]))
								{
									case 'c' : //arc
										if(id[0]+3>size) throw new IgnoreCommandException(-1);
										
										if(params.charAt(++id[0])=='s' && params.charAt(++id[0])=='e' && 
											params.charAt(++id[0])=='p') // arcsep
										{
											if(++id[0]>size) throw new IgnoreCommandException(-1);
											
											switch(params.charAt(id[0]))
											{
												case 'A' : // arcsepA = Real unit
													p.arcSepA = readEqualRealUnit(id, params);
													if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
													break;
														
												case 'B' : // arcSepB = Real unit
													p.arcSepB = readEqualRealUnit(id, params);
													if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
													break;
											
												default : 
													if(params.charAt(id[0])=='=' || isCharToIgnore(params.charAt(id[0])))
													{ // arcsep = Real
														p.arcSep = readEqualReal(id, params);
														if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
														
													}else throw new IgnoreCommandException(-1);
											}
										}else throw new IgnoreCommandException(-1);
										
										break;
										
									case 'r' : //arr
										if(id[0]+2>size) throw new IgnoreCommandException(-1);
									
										if(params.charAt(++id[0])=='o' && params.charAt(++id[0])=='w')
										{ // arrow
											if(++id[0]>size) throw new IgnoreCommandException(-1);
											
											switch(params.charAt(id[0]))
											{
												case 's' : // arrows
													
													if(++id[0]>size) throw new IgnoreCommandException(-1);
													
													switch(params.charAt(id[0]))
													{
														case 'i' : // arrowsize= Real unit Real
												
																if(id[0]+3>size) throw new IgnoreCommandException(-1);
																
																if(params.charAt(++id[0])=='z' && params.charAt(++id[0])=='e' && 
																	(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
																{
																	p.arrowSizeD = readEqualRealUnit(id, params);
																	goToNextChar(params, id); // We go to the next token
																	if(id[0]<size && (Character.isDigit(params.charAt(id[0])) || params.charAt(id[0])=='.')) 
																		p.arrowSizeN = readReal(params, id, true);// If there is the x, we read it
																	
																	if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
																	
																}else throw new IgnoreCommandException(-1);
																break;
															
														case '=':
																id[0]++;
																goToNextChar(params, id);
																char str = params.charAt(id[0]);
																
																if(str!='-')
																{
																	p.arrowStyle[0] = String.valueOf(str);
																	id[0]++;
																	goToNextChar(params, id);
																	
																	if(params.charAt(id[0])!='-')
																		throw new IgnoreCommandException(-1);
																}
																
																id[0]++;
																goToNextChar(params, id);
																
																if(id[0]<params.length())
																{
																	str = params.charAt(id[0]);
																	
																	if(str!=',' || str!=']')
																		p.arrowStyle[1] = String.valueOf(str);
																	
																	id[0]++;
																}
																break;
																
														default: throw new IgnoreCommandException(-1);
													}
													break;
													
												case 'l' : // arrowlength = Real
													if(id[0]+6>size) throw new IgnoreCommandException(-1);
													
													if(params.charAt(++id[0])=='e' && params.charAt(++id[0])=='n' && 
														params.charAt(++id[0])=='g' && params.charAt(++id[0])=='t' && 
														params.charAt(++id[0])=='h' &&
														(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
													{
														p.arrowLgth = readEqualReal(id, params);
														if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
														
													}else throw new IgnoreCommandException(-1);
													break;
													
												case 'i' : // arrowinset = Real
													if(id[0]+5>size) throw new IgnoreCommandException(-1);
													
													if(params.charAt(++id[0])=='n' && params.charAt(++id[0])=='s' && 
														params.charAt(++id[0])=='e' && params.charAt(++id[0])=='t' && 
														(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
													{
														p.arrowInset = readEqualReal(id, params);
														if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
														
													}else throw new IgnoreCommandException(-1);
													break;
												
												default : throw new IgnoreCommandException(-1);
											} // switch(params.charAt(id[0]))
										}else throw new IgnoreCommandException(-1);
										break;
										
									default : throw new IgnoreCommandException(-1);
								} // switch(params.charAt(id[0]))
								break;
								
							case 'x': // ax
								
								if(id[0]+6>size) throw new IgnoreCommandException(-1);
								
								if(params.charAt(++id[0])=='e' && params.charAt(++id[0])=='s' && 
									params.charAt(++id[0])=='s' &&
									params.charAt(++id[0])=='t' && params.charAt(++id[0])=='y' && 
									params.charAt(++id[0])=='l' && params.charAt(++id[0])=='e' && 
									(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
								{// axestyle
									p.axesStyle = readEqualString(id, params, true);
									if(id[0]<size) goToNextChar(params, id);
								}
								else throw new IgnoreCommandException(-1);
								break;
								
							default : throw new IgnoreCommandException(-1);
						}
						break;
								
						
					case 'g' :
						
						if(++id[0]>size) throw new IgnoreCommandException(-1);
						
						switch(params.charAt(id[0]))
						{
							case 'r' : //gr
								if(++id[0]>size) throw new IgnoreCommandException(-1);
								
								switch(params.charAt(id[0]))
								{
									case 'i':
										if(id[0]+1>size) throw new IgnoreCommandException(-1);
										if(params.charAt(++id[0])=='d')// grid
										{ 
											if(++id[0]>size) throw new IgnoreCommandException(-1);
											
											switch(params.charAt(id[0]))
											{
												case 'l' ://gridl
													if(id[0]+4>size) throw new IgnoreCommandException(-1);
													
													if(params.charAt(++id[0])=='a' && params.charAt(++id[0])=='b' && 
														params.charAt(++id[0])=='e' && params.charAt(++id[0])=='l')// gridlabel
													{ 
														if(++id[0]>size) throw new IgnoreCommandException(-1);
														
														switch(params.charAt(id[0]))
														{
															case 's' : // gridlabels = Real unit
																p.gridLabel = readEqualRealUnit(id, params);
																if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
																break;
																
															case 'c' : // gridlabelcolor = string
																String s = readEqualString(id, params, true);
																Color c = DviPsColors.getColour(s); // We search the colour
																if(c!=null) p.labelsGridCol = c;
																if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
																break;
															
															default : throw new IgnoreCommandException(-1);
														}
													}else throw new IgnoreCommandException(-1);
													break;
													
												case 'w' : //gridwidth = Real unit
													if(id[0]+5>size) throw new IgnoreCommandException(-1);
													
													if(params.charAt(++id[0])=='i' && params.charAt(++id[0])=='d' && 
														params.charAt(++id[0])=='t' && params.charAt(++id[0])=='h' && 
														(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
													{
														p.gridWidth = readEqualRealUnit(id, params);
														if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
														
													}else throw new IgnoreCommandException(-1);
													break;
													
												case 'c' : // gridcolor = string
													if(id[0]+5>size) throw new IgnoreCommandException(-1);
													
													if(params.charAt(++id[0])=='o' && params.charAt(++id[0])=='l' && 
														params.charAt(++id[0])=='o' && params.charAt(++id[0])=='r' && 
														(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
													{
														String s = readEqualString(id, params, true);
														Color c = DviPsColors.getColour(s); // We search the colour
														if(c!=null)	p.gridColor = c;
														if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
														
													}else throw new IgnoreCommandException(-1);
													break;
													
												case 'd' : // griddots = Integer
													if(id[0]+4>size) throw new IgnoreCommandException(-1);
													
													if(params.charAt(++id[0])=='o' && params.charAt(++id[0])=='t' && 
														params.charAt(++id[0])=='s' &&
														(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
													{
														p.gridDots = readEqualInteger(id, params);
														if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
														
													}else throw new IgnoreCommandException(-1);
													break;
													
												default : throw new IgnoreCommandException(-1);
											}
										}else throw new IgnoreCommandException(-1);
										break;
									
									case 'a'://gra
										if(++id[0]>size) throw new IgnoreCommandException(-1);
										if(params.charAt(id[0])=='d')// grad
										{ 
											if(++id[0]>size) throw new IgnoreCommandException(-1);
											switch(params.charAt(id[0]))
											{
												case 'b' : //gradbegin
													if(id[0]+5>size) throw new IgnoreCommandException(-1);
													if(params.charAt(++id[0])=='e' && params.charAt(++id[0])=='g' && 
													params.charAt(++id[0])=='i' && params.charAt(++id[0])=='n' && 
													(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
													{
														String r = readEqualString(id, params, true);
														Color c = DviPsColors.getColour(r); // We search the colour
														if(c!=null)	p.gradBegin = c;
														
														if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
														
													}else throw new IgnoreCommandException(-1);
													break;
												
												case 'e' : //gradend
													if(id[0]+3>size) throw new IgnoreCommandException(-1);
													if(params.charAt(++id[0])=='n' && params.charAt(++id[0])=='d' && 
													(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
													{
														String r = readEqualString(id, params, true);
														Color c = DviPsColors.getColour(r); // We search the colour
														if(c!=null)	p.gradEnd = c;
														
														if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
														
													}else throw new IgnoreCommandException(-1);
													break;
													
												case 'l' : //gradlines
													if(id[0]+5>size) throw new IgnoreCommandException(-1);
													if(params.charAt(++id[0])=='i' && params.charAt(++id[0])=='n' && 
														params.charAt(++id[0])=='e' && params.charAt(++id[0])=='s' && 
														(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
														{
															p.gradLines = readEqualInteger(id, params);
															if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
															
														} else throw new IgnoreCommandException(-1);
													break;
													
												case 'm' : //gradmidpoint
													if(id[0]+9>size) throw new IgnoreCommandException(-1);
													if(params.charAt(++id[0])=='i' && params.charAt(++id[0])=='d' && 
														params.charAt(++id[0])=='p' && params.charAt(++id[0])=='o' && 
														params.charAt(++id[0])=='i' && params.charAt(++id[0])=='n' && 
														params.charAt(++id[0])=='t' &&
														(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
														{
															double midPt = readEqualReal(id, params);
															if(midPt>=0 && midPt<=1)
																p.gradMidPoint = midPt;
															if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
															
														} else throw new IgnoreCommandException(-1);
													break;
													
												case 'a' : //gradangle
													if(id[0]+5>size) throw new IgnoreCommandException(-1);
													if(params.charAt(++id[0])=='n' && params.charAt(++id[0])=='g' && 
														params.charAt(++id[0])=='l' && params.charAt(++id[0])=='e' && 
														(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
														{
															p.gradAngle = readEqualReal(id, params);
															if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
															
														} else throw new IgnoreCommandException(-1);
													break;
											
												default : throw new IgnoreCommandException(-1);
											}
										}else throw new IgnoreCommandException(-1);
										
										break;
										
									default : throw new IgnoreCommandException(-1);
								}
								break;	
								
							case 'a' : //gangle
								if(id[0]+5>size) throw new IgnoreCommandException(-1);
								
								if(params.charAt(++id[0])=='n' && params.charAt(++id[0])=='g' && 
									params.charAt(++id[0])=='l' && params.charAt(++id[0])=='e' && 
									(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
									{
										p.gangle = readEqualReal(id, params);
										if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
										
									} else throw new IgnoreCommandException(-1);
								break;
									
							default : throw new IgnoreCommandException(-1);
						}
						
						
						
						break;
						
					case 'l' : 
						if(id[0]+1>size) throw new IgnoreCommandException(-1);
						
						switch(params.charAt(++id[0]))
						{
							case 'i':// li
								if(id[0]+2>size) throw new IgnoreCommandException(-1);
								if(params.charAt(++id[0])=='n' && params.charAt(++id[0])=='e')// line
								{ 
									if(++id[0]>size) throw new IgnoreCommandException(-1);
									
									switch(params.charAt(id[0]))
									{
										case 'w' : // linewidth = Real unit
											if(id[0]+5>size) throw new IgnoreCommandException(-1);
											
											if(params.charAt(++id[0])=='i' && params.charAt(++id[0])=='d' && 
												params.charAt(++id[0])=='t' && params.charAt(++id[0])=='h' &&
												(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
											{
												p.lineWidth = readEqualRealUnit(id, params);
												if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
												
											}else throw new IgnoreCommandException(-1);
											break;
											
										case 'c' : // lineColor = string
											if(id[0]+5>size) throw new IgnoreCommandException(-1);
											
											if(params.charAt(++id[0])=='o' && params.charAt(++id[0])=='l' && 
												params.charAt(++id[0])=='o' && params.charAt(++id[0])=='r' &&
												(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
											{
												String r = readEqualString(id, params, true);
												Color c = DviPsColors.getColour(r); // We search the colour
												if(c!=null)	p.lineColor = c;
												
												if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
												
											}else throw new IgnoreCommandException(-1);
											break;
											
										case 'a' : // linearc = Real unit
											if(id[0]+3>size) throw new IgnoreCommandException(-1);
											
											if(params.charAt(++id[0])=='r' && params.charAt(++id[0])=='c' && 
												(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
											{
												p.lineArc = readEqualRealUnit(id, params);
												if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
												
											}else throw new IgnoreCommandException(-1);
											break;
											
										case 's' : // linestyle = string
											if(id[0]+5>size) throw new IgnoreCommandException(-1);
											
											if(params.charAt(++id[0])=='t' && params.charAt(++id[0])=='y' && 
												params.charAt(++id[0])=='l' && params.charAt(++id[0])=='e' && 
												(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
											{
												p.lineStyle = readEqualString(id, params, false);
												if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
											}
											else throw new IgnoreCommandException(-1);
											break;
									
										default : throw new IgnoreCommandException(-1);
									}
								}else throw new IgnoreCommandException(-1);
								break;
								
							case 'a':// la
								if(id[0]+5>size) throw new IgnoreCommandException(-1);
								if(params.charAt(++id[0])=='b' && params.charAt(++id[0])=='e' &&
									params.charAt(++id[0])=='l' && params.charAt(++id[0])=='s'&& 
									(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
								{ 
									p.labels = readEqualString(id, params, false);
									if(id[0]<size) goToNextChar(params, id);
								}
								else throw new IgnoreCommandException(-1);
								break;
								
							default : throw new IgnoreCommandException(-1);
						}
						
						break;
						
					case 'f' :
						if(++id[0]>size) throw new IgnoreCommandException(-1);
						
						switch(params.charAt(id[0]))
						{
							case 'i': //fi
								if(id[0]+2>size) throw new IgnoreCommandException(-1);
								
								if(params.charAt(++id[0])=='l' && params.charAt(++id[0])=='l')//fill
								{
									if(++id[0]>size) throw new IgnoreCommandException(-1);
									if(params.charAt(id[0])=='c')//fillc
									{
										if(id[0]+5>size) throw new IgnoreCommandException(-1);
										
										if(params.charAt(++id[0])=='o' && params.charAt(++id[0])=='l' && 
											params.charAt(++id[0])=='o' && params.charAt(++id[0])=='r' && 
											(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
										{ // fillColor = string
											String r = readEqualString(id, params, true);
											Color c = DviPsColors.getColour(r); // We search the colour
											if(c!=null) p.fillColor = c;

											if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
											
										}else throw new IgnoreCommandException(-1);
									}else
									if(params.charAt(id[0])=='s')//fills
									{
										if(id[0]+5>size) throw new IgnoreCommandException(-1);
										
										if(params.charAt(++id[0])=='t' && params.charAt(++id[0])=='y' && 
											params.charAt(++id[0])=='l' && params.charAt(++id[0])=='e' && 
											(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
										{ // fillstyle = string
											String r = readEqualString(id, params, false);
											
											if(id[0]<size && params.charAt(id[0])=='*') // We add the '*' the fillstyle
											{
												r+=String.valueOf(params.charAt(id[0]));
												id[0]++;
											}
											
											p.fillStyle = r;
											if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
											
										}else throw new IgnoreCommandException(-1);
									}else throw new IgnoreCommandException(-1);
								}else throw new IgnoreCommandException(-1);
								
								break;
								
							case 'r': //fr 
								
								if(id[0]+3>size) throw new IgnoreCommandException(-1);
								
								if(params.charAt(++id[0])=='a' && params.charAt(++id[0])=='m' && params.charAt(++id[0])=='e')
								{//frame
									if(id[0]+1>size) throw new IgnoreCommandException(-1);
									
									switch(params.charAt(++id[0]))
									{
										case 'a' : // framearc
											if(id[0]+3>size) throw new IgnoreCommandException(-1);
											if(params.charAt(++id[0])=='r' && params.charAt(++id[0])=='c' && 
												(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
													{
														p.frameArc = readEqualReal(id, params);
														if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
														
													} else throw new IgnoreCommandException(-1);
											break;
											
										case 's' : // framesep
											if(id[0]+3>size) throw new IgnoreCommandException(-1);
											if(params.charAt(++id[0])=='e' && params.charAt(++id[0])=='p' && 
												(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
													{
														p.frameSep = readEqualRealUnit(id, params);
														if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
														
													} else throw new IgnoreCommandException(-1);
											break;
											
										default :
											throw new IgnoreCommandException(-1);
									}
								}else throw new IgnoreCommandException(-1);
									
								break;
								
							default : throw new IgnoreCommandException(-1);
						}
						break;
						
					case 'b' :
						if(++id[0]>size) throw new IgnoreCommandException(-1);
						
						switch(params.charAt(id[0]))
						{
							case 'o' ://bo
								if(id[0]+1>size) throw new IgnoreCommandException(-1);
								
								switch(params.charAt(++id[0]))
								{
									case 'r'://bor
										if(id[0]+3>size) throw new IgnoreCommandException(-1);
										
										if(params.charAt(++id[0])=='d' && params.charAt(++id[0])=='e' &&
											params.charAt(++id[0])=='r')//border
										{
											if(++id[0]>size) throw new IgnoreCommandException(-1);
											if(params.charAt(id[0])=='c')//borderc
											{
												if(id[0]+5>size) throw new IgnoreCommandException(-1);
												if(params.charAt(++id[0])=='o' && params.charAt(++id[0])=='l' && params.charAt(++id[0])=='o' &&
													params.charAt(++id[0])=='r' && (params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
												{ // bordercolor = string
													String r = readEqualString(id, params, true);
													Color c = DviPsColors.getColour(r); // We search the colour
													if(c!=null) p.borderColor = c;
													if(id[0]<size) goToNextChar(params, id);
													
												}else throw new IgnoreCommandException(-1);
											}else
												if(params.charAt(id[0])=='=' || isCharToIgnore(params.charAt(id[0])))
												{// border = Real unit
													p.border = readEqualRealUnit(id, params);
													if(id[0]<size) goToNextChar(params, id); 
													
												}else throw new IgnoreCommandException(-1);
										}else throw new IgnoreCommandException(-1);
										break;
										
									case 'x' ://boxsep
										if(id[0]+4>size) throw new IgnoreCommandException(-1);
										if(params.charAt(++id[0])=='s' && params.charAt(++id[0])=='e' &&
											params.charAt(++id[0])=='p' && (params.charAt(++id[0])=='=' || 
											isCharToIgnore(params.charAt(id[0]))))//boxsep
										{
											String s = readEqualString(id, params, false);
											if(s.equals("true")) 		 p.boxSep = true;//$NON-NLS-1$
											else if(s.equals("false")) p.boxSep = false;//$NON-NLS-1$
												
											if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
											
										}else throw new IgnoreCommandException(-1);
										
										break;
										
									default :
										throw new IgnoreCommandException(-1);
								}
								break;
								
								
							case 'r' :
								if(id[0]+12>size) throw new IgnoreCommandException(-1);
								
								if(params.charAt(++id[0])=='a' && params.charAt(++id[0])=='c' && params.charAt(++id[0])=='k' &&
									params.charAt(++id[0])=='e' && params.charAt(++id[0])=='t' && params.charAt(++id[0])=='l' && 
									params.charAt(++id[0])=='e' && params.charAt(++id[0])=='n' && params.charAt(++id[0])=='g' && 
									params.charAt(++id[0])=='t' && params.charAt(++id[0])=='h' && (params.charAt(++id[0])=='=' || 
									isCharToIgnore(params.charAt(id[0]))))
									{
										p.arrowBrLgth = readEqualReal(id, params);
										if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
										
									} else throw new IgnoreCommandException(-1);
								break;
								
							default : throw new IgnoreCommandException(-1);
						}
						break;
						
					case 'c' :
						if(++id[0]>size) throw new IgnoreCommandException(-1);
						
						switch(params.charAt(id[0]))
						{
							case 'o' : // cornersize = absolute/relative
								if(id[0]+9>size) throw new IgnoreCommandException(-1);
								
								if(params.charAt(++id[0])=='r' && params.charAt(++id[0])=='n' && params.charAt(++id[0])=='e' &&
									params.charAt(++id[0])=='r' && params.charAt(++id[0])=='s' && params.charAt(++id[0])=='i' &&
									params.charAt(++id[0])=='z' && params.charAt(++id[0])=='e' &&(params.charAt(++id[0])=='=' || 
									isCharToIgnore(params.charAt(id[0]))))
									{
										String r = readString(params, id, false); // We read the parameter
										r = r.toLowerCase();
										if(r.equals(PSTricksConstants.TOKEN_ABSOLUTE))
											p.isCornerRel = false;
										else if(r.equals(PSTricksConstants.TOKEN_RELATIVE))
											p.isCornerRel = true;
										
										if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
										
									} else throw new IgnoreCommandException(-1);
								break;
								
							case 'u' : // curvate = Real Real Real
								if(id[0]+6>size) throw new IgnoreCommandException(-1);
								
								if(params.charAt(++id[0])=='r' && params.charAt(++id[0])=='v' && params.charAt(++id[0])=='a' &&
									params.charAt(++id[0])=='t' && params.charAt(++id[0])=='e' && (params.charAt(++id[0])=='=' || 
									isCharToIgnore(params.charAt(id[0]))))
									{
										p.curvature1 = readEqualReal(id, params); // We read the first parameter
										p.curvature2 = readReal(params, id, true); // We read the second parameter
										p.curvature3 = readReal(params, id, true); // We read the third parameter
										if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
										
									} else throw new IgnoreCommandException(-1);
								break;
								
							default : throw new IgnoreCommandException(-1);
						}
						break;
						
					case 'o' : // origin = {Real unit, Real unit}
					
						if(id[0]+6>size) throw new IgnoreCommandException(-1);
						
						if(params.charAt(++id[0])=='r' && params.charAt(++id[0])=='i' && params.charAt(++id[0])=='g' &&
							params.charAt(++id[0])=='i' && params.charAt(++id[0])=='n' && (params.charAt(++id[0])=='=' || 
							isCharToIgnore(params.charAt(id[0]))))
						{
							readChar(params, '=', id);
							readChar(params, '{', id);

							double x=Double.NaN, y=Double.NaN;
							
							goToNextChar(params, id); // We go to the next token
							if(id[0]<size && (Character.isDigit(params.charAt(id[0])) || params.charAt(id[0])=='.' || 
								params.charAt(id[0])=='-'))
								x = readRealUnit(id, params, true);// If there is the x, we read it
							
							readChar(params, ',', id);
							
							goToNextChar(params, id); // We go to the next token
							if(id[0]<size && (Character.isDigit(params.charAt(id[0])) || params.charAt(id[0])=='.' || 
								params.charAt(id[0])=='-')) 
								y = readRealUnit(id, params, true);// If there is the y, we read it
							
							readChar(params, '}', id);
							
							if(!Double.isNaN(x)) p.origin.x = x;
							if(!Double.isNaN(y)) p.origin.y = y;
							
							if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma

						}else throw new IgnoreCommandException(-1);
						break;
						
					case 'h' : 
						
						if(id[0]+4>size) throw new IgnoreCommandException(-1);
						
						if(params.charAt(++id[0])=='a' && params.charAt(++id[0])=='t' && 
							params.charAt(++id[0])=='c' && params.charAt(++id[0])=='h')
						{
							if(++id[0]>size) throw new IgnoreCommandException(-1);
							
							switch(params.charAt(id[0]))
							{
								case 'a' : // hatchangle = Real
									if(id[0]+5>size) throw new IgnoreCommandException(-1);
									
									if(params.charAt(++id[0])=='n' && params.charAt(++id[0])=='g' && 
										params.charAt(++id[0])=='l' && params.charAt(++id[0])=='e' &&
										(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
										{
											p.hatchAngle = readEqualReal(id, params);
											if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
											
										}else throw new IgnoreCommandException(-1);
									break;
								
								case 'w' : // hatchwidth = Real unit
									if(id[0]+5>size) throw new IgnoreCommandException(-1);
									
									if(params.charAt(++id[0])=='i' && params.charAt(++id[0])=='d' && 
										params.charAt(++id[0])=='t' && params.charAt(++id[0])=='h' &&
										(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
										{
											p.hatchWidth = readEqualRealUnit(id, params);
											if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
											
										}else throw new IgnoreCommandException(-1);
									break;	
									
								case 's' : // hatchsep = Real unit
									if(id[0]+3>size) throw new IgnoreCommandException(-1);
									
									if(params.charAt(++id[0])=='e' && params.charAt(++id[0])=='p' && 
										(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
										{
											p.hatchSep = readEqualRealUnit(id, params);
											if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
											
										}else throw new IgnoreCommandException(-1);
									break;
									
								case 'c' : // hatchcolor = colour
									if(id[0]+5>size) throw new IgnoreCommandException(-1);
									
									if(params.charAt(++id[0])=='o' && params.charAt(++id[0])=='l' && 
										params.charAt(++id[0])=='o' && params.charAt(++id[0])=='r' &&
										(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
										{
											String r = readEqualString(id, params, true);
											Color c = DviPsColors.getColour(r); // We search the colour
											if(c!=null) p.hatchCol = c;
	
											if(id[0]<size) goToNextChar(params, id); // We go to the next token : ','
											
										}else throw new IgnoreCommandException(-1);
									break;	
									
								default : throw new IgnoreCommandException(-1);
							}
						}else throw new IgnoreCommandException(-1);
						break;
				
					case 'r' :// rbracketlength = Real
						
						if(id[0]+14>size) throw new IgnoreCommandException(-1);
						
						if(params.charAt(++id[0])=='b' && params.charAt(++id[0])=='r' && params.charAt(++id[0])=='a' &&
							params.charAt(++id[0])=='c' && params.charAt(++id[0])=='k' && params.charAt(++id[0])=='e' &&
							params.charAt(++id[0])=='t' && params.charAt(++id[0])=='l' && params.charAt(++id[0])=='e' &&
							params.charAt(++id[0])=='n' && params.charAt(++id[0])=='g' && params.charAt(++id[0])=='t' &&
							params.charAt(++id[0])=='h' && (params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
							{
								p.arrowrBrLgth = readEqualReal(id, params);
								if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
								
							} else throw new IgnoreCommandException(-1);
						break;
						
					case 't' :
						if(id[0]+1>size) throw new IgnoreCommandException(-1);
						
						switch(params.charAt(++id[0]))
						{
							case 'b': //tb
								if(params.charAt(++id[0])=='a' && params.charAt(++id[0])=='r' &&
									params.charAt(++id[0])=='s' && params.charAt(++id[0])=='i' && params.charAt(++id[0])=='z' &&
									params.charAt(++id[0])=='e' && (params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
								{ // tbarsize = Real unit Real
									p.arrowTBarSD = readEqualRealUnit(id, params);
									if(id[0]<size) goToNextChar(params, id); // We go to the next token
									if(id[0]<size && (Character.isDigit(params.charAt(id[0])) || params.charAt(id[0])=='.')) 
										p.arrowTBarSN = readReal(params, id, true);// If there is the x, we read it
									
									if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
								}
								else throw new IgnoreCommandException(-1);
								break;
								
							case 'i': //ti
								if(id[0]+3>size) throw new IgnoreCommandException(-1);
								if(params.charAt(++id[0])=='c' && params.charAt(++id[0])=='k' && 
									params.charAt(++id[0])=='s')
								{ // ticks
									if(id[0]+1>size) throw new IgnoreCommandException(-1);
									switch(params.charAt(++id[0]))
									{
										case 't'://tickst
											if(id[0]+4>size) throw new IgnoreCommandException(-1);
											if(params.charAt(++id[0])=='y' && params.charAt(++id[0])=='l' &&
												params.charAt(++id[0])=='e' && (params.charAt(++id[0])=='=' || 
												isCharToIgnore(params.charAt(id[0]))))
											{//tickstyle = string
												p.ticksStyle = readString(params, id, false);
												if(id[0]<size) goToNextChar(params, id);
											}
											else throw new IgnoreCommandException(-1);
											break;
											
										case 'i'://ticksi
											if(id[0]+3>size) throw new IgnoreCommandException(-1);
											if(params.charAt(++id[0])=='z' && params.charAt(++id[0])=='e' &&
												(params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
											{//ticksize = real unit
												p.ticksSize = readEqualRealUnit(id, params);
												if(id[0]<size) goToNextChar(params, id);
											}
											else throw new IgnoreCommandException(-1);
											break;
											
										default :
											if(params.charAt(id[0])=='=' || isCharToIgnore(params.charAt(id[0])))//ticks = string
											{
												p.ticks = readString(params, id, false);
												if(id[0]<size) goToNextChar(params, id);
											}
											else throw new IgnoreCommandException(-1);
									}
								}
								break;
						}
						break;
				
					case 'u' : // unit = Real unit
						
						if(id[0]+4>size) throw new IgnoreCommandException(-1);
						
						if(params.charAt(++id[0])=='n' && params.charAt(++id[0])=='i' && 
							params.charAt(++id[0])=='t' && (params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
						{
							double r = readEqualReal(id, params); // We read the parameter
							
							goToNextChar(params, id); // We go to the next token
							
							if(id[0]<size && Character.isLetter(params.charAt(id[0])) && params.charAt(id[0])!=',') 
							{
								String lgth = readString(params, id, false);// If there is a unit of length, we read it
								r = convertInCm(r, lgth);
								p.unit = r;
							}
							else
								p.unit *= r;
							
							if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
							
						}else throw new IgnoreCommandException(-1);
						break;
						
					case 'x' : // xunit = Real unit
						
						if(id[0]+5>size) throw new IgnoreCommandException(-1);
						
						if(params.charAt(++id[0])=='u' && params.charAt(++id[0])=='n' && params.charAt(++id[0])=='i' && 
							params.charAt(++id[0])=='t' && (params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
						{
							double r = readEqualReal(id, params); // We read the parameter
							goToNextChar(params, id); // We go to the next token
							
							if(id[0]<size && Character.isLetter(params.charAt(id[0])) && params.charAt(id[0])!=',') 
							 	p.xUnit = convertInCm(r, readString(params, id, false));
							else p.xUnit *= r;
							
							if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
							
						}else throw new IgnoreCommandException(-1);
						break;
						
					case 'y' : // yunit = Real unit
						if(id[0]+5>size) throw new IgnoreCommandException(-1);
						
						if(params.charAt(++id[0])=='u' && params.charAt(++id[0])=='n' && params.charAt(++id[0])=='i' && 
							params.charAt(++id[0])=='t' && (params.charAt(++id[0])=='=' || isCharToIgnore(params.charAt(id[0]))))
						{
							double r = readEqualReal(id, params); // We read the parameter
							goToNextChar(params, id); // We go to the next token
							
							if(id[0]<size && Character.isLetter(params.charAt(id[0])) && params.charAt(id[0])!=',') 
							     p.yUnit = convertInCm(r, readString(params, id, false));
							else p.yUnit *= r;
							
							if(id[0]<size) goToNextChar(params, id); // We go to the next token : the comma
							
						}else throw new IgnoreCommandException(-1);
						break;
						
					case 'O': 
						if(id[0]+1>size) throw new IgnoreCommandException(-1);
						
						switch(params.charAt(++id[0]))
						{
							case 'x': // Ox
								p.ox = readEqualReal(id, params);
								if(id[0]<size) goToNextChar(params, id);
								break;
								
							case 'y': // Oy
								p.oy = readEqualReal(id, params);
								if(id[0]<size) goToNextChar(params, id); 
								break;
								
							default: throw new IgnoreCommandException(-1);
						}
						break;
						
					case 'D':
						if(id[0]+1>size) throw new IgnoreCommandException(-1);
						
						switch(params.charAt(++id[0]))
						{
							case 'x': // Dx
								p.dxIncrement = readEqualReal(id, params);
								if(id[0]<size) goToNextChar(params, id);
								break;
								
							case 'y': // Dy
								p.dyIncrement = readEqualReal(id, params);
								if(id[0]<size) goToNextChar(params, id); 
								break;
								
							default: throw new IgnoreCommandException(-1);
						}
						break;
						
					case 9 : // '\t'
					case ',' :
					case ' ' :
						id[0]++;
						break;
						
					default : 
						if(isCharToIgnore(params.charAt(id[0]))) id[0]++;
						else throw new IgnoreCommandException(-1);
				}//switch
			}
			catch(IgnoreCommandException e) // We jump to the next parameter
			{
				int cptPar = 0;
				int cptBrack = 0;
				while(id[0]<size && (params.charAt(id[0])!=',' || cptPar!=0 || cptBrack!=0))
				{
					if(params.charAt(id[0])=='{') cptBrack++;
					else if(params.charAt(id[0])=='(') cptPar++;
					else if(params.charAt(id[0])=='}') cptBrack--;
					else if(params.charAt(id[0])==')') cptPar--;
					id[0]++;
				}
				id[0]++;
			}
		}//while
	}//public void parseOptionsFigure
	
	
	
	
	
	
	/**
	 * Allows to convert a value with a given unit of length (cm, pt, mm, in, or null)
	 * to centimetre
	 * @param value The value to convert
	 * @param lgth The unit of length of value
	 * @return The value in centimetre
	 * @throws IgnoreCommandException If the given unit of length is invalid
	 */
	public static double convertInCm(double value, String lgth) throws IgnoreCommandException
	{
		if(lgth==null || lgth.length()==0) return value;
		lgth = lgth.toLowerCase();
		
		if(lgth.equals(PSTricksConstants.TOKEN_PS_PT))
			value/= PSTricksConstants.CM_VAL_PT; //by default the unit is postscript point
		else if(lgth.equals(PSTricksConstants.TOKEN_INCH))
			value*=PSTricksConstants.INCH_VAL_CM;
		else if(lgth.equals(PSTricksConstants.TOKEN_MM))
			value/=10.;
		else if(!lgth.equals(PSTricksConstants.TOKEN_CM))
			throw new IgnoreCommandException(-1);
		
		return value;
	}




	/**
	 * Allows to read the next real in the string.
	 * @param str The source.
	 * @param id The position to start the research (we use a pointer to return the position at the end of the research.
	 * @param errorOnCharBefore If a non-digit character is parsed before the real, and IgnoreCommand exception will occur.
	 * @return The real.
	 * @throws IgnoreCommandException If the next real is unreachable.
	 */
	public static double readReal(String str, int[] id, boolean errorOnCharBefore) throws IgnoreCommandException
	{
		try
		{
			if(id==null || id.length<1) throw new IgnoreCommandException(-1);
			
			int size = str.length(), i = id[0];
			String realS = ""; //$NON-NLS-1$
			boolean point = false;
			boolean negative = false;
			boolean positive = false;
			boolean isNegatif = false;
			
			while(i<size && str.charAt(i)!='.' && str.charAt(i)!='-' && str.charAt(i)!='+' && (str.charAt(i)<'0' || str.charAt(i)>'9')) 
			{
				if(errorOnCharBefore && str.charAt(i)!='\t' && str.charAt(i)!='\n' && str.charAt(i)!=' ' && str.charAt(i)!='\r') 
					throw new IgnoreCommandException(-1);
				i++;
			}
			
			if(i>=size) throw new IgnoreCommandException(-1);
			if(str.charAt(i)=='.') 
			{ 
				point = true; 
				realS='.'+realS;
			}
			else if(str.charAt(i)=='-') { negative = true; isNegatif = true; }
			else if(str.charAt(i)=='+') positive = true;
			else realS+=String.valueOf(str.charAt(i));
			i++;
			boolean again = true;
			
			while(again)
			{
				again = false;
				while(i<size && str.charAt(i)=='+')
				{
					positive=true;
					again=true;
					i++;
				}
				
				while(i<size && str.charAt(i)=='-')
				{
					negative=true;
					again=true;
					i++;
					isNegatif=!isNegatif;
				}
			}
			
			while(i<size && (str.charAt(i)=='.' || str.charAt(i)=='-' || str.charAt(i)=='+' ||
					(str.charAt(i)>='0' && str.charAt(i)<='9')))
			{
				if(str.charAt(i)=='.' && point) throw new IgnoreCommandException(-1);
				else if(str.charAt(i)=='-' && negative) throw new IgnoreCommandException(-1);
				else if(str.charAt(i)=='+' && positive) throw new IgnoreCommandException(-1);
				
				realS+=String.valueOf(str.charAt(i));
				i++;
			}
			
			id[0] = i;
			
			if(isNegatif)
				realS='-'+realS;
	
			return Double.valueOf(realS);
			
		}catch(Exception e)
		{
			e.printStackTrace();
			throw new IgnoreCommandException(-1);
		}
	}
	
	
	
	
	
	
	/**
	 * Allows to read the next string in the source str (the separator is ' ')
	 * @param str The source
	 * @param id The position to start the research (we use a pointer to return the position
	 * at the end of the research
	 * @return The string
	 * @throws IgnoreCommandException If the next string is unreachable
	 */
	public static String readString(String str, int[] id, boolean withNumber) 
				throws IgnoreCommandException
	{
		try
		{
			if(id==null || id.length<1) throw new IgnoreCommandException(-1);
		
			int size = str.length(), i = id[0];
			String string = "";//$NON-NLS-1$

			if(withNumber)
			{
				while(i<size && (str.charAt(i)<'a' || str.charAt(i)>'z') &&
					(str.charAt(i)<'A' || str.charAt(i)>'Z') && (str.charAt(i)<'0' || str.charAt(i)>'9')) i++;
				if(i>=size) throw new IgnoreCommandException(-1);
				while(i<size && ((str.charAt(i)>='a' && str.charAt(i)<='z') ||
						(str.charAt(i)>='A' && str.charAt(i)<='Z')|| (str.charAt(i)>='0' && str.charAt(i)<='9')))
				{
					string+=String.valueOf(str.charAt(i));
					i++;
				}
			}
			else
			{
				
				while(i<size && (str.charAt(i)<'a' || str.charAt(i)>'z') &&
					(str.charAt(i)<'A' || str.charAt(i)>'Z')) i++;
				if(i>=size) throw new IgnoreCommandException(-1);
				while(i<size && ((str.charAt(i)>='a' && str.charAt(i)<='z') ||
						(str.charAt(i)>='A' && str.charAt(i)<='Z')))
				{
					string+=String.valueOf(str.charAt(i));
					i++;
				}
			}
			id[0] = i;
			
			return string;
			
		}catch(Exception e) { throw new IgnoreCommandException(-1); }
	}
	
	
	
	
	
	
	/**
	 * Allows to read the next integer in the string
	 * @param str The source
	 * @param id The position to start the research (we use a pointer to return the position
	 * at the end of the research
	 * @return The integer
	 * @throws IgnoreCommandException If the next integer is unreachable
	 */
	public static int readInteger(String str, int[] id) throws IgnoreCommandException
	{
		try
		{
			if(id==null || id.length<1) throw new IgnoreCommandException(-1);
			
			int size = str.length(), i = id[0];
			String realS = "";//$NON-NLS-1$
			
			while(i<size && ((str.charAt(i)<'0' || str.charAt(i)>'9') && str.charAt(id[0])!='+')) i++;
			if(i>=size) throw new IgnoreCommandException(-1);
			
			boolean positif = str.charAt(id[0])=='+';
			if(positif) i++;
			while(str.charAt(i)=='+')
				i++;
			
			while(i<size && ((str.charAt(i)>='0' && str.charAt(i)<='9') || str.charAt(i)=='+'))
			{
				if(str.charAt(i)=='+')
				{
					if(positif)
						throw new IgnoreCommandException(-1);
					positif = true;
				}
				else
				{
					realS+=String.valueOf(str.charAt(i));
					i++;
				}
			}
			
			id[0] = i;
			
			return Integer.valueOf(realS);
		
		}catch(Exception e)
		{
			e.printStackTrace();
			throw new IgnoreCommandException(-1);
		}
	}
	
	
	
	
	
	
	
	/**
	 * Allows to go to the next char (not ' ', '\t', '\n', '\r')
	 * @param str The source
	 * @param id The position of start
	 * @throws IgnoreCommandException If there is no char
	 */
	public static void goToNextChar(String str, int[] id) throws IgnoreCommandException
	{
		try
		{
			if(str==null || id[0]<0) throw new IgnoreCommandException(-1); 
			
			int size = str.length(), i = id[0];
			
			while(i<size && isCharToIgnore(str.charAt(i)))
				i++;
		
			id[0] = i;
			
		}catch(Exception e)
		{
			e.printStackTrace();
			throw new IgnoreCommandException(-1);
		}
	}
	
	
	
	
	
	/**
	 * Allows to read '= Real unit' 
	 * @param id The position of the start of the research (a pointer : use id[0])
	 * @param params The string to parse
	 * @return The read value in cm
	 * @throws IgnoreCommandException 
	 */
	public static double readEqualRealUnit(int[] id, String params) throws IgnoreCommandException
	{
		try
		{
			int size = params.length();
			
			while(id[0]<size && params.charAt(id[0])!='=') id[0]++; // We search the '='
			if(id[0]>=size) throw new IgnoreCommandException(-1);
			id[0]++;
			
			goToNextChar(params, id);
			if((params.charAt(id[0])<'0' || params.charAt(id[0])>'9') &&
				params.charAt(id[0])!='.' && params.charAt(id[0])!='-')
				throw new IgnoreCommandException(-1);
	
			double r = readReal(params, id, true); // We read the parameter
	
			goToNextChar(params, id); // We go to the next token
	
			String lgth = null;
			if(id[0]<size && Character.isLetter(params.charAt(id[0]))) 
				lgth = readString(params, id, false);// If there is a unit of length, we read it
			
			return convertInCm(r, lgth);
		}catch(Exception e)
		{ throw new IgnoreCommandException(-1); }
	}
	
	
	
	
	/**
	 * Allows to read 'Real unit' converted in cm.
	 * @param id The position of the start of the research (a pointer : use id[0])
	 * @param params The string to parse
	 * @param errorOnCharBefore If a non-digit character is parsed before the real, and IgnoreCommand exception will occur.
	 * @return The read value 
	 * @throws IgnoreCommandException 
	 */
	public static double readRealUnit(int[] id, String params, boolean errorOnCharBefore) throws IgnoreCommandException
	{
		int size = params.length(), i;

		goToNextChar(params, id); // We go to the next token
		double r = readReal(params, id, errorOnCharBefore); // We read the parameter	
		
		goToNextChar(params, id);
		i = id[0];
		String lgth = null;

		if(id[0]<size && ((params.charAt(id[0])>='a' && params.charAt(id[0])<='z') ||
			(params.charAt(id[0])>='A' && params.charAt(id[0])<='Z')))
			try
			{
				if(id[0]<size) 
					lgth = readString(params, id, false);// If there is a unit of length, we read it
				
				r = convertInCm(r, lgth);
			}catch(Exception e) { id[0] = i; }
		
		return r;
	}
	
	
	
	
	
	/**
	 * Allows to read '= string' 
	 * @param id The position of the start of the research (a pointer : use id[0])
	 * @param withNumber If true The string can contains number like 'color0'
	 * @param params The string to parse
	 * @return The read value 
	 * @throws IgnoreCommandException 
	 */
	public static String readEqualString(int[] id, String params, boolean withNumber) throws IgnoreCommandException
	{
		int size = params.length();
		
		while(id[0]<size && params.charAt(id[0])!='=') id[0]++; // We search the '='
		if(id[0]>=size) throw new IgnoreCommandException(-1);
		id[0]++;
		
		return readString(params, id, withNumber); 
	}
	
	
	

	
	
	/**
	 * Allows to read '= Real' 
	 * @param id The position of the start of the research (a pointer : use id[0])
	 * @param params The string to parse
	 * @return The read value 
	 * @throws IgnoreCommandException 
	 */
	public static double readEqualReal(int[] id, String params) throws IgnoreCommandException
	{
		int size = params.length();
		
		while(id[0]<size && params.charAt(id[0])!='=') id[0]++; // We search the '='
		if(id[0]>=size) throw new IgnoreCommandException(-1);
		id[0]++;
		
		goToNextChar(params, id);
		if((params.charAt(id[0])<'0' || params.charAt(id[0])>='9') &&
			params.charAt(id[0])!='.' && params.charAt(id[0])!='-' && params.charAt(id[0])!='+')
			throw new IgnoreCommandException(-1);
		
		return readReal(params, id, true); // We read the parameter
	}
	
	
	
	/**
	 * Allows to read '= Integer' 
	 * @param id The position of the start of the research (a pointer : use id[0])
	 * @param params The string to parse
	 * @return The read value 
	 * @throws IgnoreCommandException 
	 */
	public static int readEqualInteger(int[] id, String params) throws IgnoreCommandException
	{
		int size = params.length();
		
		while(id[0]<size && params.charAt(id[0])!='=') id[0]++; // We search the '='
		if(id[0]>=size) throw new IgnoreCommandException(-1);
		id[0]++;
		
		goToNextChar(params, id);
		if((params.charAt(id[0])<'0' || params.charAt(id[0])>='9') && params.charAt(id[0])!='+')
			throw new IgnoreCommandException(-1);
		
		return readInteger(params, id); // We read the parameter
	}
	
	
	
	
	
	/**
	 * Allows to read one coordinate: (x,y). Beware, for missing coordinates like (5,)
	 * this function will return (5,NaN) instead of (5,1); because in several cases, 
	 * the missing coordinate is very important.
	 * @param s The string to parse.
	 * @param i The position of the start.
	 * @param nLine The number of the line where the code is.
	 * @return The read coordinate.
	 * @throws InvalidFormatCommandException If a problem occurred.
	 */
	public static Point2D.Double readOneCoordinate(String s, int i, int nLine, boolean withUnit) throws InvalidFormatCommandException
	{
		try
		{
			int size = s.length();
			int[] id = {i};
		
			while(i<size && s.charAt(id[0])!='(') id[0]++; // We search the '('
			if(i>=size) throw new InvalidFormatCommandException(nLine);
			id[0]++;
			
			double x;
			
			goToNextChar(s, id); // We go to the next token
		
			if(id[0]<size && (Character.isDigit(s.charAt(id[0])) || s.charAt(id[0])=='.' || 
				s.charAt(id[0])=='-' || s.charAt(id[0])=='+')) 
				if(withUnit) x = readRealUnit(id, s, true);
				else x = readReal(s, id, true);
			else x = Double.NaN;
		
			while(id[0]<size && s.charAt(id[0])!=',')// We search the ','
			{
				if(!isCharToIgnore(s.charAt(id[0])))
					throw new InvalidFormatCommandException(nLine);
				id[0]++; // We search the ')'
			}
			
			if(id[0]>=size) throw new InvalidFormatCommandException(nLine);
			id[0]++;
			
			double y;
			goToNextChar(s, id); // We go to the next token
			
			if(id[0]<size && (Character.isDigit(s.charAt(id[0])) || s.charAt(id[0])=='.' || s.charAt(id[0])=='-' || s.charAt(id[0])=='+')) 
				if(withUnit) y = readRealUnit(id, s, true);
				else y = readReal(s, id, true);
			else y = Double.NaN;
			
			while(id[0]<size && s.charAt(id[0])!=')') 
			{
				if(!isCharToIgnore(s.charAt(id[0])))
					throw new InvalidFormatCommandException(nLine);
				id[0]++; // We search the ')'
			}
			
			if(id[0]>=size) throw new InvalidFormatCommandException(nLine);
			id[0]++;
			
			return new Point2D.Double(x, y);
			
		}catch(Exception e){throw new InvalidFormatCommandException(nLine);}
	}





	/**
	 * Actions to do when a line must be parsed
	 */
	public abstract int actionsOnLine(int line, boolean hasStar, Vector<PSTricksParameters> ps,
			String params) throws InvalidFormatCommandException;


	/**
	 * Actions to do when a polygon must be parsed
	 */
	public abstract int actionsOnPolygon(int line, boolean hasStar, Vector<PSTricksParameters> ps, 
			String string) throws InvalidFormatCommandException;


	/**
	 * Actions to do when a diamond or a triangle must be parsed
	 */
	public abstract int actionsOnDiamondTriangle(int line, boolean hasStar, Vector<PSTricksParameters> ps, 
			String string, boolean isDiamondNotTriangle) throws InvalidFormatCommandException;


	/**
	 * Actions to do when a Bézier curve must be parsed
	 */
	public abstract int actionsOnBezierCurve(int line, boolean hasStar, Vector<PSTricksParameters> ps, 
			String string) throws InvalidFormatCommandException;


	/**
	 * Actions to do when an arc or a wedge must be parsed.
	 */
	public abstract int actionsOnArcWedge(int line, boolean hasStar, Vector<PSTricksParameters> ps, String params, 
			boolean isArc) throws InvalidFormatCommandException;


	/**
	 * Actions to do when a grid must be parsed.
	 */
	public abstract int actionsOnGridAxes(int line, Vector<PSTricksParameters> ps, Point2D.Double pictureSWPt, 
			Point2D.Double pictureNEPt, String string, boolean isGrid) throws InvalidFormatCommandException;


	

	/**
	 * Allows to parse a psset command.
	 * @param str The string to parse.
	 * @param p The parameters to set.
	 */
	public static void parsepssetCommand(String str, PSTricksParameters p)
	{
		try
		{
			if(str==null || p==null) return;
			
			int lgth = str.length();
			int[] i = {0};
			
			while(i[0]<lgth && str.charAt(i[0])!='{') i[0]++;
			if(i[0]>=lgth) throw new IgnoreCommandException(-1);
			i[0]++;
			
			goToNextChar(str, i);
			parseOptionsFigure(str.substring(i[0]), p, !p.psCustomP.fromPsCustom);
		
		}catch(Exception e){e.printStackTrace();}
	}
	
	
	
	
/*
	public static void parseuputCommand(String str, @SuppressWarnings("unused")boolean hasStar, Vector<PSTricksParameters> params)
				throws Exception
	{
		if(str==null || str.length()==0 || params==null) throw new InvalidFormatCommandException("uput", -1);
		
		int[] i = {0};
		int size = str.length();
		double labelSep=PSTricksConstants.DEFAULT_LABEL_SEP, rotation=0, refA=0;
		Point2D.Double coord = (Point2D.Double)params.lastElement().origin.clone();
		String refAngle=null;
	//	boolean isSum = false;
		
		goToNextChar(str, i);
		
		if(i[0]<size && str.charAt(i[0])=='{') // The labelsep
		{
			i[0]++;
			labelSep = readRealUnit(i, str);
			if(i[0]<size) goToNextChar(str, i);
			if(str.charAt(i[0])!='}') throw new InvalidFormatCommandException("uput", -1);
			i[0]++;
			if(i[0]<size) goToNextChar(str, i);
		}
		
		if(i[0]<size && str.charAt(i[0])=='[') // The refangle
		{
			i[0]++;
			refAngle = readString(str, i, true);
			if(i[0]<size) goToNextChar(str, i);
			if(str.charAt(i[0])!=']') throw new InvalidFormatCommandException("uput", -1);
			i[0]++;
			if(i[0]<size) goToNextChar(str, i);
			
			try{ refA = Double.valueOf(refAngle);}
			catch(Exception e)
			{
				if(refAngle.length()==1)
				{
					char c = refAngle.charAt(0);
					
					if(c==PSTricksConstants.TOKEN_UPUT_DOWN)
						refA = PSTricksConstants.DEFAULT_UPUT_DOWN_ANGLE;
					else if(c==PSTricksConstants.TOKEN_UPUT_UP)
						refA = PSTricksConstants.DEFAULT_UPUT_UP_ANGLE;
					else if(c==PSTricksConstants.TOKEN_UPUT_LEFT)
						refA = PSTricksConstants.DEFAULT_UPUT_LEFT_ANGLE;
					else if(c==PSTricksConstants.TOKEN_UPUT_RIGHT)
						refA = PSTricksConstants.DEFAULT_UPUT_RIGHT_ANGLE;
				}else
				{
					char c2 = refAngle.charAt(1), c1 = refAngle.charAt(0);
					
					
					if( (c1==PSTricksConstants.TOKEN_UPUT_DOWN && c2==PSTricksConstants.TOKEN_UPUT_LEFT) ||
						(c2==PSTricksConstants.TOKEN_UPUT_DOWN && c1==PSTricksConstants.TOKEN_UPUT_LEFT))
						refA = PSTricksConstants.DEFAULT_UPUT_UP_RIGHT_ANGLE;
					else if( (c1==PSTricksConstants.TOKEN_UPUT_UP && c2==PSTricksConstants.TOKEN_UPUT_RIGHT) ||
						(c2==PSTricksConstants.TOKEN_UPUT_UP && c1==PSTricksConstants.TOKEN_UPUT_RIGHT))
						refA = PSTricksConstants.DEFAULT_UPUT_UP_RIGHT_ANGLE;
					else if( (c1==PSTricksConstants.TOKEN_UPUT_UP && c2==PSTricksConstants.TOKEN_UPUT_LEFT) ||
						(c2==PSTricksConstants.TOKEN_UPUT_UP && c1==PSTricksConstants.TOKEN_UPUT_LEFT))
						refA = PSTricksConstants.DEFAULT_UPUT_UP_LEFT_ANGLE;
					else if( (c1==PSTricksConstants.TOKEN_UPUT_DOWN && c2==PSTricksConstants.TOKEN_UPUT_LEFT) ||
						(c2==PSTricksConstants.TOKEN_UPUT_DOWN && c1==PSTricksConstants.TOKEN_UPUT_LEFT))
						refA = PSTricksConstants.DEFAULT_UPUT_DOWN_LEFT_ANGLE;
					else if(c1==PSTricksConstants.TOKEN_UPUT_DOWN && c2==PSTricksConstants.TOKEN_UPUT_DOWN)
						refA = PSTricksConstants.DEFAULT_UPUT_DOWN_ANGLE;
					else if(c1==PSTricksConstants.TOKEN_UPUT_UP && c2==PSTricksConstants.TOKEN_UPUT_UP)
						refA = PSTricksConstants.DEFAULT_UPUT_UP_ANGLE;
					else if(c1==PSTricksConstants.TOKEN_UPUT_LEFT && c2==PSTricksConstants.TOKEN_UPUT_LEFT)
						refA = PSTricksConstants.DEFAULT_UPUT_LEFT_ANGLE;
					else if(c1==PSTricksConstants.TOKEN_UPUT_RIGHT && c2==PSTricksConstants.TOKEN_UPUT_RIGHT)
						refA = PSTricksConstants.DEFAULT_UPUT_RIGHT_ANGLE;
					else if(c1==PSTricksConstants.TOKEN_UPUT_RIGHT && c2==PSTricksConstants.TOKEN_UPUT_LEFT)
						refA = PSTricksConstants.DEFAULT_UPUT_LEFT_ANGLE;
					else if(c1==PSTricksConstants.TOKEN_UPUT_LEFT && c2==PSTricksConstants.TOKEN_UPUT_RIGHT)
						refA = PSTricksConstants.DEFAULT_UPUT_RIGHT_ANGLE;
					else if(c1==PSTricksConstants.TOKEN_UPUT_DOWN && c2==PSTricksConstants.TOKEN_UPUT_UP)
						refA = PSTricksConstants.DEFAULT_UPUT_UP_ANGLE;
					else if(c1==PSTricksConstants.TOKEN_UPUT_UP && c2==PSTricksConstants.TOKEN_UPUT_DOWN)
						refA = PSTricksConstants.DEFAULT_UPUT_DOWN_ANGLE;
				}
			}
		}
		else throw new InvalidFormatCommandException("uput", -1);
		
		if(i[0]<size && str.charAt(i[0])=='{') // The rotation
		{
			try
			{
				i[0]++;
				if(str.charAt(i[0])=='*') { i[0]++; }
				if(Character.isDigit(str.charAt(i[0])) || str.charAt(i[0])=='-' || str.charAt(i[0])=='.')
					rotation = readReal(str, i);
				else 
				{
					String rot = readString(str, i, true); 
					if(rot.equals(PSTricksConstants.TOKEN_PUT_ANGLE_0)==0)
						rotation = 0;
					else if(rot.equals(PSTricksConstants.TOKEN_PUT_ANGLE_0_STAR)==0)
					{
						rotation = 0;
				//		isSum = true;
					}
					else if(rot.equals(PSTricksConstants.TOKEN_PUT_ANGLE_90)==0)
						rotation = 90;
					else if(rot.equals(PSTricksConstants.TOKEN_PUT_ANGLE_90_STAR)==0)
					{
						rotation = 90;
					//	isSum = true;
					}
					else if(rot.equals(PSTricksConstants.TOKEN_PUT_ANGLE_180)==0)
						rotation = 180;
					else if(rot.equals(PSTricksConstants.TOKEN_PUT_ANGLE_180_STAR)==0)
					{
						rotation = 180;
				//		isSum = true;
					}
					else if(rot.equals(PSTricksConstants.TOKEN_PUT_ANGLE_270)==0)
						rotation = 270;
					else if(rot.equals(PSTricksConstants.TOKEN_PUT_ANGLE_270_STAR)==0)
					{
						rotation = 270;
					//	isSum = true;
					}
				}
			}catch(Exception e) {throw new InvalidFormatCommandException("uput", -1);}

			if(i[0]<size) goToNextChar(str, i);
			if(str.charAt(i[0])!='}') throw new InvalidFormatCommandException("uput", -1);
			i[0]++;
			if(i[0]<size) goToNextChar(str, i);
		}
		
		if(i[0]<size && str.charAt(i[0])=='(') // The coordinate
		{
			coord = readOneCoordinate(str, i[0], -1);
			if(coord==null) throw new InvalidFormatCommandException("uput", -1);
			while(i[0]<size && str.charAt(i[0])!=')') i[0]++;
			if(i[0]>=size) throw new IgnoreCommandException(-1);
			i[0]++;
			if(i[0]<size) goToNextChar(str, i);
		}
		else throw new InvalidFormatCommandException("uput", -1);
		
		if(i[0]<size && str.charAt(i[0])=='{')
		{
			PSTricksParameters p = new PSTricksParameters(params.lastElement());

			if(Double.isNaN(coord.x)) coord.x = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
			if(Double.isNaN(coord.y)) coord.y = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
			
			p.setPosition(coord.x, coord.y, true, rotation, 0, PositionParameters.REF_POINT_RIGHT, labelSep, refA);
			if(params.size()>0)
			{
				PSTricksParameters p2 = params.lastElement();
				p.origin.setLocation(p2.origin.x, p2.origin.y);
			}
			params.add(p);
		}
		else throw new InvalidFormatCommandException("uput", -1);
	}
	*/
	
	
	
	
	/**
	 * Allows to parse a rput command
	 * @param str The string to parse.
	 * @param hasStar If it is rput*
	 * @param params The PsTricks parameters
	 * @throws Exception If the command is invalid
	 */
	public static void parserputCommand(String str, boolean hasStar, Vector<PSTricksParameters> params) throws Exception
	{
		if(str==null || str.length()==0 || params==null) throw new InvalidFormatCommandException("rput", -1);//$NON-NLS-1$
		
		int[] i = {0};
		int size = str.length();
		boolean isValidFormat = false;
		boolean isSumRot=true;
		String refPt = null;
		double rotation = 0;
		Point2D.Double coord = (Point2D.Double)params.lastElement().origin.clone();
		
		goToNextChar(str, i);
			
		if(i[0]<size && str.charAt(i[0])=='[')
		{
			i[0]++;
			refPt = readString(str, i, false);
			if(refPt==null || refPt.length()==0) throw new InvalidFormatCommandException("rput", -1);//$NON-NLS-1$
			if(i[0]<size) goToNextChar(str, i);
			if(str.charAt(i[0])!=']') throw new InvalidFormatCommandException("rput", -1);//$NON-NLS-1$
			i[0]++;
			if(i[0]<size) goToNextChar(str, i);
		}
		
		if(i[0]<size && str.charAt(i[0])=='{')
		{
			try
			{
				isValidFormat = true;
				i[0]++;
				if(str.charAt(i[0])=='*') { isSumRot=false; i[0]++; }
				if(Character.isDigit(str.charAt(i[0])) || str.charAt(i[0])=='-' || str.charAt(i[0])=='.')
					rotation = readReal(str, i, true);
				else 
				{
					String rot = readString(str, i, true); 
					if(rot.equals(PSTricksConstants.TOKEN_PUT_ANGLE_0))
						rotation = 0;
					else if(rot.equals(PSTricksConstants.TOKEN_PUT_ANGLE_0_STAR))
					{
						rotation = 0;
						isSumRot = false;
					}else if(rot.equals(PSTricksConstants.TOKEN_PUT_ANGLE_90))
						rotation = 90;
					else if(rot.equals(PSTricksConstants.TOKEN_PUT_ANGLE_90_STAR))
					{
						rotation = 90;
						isSumRot = false;
					}else if(rot.equals(PSTricksConstants.TOKEN_PUT_ANGLE_180))
						rotation = 180;
					else if(rot.equals(PSTricksConstants.TOKEN_PUT_ANGLE_180_STAR))
					{
						rotation = 180;
						isSumRot = false;
					}else if(rot.equals(PSTricksConstants.TOKEN_PUT_ANGLE_270))
						rotation = 270;
					else if(rot.equals(PSTricksConstants.TOKEN_PUT_ANGLE_270_STAR))
					{
						rotation = 270;
						isSumRot = false;
					}
				}
			}catch(Exception e) {throw new InvalidFormatCommandException("rput", -1);}//$NON-NLS-1$

			if(i[0]<size) goToNextChar(str, i);
			if(str.charAt(i[0])!='}') throw new InvalidFormatCommandException("rput", -1);//$NON-NLS-1$
			i[0]++;
			if(i[0]<size) goToNextChar(str, i);
		}
		
		if(i[0]<size && str.charAt(i[0])=='(')
		{
			isValidFormat = true;
			coord = readOneCoordinate(str, i[0], -1, true);
			if(coord==null) throw new InvalidFormatCommandException("rput", -1);//$NON-NLS-1$
			while(i[0]<size && str.charAt(i[0])!=')') i[0]++;
			if(i[0]>=size) throw new IgnoreCommandException(-1);
			i[0]++;
			if(i[0]<size) goToNextChar(str, i);
		}
		
		if(i[0]<size && str.charAt(i[0])=='{')
		{
			PSTricksParameters p = new PSTricksParameters(params.lastElement());

			if(Double.isNaN(coord.x)) coord.x = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
			if(Double.isNaN(coord.y)) coord.y = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
			
			int xRefPoint = PositionParameters.REF_POINT_DEFAULT_CENTER;
			int yRefPoint = PositionParameters.REF_POINT_DEFAULT_CENTER;
			if(refPt==null) xRefPoint = yRefPoint = PositionParameters.REF_POINT_DEFAULT_CENTER;
			else
			{
				char c = refPt.charAt(0);
				
				if(c==PSTricksConstants.TOKEN_RPUT_BASE)
					xRefPoint = PositionParameters.REF_POINT_BASE;
				else if(c==PSTricksConstants.TOKEN_RPUT_BOTTOM)
					xRefPoint = PositionParameters.REF_POINT_BOTTOM;
				else if(c==PSTricksConstants.TOKEN_RPUT_TOP)
					xRefPoint = PositionParameters.REF_POINT_TOP;
				else if(c==PSTricksConstants.TOKEN_RPUT_LEFT)
					yRefPoint = PositionParameters.REF_POINT_LEFT;
				else if(c==PSTricksConstants.TOKEN_RPUT_RIGHT)
					yRefPoint = PositionParameters.REF_POINT_RIGHT;

				if(refPt.length()>1)
				{
					c = refPt.charAt(1);
					if(c==PSTricksConstants.TOKEN_RPUT_BASE)
						xRefPoint = PositionParameters.REF_POINT_BASE;
					else if(c==PSTricksConstants.TOKEN_RPUT_BOTTOM)
						xRefPoint = PositionParameters.REF_POINT_BOTTOM;
					else if(c==PSTricksConstants.TOKEN_RPUT_TOP)
						xRefPoint = PositionParameters.REF_POINT_TOP;
					else if(c==PSTricksConstants.TOKEN_RPUT_LEFT)
						yRefPoint = PositionParameters.REF_POINT_LEFT;
					else if(c==PSTricksConstants.TOKEN_RPUT_RIGHT)
						yRefPoint = PositionParameters.REF_POINT_RIGHT;
				}
			}
			
			p.setPosition(coord.x, coord.y, isSumRot, rotation, 0, xRefPoint+yRefPoint, 0., 0);
			if(params.size()>0)
			{
				PSTricksParameters p2 = params.lastElement();
				p.origin.setLocation(p2.origin.x, p2.origin.y);
			}
			params.add(p);
		}else throw new InvalidFormatCommandException("rput", -1);//$NON-NLS-1$
		
		if(!isValidFormat) throw new InvalidFormatCommandException("rput", -1);//$NON-NLS-1$
	}
	
	
	
	
	/**
	 * Allow to move a point according to the parameters (Useful for dots, texts).<br>
	 */
	public static Point2D.Double moveCoordinate(Vector<PSTricksParameters> params, Point2D.Double pt)
	{
		if(params==null)
			throw new IllegalArgumentException();
		
		int i, size = params.size();
		double lastAngle=0;
		Point2D.Double newPt = new Point2D.Double(0, 0);
		PSTricksParameters p;
		PositionParameters pos;
		Vector<Point2D.Double> gcs = new Vector<Point2D.Double>();
		boolean started = false;
		
		if(size==0) return new Point2D.Double(0,0);
		gcs.add(new Point2D.Double(0,0));
		
		for(i=1; i<size; i++)
		{
			p = params.elementAt(i);
			pos = p.getPosition();
			
			if(pos!=null)
			{
				newPt.x+=pos.dx * (p.unit==PSTricksConstants.DEFAULT_UNIT?  p.xUnit : p.unit);
				newPt.y+=pos.dy * (p.unit==PSTricksConstants.DEFAULT_UNIT? p.yUnit : p.unit);

				if(pos.angleRput!=0 || !pos.isSumRotation)
				{
					if(started)	
						newPt = rotatePoint(newPt, gcs.lastElement(), toRadians(lastAngle));
					
					gcs.add((Point2D.Double)newPt.clone());
					
					if(pos.isSumRotation)
						 lastAngle += (360/params.elementAt(i-1).degrees)*pos.angleRput;
					else lastAngle =  (360/params.elementAt(i-1).degrees)*pos.angleRput;
					
					started = true;
				}
			}
		}
		
		if(!gcs.isEmpty() && started)
			newPt = rotatePoint(new Point2D.Double(newPt.x+pt.x, newPt.y+pt.y), gcs.lastElement(), toRadians(lastAngle));
			
		if(!started)
			newPt.setLocation(newPt.x+pt.x, newPt.y+pt.y);
		
		gcs.removeAllElements();
		
		return newPt;
	}

	
	
	
	/**
	 * Allows to rotate a point with as reference an other point
	 * @param p The point to rotate
	 * @param gravityC The point of reference
	 * @param theta The angle of rotation (in radian)
	 * @return The rotated point
	 */
	public static Point2D.Double rotatePoint(Point2D.Double p, Point2D.Double gravityC, double theta)
	{
		if(theta==0) return (Point2D.Double)p.clone();
		
		Point2D.Double pt = new Point2D.Double();

		pt.x = cos(theta) * (p.x - gravityC.x) - sin(theta)* (p.y - gravityC.y) + gravityC.x;
		pt.y = sin(theta) * (p.x - gravityC.x) + cos(theta)* (p.y - gravityC.y) + gravityC.y;

		return pt;
	}



	/**
	 * Allows to parse the command \degrees[...].
	 * @param cmd The command to parse.
	 * @param p The current parameters.
	 * @param nLine The number of the line of the command.
	 * @return The position of the end of the command in the string cmd.
	 * @throws InvalidFormatCommandException If the format of the command is not correct.
	 */
	public int parseDegreesCommand(String cmd, PSTricksParameters p, int nLine) throws InvalidFormatCommandException
	{
		try
		{
			int[] i = {0};
			boolean end = false;
			
			while(i[0]<cmd.length() && cmd.charAt(i[0])!='[')
			{
				if(cmd.charAt(i[0])=='\\') end = true;
				else i[0]++;
			}
			
			if(i[0]>=cmd.length()) throw new InvalidFormatCommandException(nLine);
			
			double degrees;
			
			if(end) degrees = 360;
			else    degrees = readReal(cmd, i, true);
			
			p.onRadians = false;
			p.degrees = degrees;
			
			if(end) return 0;
			return i[0]+1;
			
		}catch(Exception e)
		{
			throw new InvalidFormatCommandException("degrees", nLine); //$NON-NLS-1$
		}
	}


	

	/**
	 * Action to do when a framed box is parsed.
	 * @since 1.7
	 * @param type The kind of box :<br>
	 * 0: rectangle<br>
	 * 1: circle<br>
	 * 2: triangle<br>
	 * 3: diamond<br>
	 * 4: ellipse<br>
	 * 5: double rectangle<br>
	 * 6: rectangle with shadow<br>
	 * @throws InvalidFormatCommandException When the format of the command is invalid.
	 */
	public abstract int actionsOnFramedBox(int nLine, boolean hasStar, Vector<PSTricksParameters> ps, 
			String cmd, String params, int type) throws InvalidFormatCommandException;
	
	
	
	/**
	 * Action to do when one or several framed boxes have been parsed.
	 * @since 1.7
	 * @throws InvalidFormatCommandException When the format of the command is invalid.
	 */
	public abstract void actionsOnTerminatedFramedBoxes(Vector<PSTricksParameters> params) 
			throws InvalidFormatCommandException;



	
	/**
	 * Parse the \begin{figure} command.
	 * @param f The code to parse.
	 * @param nLine The number of the current line.
	 * @throws InvalidFormatCommandException 
	 * @since 0.3
	 */
	public static int parseBeginFigureCmd(String f, int nLine) throws InvalidFormatCommandException
	{
		int[] i = {0};
		try
		{
			goToNextChar(f, i);
			if(f.charAt(i[0])=='[')
			{
				i[0]++;
				goToNextChar(f, i);
				String code = readString(f, i, false);
				goToNextChar(f, i);
				
				if(f.charAt(i[0])!=']')
					throw new InvalidFormatCommandException("\\begin{figure}", nLine); //$NON-NLS-1$
				
				PSTricksParameters.tokenPosition = code;
				return i[0]+1;
			}
			throw new InvalidFormatCommandException("\\begin{figure}", nLine); //$NON-NLS-1$
			
		}catch(IgnoreCommandException e)
		{
			throw new InvalidFormatCommandException("\\begin{figure}", nLine); //$NON-NLS-1$
		}
		
	}


	/**
	 * Parses the command <code>pscustom</code>.
	 * @param line The current line.
	 * @param hasStar True if the command <code>pscustom</code> has a star.
	 * @param params The pstricks parameters.
	 * @param txt The text to parse.
	 * @return The index of the last parsed token.
	 * @throws InvalidFormatCommandException If the command is invalid.
	 * @since 0.3
	 */
	public int parsePsCustomCommand(int line, boolean hasStar, Vector<PSTricksParameters> params, 
									String txt) throws InvalidFormatCommandException
	{
		int i[] = {0};
		PSTricksParameters pstp 	= new PSTricksParameters(params.lastElement());
		params.add(pstp);
		pstp.psCustomP.hasStar 		= hasStar;
		pstp.psCustomP.fromPsCustom = true;
		
		try
		{
			goToNextChar(txt, i);
			
			if(txt.charAt(i[0])=='[') // Analysis of the options
			{
				int j = txt.indexOf("]", i[0]+1);//$NON-NLS-1$
				if(j==-1) throw new InvalidFormatCommandException("pscustom", line); //$NON-NLS-1$
				
				parseOptionsFigure(txt.substring(i[0]+1, j), pstp, true);
				i[0] = j+1;
				goToNextChar(txt, i);
			}
			
			if(txt.charAt(i[0])=='{')
				i[0]++;
			else
				throw new InvalidFormatCommandException("pscustom", line); //$NON-NLS-1$
			
			pstp.isShadow	= false; //shadow, double line must be activated be \openshadow.
			pstp.dbleLine	= false;
		
			return i[0];
			
		} catch(IgnoreCommandException e)
		{
			throw new InvalidFormatCommandException("pscustom", line); //$NON-NLS-1$
		}
	}



	/**
	 * Actions to do when a <code>pscustom</code> command is finished.
	 * @param param The pstricks parameters.
	 * @since 0.3
	 */
	public abstract void actionsOnterminatedPsCustom(PSTricksParameters param) throws NotFullyManagedException;



	
	/**
	 * Parses to parameters of the pstricks command <code>\openshadow</code>.
	 * @param param The pstricks parameters.
	 * @param line The number of the current line.
	 * @param txt The code to parse.
	 * @return The position of the last parsed token.
	 * @throws InvalidFormatCommandException If the command is invalid.
	 * @since 0.3
	 */
	public int parseOpenShadowCommand(PSTricksParameters param, int line, String txt) 
					throws InvalidFormatCommandException
	{
		int i[] = {0};
		
		try
		{
			goToNextChar(txt, i);
			
			if(txt.charAt(i[0])=='[') // Analysis of the options
			{
				PSTricksParameters pstp = new PSTricksParameters(param);
				
				int j = txt.indexOf("]", i[0]+1);//$NON-NLS-1$
				if(j==-1) throw new InvalidFormatCommandException("openshadow", line); //$NON-NLS-1$
				
				parseOptionsFigure(txt.substring(i[0]+1, j), pstp, true);
				i[0] = j+1;
				
				param.shadowAngle 	= pstp.shadowAngle;
				param.shadowCol		= pstp.shadowCol;
				param.shadowSize	= pstp.shadowSize;
			}
			param.isShadow = true;
			
			return i[0];
			
		} catch(IgnoreCommandException e)
		{
			throw new InvalidFormatCommandException("openshadow", line); //$NON-NLS-1$
		}
	}



	/**
	 * Actions to do when a command <code>\newpath</code> is parsed.
	 * @param line The number of the current line.
	 * @since 0.3
	 */
	public abstract void actionsOnNewPath(int line);


	
	/**
	 * Actions to do when the command <code>\lineto</code is parsed.
	 * @param line The number of the current line.
	 * @param params The pstricks parameters.
	 * @param txt The code to parse.
	 * @return The position of the last parsed token.
	 * @throws InvalidFormatCommandException If the format of the command is not valid.
	 * @since 0.3
	 */
	public abstract int actionsOnLineTo(int line, Vector<PSTricksParameters> params, String txt) 
										throws InvalidFormatCommandException;
	
	
	
	/**
	 * Actions to do when the command <code>\curveto</code is parsed.
	 * @param line The number of the current line.
	 * @param params The pstricks parameters.
	 * @param txt The code to parse.
	 * @return The position of the last parsed token.
	 * @throws InvalidFormatCommandException If the format of the command is not valid.
	 * @since 0.3
	 */
	public abstract int actionsOnCurveTo(int line, Vector<PSTricksParameters> params, String txt) 
										throws InvalidFormatCommandException;
	
	
	
	/**
	 * Actions to do when the command <code>\moveto</code is parsed.
	 * @param line The number of the current line.
	 * @param params The pstricks parameters.
	 * @param txt The code to parse.
	 * @return The position of the last parsed token.
	 * @throws InvalidFormatCommandException If the format of the command is not valid.
	 * @since 0.3
	 */
	public abstract int actionsOnMoveTo(int line, Vector<PSTricksParameters> params, String txt) 
										throws InvalidFormatCommandException;


	
	/**
	 * Define actions to do when the pstricks command <code>\closepath</path> is parsed.
	 * @param line The number of the current line.
	 * @since 0.3
	 */
	public abstract void actionsOnClosePathCommand(int line);
	
	
	
	/**
	 * Parses the pstricks commands <code>\lineto</code> and <code>\moveto</code>.
	 * @param nLine The number of the line.
	 * @param txt The code to parse.
	 * @param p1 The point to parse.
	 * @return The position of the last parsed token.
	 * @throws InvalidFormatCommandException If the command is not valid.
	 * @throws IllegalArgumentException If txt or p1 is null.
	 * @since 0.3
	 */
	public int parseParametersLineMoveTo(int nLine, String txt, Point2D.Double p1) throws InvalidFormatCommandException
	{
		if(p1==null || txt==null)
			throw new IllegalArgumentException();
		
		p1.setLocation(readOneCoordinate(txt, 0, nLine, true));
		
		if(Double.isNaN(p1.x)) p1.x = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
		if(Double.isNaN(p1.y)) p1.y = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
			
		return txt.indexOf(')', 1);
	}
	
	
	
	/**
	 * Parses the pstricks command <code>\curveto</code>.
	 * @param nLine The number of the line.
	 * @param txt The code to parse.
	 * @param p1 The first point to parse.
	 * @param p2 The second point to parse.
	 * @param p3 The third point to parse.
	 * @return The position of the last parsed token.
	 * @throws InvalidFormatCommandException If the command is not valid.
	 * @throws IllegalArgumentException If txt or p1 is null.
	 * @since 0.3
	 */
	public int parseParametersCurveTo(int nLine, String txt, Point2D.Double p1, Point2D.Double p2, 
						Point2D.Double p3) throws InvalidFormatCommandException
	{
		int i = 0;
		
		p1.setLocation(readOneCoordinate(txt, 0, nLine, true));
		
		if(Double.isNaN(p1.x)) p1.x = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
		if(Double.isNaN(p1.y)) p1.y = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
			
		i = txt.indexOf(')', 1);
		p2.setLocation(readOneCoordinate(txt, i+1, nLine, true));
		
		if(Double.isNaN(p2.x)) p2.x = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
		if(Double.isNaN(p2.y)) p2.y = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
		
		i = txt.indexOf(')', i+1);
		p3.setLocation(readOneCoordinate(txt, i+1, nLine, true));
		
		if(Double.isNaN(p3.x)) p3.x = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
		if(Double.isNaN(p3.y)) p3.y = PSTricksConstants.DEFAULT_VALUE_MISSING_COORDINATE;
		
		return txt.indexOf(')', i+1);
	}
	
	
	
	/**
	 * Parses the parameters of the <code>rgbcolor</code> command (the name of the command is supposed parsed; for instance the parameters
	 * to parse can be <code>{rgbColourName}{0.1 0.3 0.1}</code>. The parsed colour is added to the <code>DviPsColors</code> container.
	 * @param p The code to parse.
	 * @throws InvalidFormatCommandException If the format of the command is not valid.
	 * @since 2.0.0
	 */
    public static void parseNewrgbcolorParameters(String p) throws InvalidFormatCommandException
	{
		String cmdName = "newrgbcolor";//$NON-NLS-1$
		try
		{
			if(p == null || p.length() < 10)
				throw new IllegalArgumentException(p);

			String colourName;
			double r;
			double g;
			double b;
			int[] i = {0};
			
			if(p.charAt(i[0]) != '{')
				throw new InvalidFormatCommandException(cmdName, -1);
			
			i[0]++;
			colourName = readString(p, i, true);
			goToNextChar(p, i);
			
			if(p.charAt(i[0]) != '}')
				throw new InvalidFormatCommandException(cmdName, -1);
			
			i[0]++;
			goToNextChar(p, i);
			
			if(p.charAt(i[0]) != '{')
				throw new InvalidFormatCommandException(cmdName, -1);
			
			i[0]++;
			r = readReal(p, i, true);
			g = readReal(p, i, true);
			b = readReal(p, i, true);
			goToNextChar(p, i);
			
			if(p.charAt(i[0]) != '}')
				throw new InvalidFormatCommandException(cmdName, -1);
			
			if(r < 0 || r > 1 || g < 0 || g > 1 || b < 0 || b > 1)
				throw new InvalidFormatCommandException(cmdName, -1);

			DviPsColors.addUserColour(new Color((int)(r*255), (int)(g*255), (int)(b*255)), colourName);
			
		}catch(IgnoreCommandException ex) { throw new InvalidFormatCommandException(cmdName, -1); }
	}

	
	
	
	
	/**
	 * Parses the parameters of the <code>hsbcolor</code> command (the name of the command is supposed parsed; for instance the parameters
	 * to parse can be <code>{hsbColourName}{0.1 0.3 0.1}</code>. The parsed colour is added to the <code>DviPsColors</code> container.
	 * @param p The code to parse.
	 * @throws InvalidFormatCommandException If the format of the command is not valid.
	 * @since 2.0.0
	 */
    public static void parseNewhsbcolorParameters(String p) throws InvalidFormatCommandException
	{
		String cmdName = "newhsbcolor";//$NON-NLS-1$
		try
		{
			if(p == null || p.length() < 10)
				throw new IllegalArgumentException(p);

			String colourName;
			double h;
			double s;
			double b;
			int[] i = {0};
			
			if(p.charAt(i[0]) != '{')
				throw new InvalidFormatCommandException(cmdName, -1);
			
			i[0]++;
			colourName = readString(p, i, true);
			goToNextChar(p, i);
			
			if(p.charAt(i[0]) != '}')
				throw new InvalidFormatCommandException(cmdName, -1);
			
			i[0]++;
			goToNextChar(p, i);
			
			if(p.charAt(i[0]) != '{')
				throw new InvalidFormatCommandException(cmdName, -1);
			
			i[0]++;
			h = readReal(p, i, true);
			s = readReal(p, i, true);
			b = readReal(p, i, true);
			goToNextChar(p, i);
			
			if(p.charAt(i[0]) != '}')
				throw new InvalidFormatCommandException(cmdName, -1);
			
			if(h < 0 || h > 1 || s < 0 || s > 1 || b < 0 || b > 1)
				throw new InvalidFormatCommandException(cmdName, -1);

			DviPsColors.addUserColour(Color.getHSBColor((float)h, (float)s, (float)b), colourName);
			
		}catch(IgnoreCommandException ex) { throw new InvalidFormatCommandException(cmdName, -1); }
	}




	/**
	 * Parses the parameters of the <code>cmykcolor</code> command (the name of the command is supposed parsed; for instance the parameters
	 * to parse can be <code>{cmykColourName}{0.1 0.3 0.1 0.8}</code>. 
	 * The parsed colour is added to the <code>DviPsColors</code> container.
	 * @param p The code to parse.
	 * @throws InvalidFormatCommandException If the format of the command is not valid.
	 * @since 2.0.0
	 */
	public static void parseNewcmykcolorParameters(String p) throws InvalidFormatCommandException
	{
		String cmdName = "newcmykcolor";//$NON-NLS-1$
		try
		{
			if(p == null || p.length() < 12)
				throw new IllegalArgumentException(p);

			String colourName;
			double c;
			double m;
			double y;
			double k;
			int[] i = {0};
			
			if(p.charAt(i[0]) != '{')
				throw new InvalidFormatCommandException(cmdName, -1);
			i[0]++;
			colourName = readString(p, i, true);
			goToNextChar(p, i);
			
			if(p.charAt(i[0]) != '}')
				throw new InvalidFormatCommandException(cmdName, -1);
			i[0]++;
			goToNextChar(p, i);
			
			if(p.charAt(i[0]) != '{')
				throw new InvalidFormatCommandException(cmdName, -1);
			i[0]++;
			c = readReal(p, i, true);
			m = readReal(p, i, true);
			y = readReal(p, i, true);
			k = readReal(p, i, true);
			goToNextChar(p, i);
			
			if(p.charAt(i[0]) != '}')
				throw new InvalidFormatCommandException(cmdName, -1);
			
			if(c < 0 || c > 1 || m < 0 || m > 1 || y < 0 || y > 1 || k < 0 || k > 1)
				throw new InvalidFormatCommandException(cmdName, -1);

			DviPsColors.addUserColour(DviPsColors.CMYK2RGB(c, m, y, k), colourName);
			
		}catch(IgnoreCommandException ex) { throw new InvalidFormatCommandException(cmdName, -1); }
	}




	/**
	 * Parses the parameters of the <code>newgray</code> command (the name of the command is supposed parsed; for instance the parameters
	 * to parse can be <code>{grayColourName}{0.1}</code>
	 * @param p The code to parse.
	 * @throws InvalidFormatCommandException If the format of the command is not valid.
	 * @since 2.0.0
	 */
	public static void parseNewgrayParameters(String p) throws InvalidFormatCommandException
	{
		String cmdName = "newgray";//$NON-NLS-1$
		try
		{
			if(p == null || p.length() < 6)
				throw new IllegalArgumentException(p);

			String colourName;
			double g;
			int[] i = {0};
			
			if(p.charAt(i[0]) != '{')
				throw new InvalidFormatCommandException(cmdName, -1);
			i[0]++;
			colourName = readString(p, i, true);
			goToNextChar(p, i);
			
			if(p.charAt(i[0]) != '}')
				throw new InvalidFormatCommandException(cmdName, -1);
			
			i[0]++;
			goToNextChar(p, i);
			
			if(p.charAt(i[0]) != '{')
				throw new InvalidFormatCommandException(cmdName, -1);
			
			i[0]++;
			g = readReal(p, i, true);
			goToNextChar(p, i);
			
			if(p.charAt(i[0]) != '}')
				throw new InvalidFormatCommandException(cmdName, -1);
			
			if(g < 0 || g > 1)
				throw new InvalidFormatCommandException(cmdName, -1);

			DviPsColors.addUserColour(DviPsColors.gray2RBG(g), colourName);
			
		}catch(IgnoreCommandException ex) { throw new InvalidFormatCommandException(cmdName, -1); }
	}

	
	
	
	/**
	 * Checks if the given character is a character to ignore (' ', '\t', '\n', '\r').
	 * @param c The character to check.
	 * @return True if the given character must be ignored.
	 * @since 2.0.0
	 */
	public static boolean isCharToIgnore(char c)
	{
		return c==' ' || c==9 || c==10 || c==13;
	}



	
	/**
	 * Parses the command <code>rotateBox{angle}{..}</code>. In fact it parses only {angle}{, the name of the command is supposed parsed and the 
	 * first character of <code>cmd</code> must be {.
	 * @param cmd The command to parse.
	 * @param params The PSTricks stack of parameters.
	 * @throws InvalidFormatCommandException If the command is not valid.
	 * @since 2.0.0
	 */
	public static void parseRotateBoxCommand(String cmd, Vector<PSTricksParameters> params) throws InvalidFormatCommandException
	{
		String cmdName = "rotateBox";//$NON-NLS-1$
		int i[] = {0};
		
		if(cmd==null || cmd.length()==0)
			throw new InvalidFormatCommandException(cmdName, -1);
		
		if(params==null)
			throw new IllegalArgumentException();
		
		if(cmd.charAt(i[0])!='{')
			throw new InvalidFormatCommandException(cmdName, -1);
		
		i[0]++;
		try
		{
			double angle = readReal(cmd, i, true);
			goToNextChar(cmd, i);
			
			if(cmd.charAt(i[0])!='}')
				throw new InvalidFormatCommandException(cmdName, -1);
			
			i[0]++;
			goToNextChar(cmd, i);
			
			if(cmd.charAt(i[0])!='{')
				throw new InvalidFormatCommandException(cmdName, -1);
			
			if(params.isEmpty())
				params.add(new PSTricksParameters());
			else
				params.add(new PSTricksParameters(params.lastElement()));
			
			params.lastElement().setPosition(0, 0, true, angle, 0., 0, 0., 0.);
		}
		catch(IgnoreCommandException e) { throw new InvalidFormatCommandException(cmdName, -1); }
	}
	
	
	
	
	/**
	 * Goes to the next occurrence of the given character and jumps it.
	 * @param str The code to parse.
	 * @param id input: the position of the first character to analyse. Output: the position of the character just after the found character.
	 * @throws IgnoreCommandException If the given character is unreachable or if a character (not a ignorable character) is 
	 * reached before a comma.
	 */
	public static void readChar(String str, char c, int[] id) throws IgnoreCommandException
	{
		try
		{
			if(isCharToIgnore(c))
				throw new IllegalArgumentException();
			
			if(id==null || id.length<1) 
				throw new IgnoreCommandException(-1);
			
			int size = str.length();
			
			while(id[0]<size && isCharToIgnore(str.charAt(id[0])) && str.charAt(id[0])!=c) 
				id[0]++;
			
			if(id[0]>=size || str.charAt(id[0])!=c) throw new IgnoreCommandException(-1);

			id[0]++;
			
		}catch(Exception e)
		{
			e.printStackTrace();
			throw new IgnoreCommandException(-1);
		}
	}



	
	public static void parseputCommand(String str, Vector<PSTricksParameters> params) throws InvalidFormatCommandException, IgnoreCommandException
	{
		if(str==null || str.length()==0 || params==null) throw new InvalidFormatCommandException("put", -1);//$NON-NLS-1$
		
		int i[] = {0};
		int size = str.length();
		
		if(i[0]<size && str.charAt(i[0])=='(')
		{
			Point2D.Double coord = readOneCoordinate(str, i[0], -1, true);
			PSTricksParameters p = new PSTricksParameters(params.lastElement());
			
			if(coord==null) throw new InvalidFormatCommandException("put", -1);//$NON-NLS-1$
			
			while(i[0]<size && str.charAt(i[0])!=')') i[0]++;
			
			if(i[0]>=size) throw new IgnoreCommandException(-1);
			
			i[0]++;
			
			if(i[0]<size) goToNextChar(str, i);

			p.setPosition(coord.x, coord.y, true, 0, 0, PositionParameters.REF_POINT_LEFT+PositionParameters.REF_POINT_BASE,
					0, PositionParameters.REF_POINT_DEFAULT_CENTER);

			if(params.size()>0)
			{
				PSTricksParameters p2 = params.lastElement();
				p.origin.setLocation(p2.origin.x, p2.origin.y);
			}
			
			params.add(p);
		}
	}
}
