// license:BSD-3-Clause
// copyright-holders:

/***********************************************************************************************************************************

Skeleton driver for Secoinsa Serie 20 minicomputer.

Secoinsa (Sociedad Española de Comunicaciones e Informática, S.A.) was a company founded in 1975 by Fujitsu, the Spanish
National Institute for Industry (Instituto Nacional de Industria, INI), Telefónica, and other investors (mainly banks).
It primarily manufactured and sold FACOM clones for the Spanish market.

In 1986, the company was split and sold. Telefónica acquired the professional services division, while Fujitsu España
took the hardware and software areas.

Key features for Secoinsa Serie 20:
    -CPU: Z80B.
    -RAM: 80 Kbytes, expandable to 272 Kbytes.
    -Peripherals: Terminal interface, RS-232 serial port for printer, two sync/async communication channels.
    -Screen: Green phosphor with 24 lines of 80 characters, with an option for NSP terminal with 12/25 lines of 69/80/132 characters.
    -Keyboard: QWERTY, 102 keys.
    -Mass storage: One or two 8" floppy disk drives of 640 Kbytes or 1.2 Mbytes. Winchester hard drives of 10 or 20 Mbytes.
    -Operating System: OASIS.
    -Programming languages: BASIC (interpreter and compiler), Spanish EXEC.

The dumped Secoinsa Serie 20 was found with the following PCBs:

PCB SECOINSA SM16B-0200-0170
     _______________________________________________________________
    |    __________    __________    __________       __________   |
    |   |SN74LS30N|   |SN74LS112AN  |_PAL_3U__|  ::  |SN74LS125AN  |
    |    __________    __________    __________  ::   __________   |
    |   |SN74LS240N   |SN74LS74AN   |SN74LS125AN ::  |MC 1489AL|  _|_
    |                                __________                  |   |
    |    __________    __________   |_SN7417N_|  ::   __________ |   |
  __|   |SN74LS374N   |SN74LS112AN   __________  ::  |MC 1489AL| |   |
 |__                                |_SN7416N_|  ::              |   |
 |__     __________    __________    __________       __________ |   |
 |__ ·· |SN74LS374N   |_74F138__|   |SN74LS74AN  ::  |SN75188J_| |   |
 |__ ··  __________    __________    __________  ::   __________ |   |
 |__ ·· |SN74LS374N   |_74F138__|   |SN74LS08N|  ::  |_DS3486N_| |   |
 |__ ··  __________    __________    __________       __________ |   |
 |__ ·· |SN74LS374N   |_PAL_2P__|   |SN74LS74AN ·· : |SN75188J_| |   |
 |__                                ··········     :             |   |
 |__     __________      ____________________         __________ |   |
 |__    |SN74LS240N     | AMD Z8530PC       |        |MC_1489AL| |___|
 |__                    |___________________|         __________   |
 |__     __________         ··  ::::::::::           |_MC3487L_|   |
 |__    |SN74LS244N      ____________________         __________   |
 |__                    | AMD P8237A-5      |        |SN74LS112AN  |
 |__     __________     |___________________|         __________   |
 |__    |SN74LS244N             ::::::::::           |SN74LS112AN  |
 |__                     ____________________         __________   |
 |__     __________     | AMD P8237A-5      |        |_DS3486N_|   |
 |__    |SN74LS245N     |___________________|         __________   |
 |__                            :::..:::::           |MC_1489AL|   |
 |__     __________      ____________________         __________  _|_
 |__    |SN74LS373N     | AMD Z8530PC       |        |_MC3487L_| |   |
 |__     __________     |___________________|         __________ |   |
 |__    |SN74LS245N                               :  |MC 1489AL| |   |
 |__     __________    __________    __________   :   __________ |   |
 |__    |SN74LS244N   SM25LS2521PC  |SN74LS74AN   :  |SN75188J_| |   |
 |__     __________    __________    __________  ::   __________ |   |
 |__    |SN74LS74AN   |SN74LS112AN  |SN74LS112AN  :  |_MC3486P_| |   |
    |    __________    __________    __________   :   __________ |   |
    |   |_PAL_1D__| ..|SN74LS74AN   |SN74LS09N|      |SN75188J_| |   |
    |    __________ .. __________                ..              |   |
    |   |_PAL_1C__| ..|SN74LS74AN :  __________  ..   __________ |   |
    |    __________ .. __________   |_SN7416N_|  ..  |MC_1489AL| |   |
    |   |_PAL_1B__| ..|SN74LS08N|                                |___|
    |    __________    __________    __________       __________   |
    |   |_SN7417N_|   |SN74LS125AN  |_SN7417N_|      |MC_1489AL|   |
    |    __________    __________    __________       __________   |
    |   |SN74LS165N   |SN74LS165N   |SN74LS74AN      |MC14040BCP   |
    |______________________________________________________________|



PCB SECOINSA SM16B-0200-0140:
 -Floppy controller with FDC9229BT, NEC D765AC.
 -Serial communications with Z8530DC, Signetics 2651.
 -RTC with MM58167AN.
     _______________________________________________________________
    |    __________     __________    __________       __________  |
    |__ |SN74LS165AN   |MC14538BCP   |SN74LS74AN      |_PAL_1A__|  |
    |__| __________     __________    __________       __________  |
    |__|| DIPS x 8| .. |SN74LS139AN  |SN74LS74AN      |_DM7416N_|  |
    |    __________ ··  __________    __________       __________  |
    |__ |_DM7416N_| ·· |_SN7417N_|   |SN74LS74AN  :   AM25LS252IPC |
    |__| ________________________                                  |
    |__|| NEC D765AC            |     __________       __________  |___
    |__||                       |    |SN74LS04N|      |DM74LS244N   __|
    |__||_______________________|                                   __|
    |__| __________     __________    __________       __________   __|
    |__||DM74LS244N    |SN74LS139AN  |SN74LS240N      |SN74LS245N   __|
    |__|                    .. :                                    __|
    |__| __________                   __________       __________   __|
    |__||SN74LS240N     ::::::       |SN74LS245N      |SN74LS46INS  __|
    |__| __________     __________    __________       __________   __|
    |   |_SN7417N_|    |FDC9229BT|   |_PAL_2G__|      |SN74LS46INS  __|
    |__  ________________             __________       __________   __|
    |__||S 8212 2651 N  |            |74SC374P_|      |_PAL_1H__|   __|
    |__||_______________|             __________       __________   __|
    |__| ________________________    |_SN7417N_|      |DM74LS244N   __|
    |__||AMD Z8530DC            |     __________       __________   __|
    |__||                       |    |SN74LS240N      |_PAL_1K__|   __|
    |__||_______________________|                      __________   __|
    |__| __________     __________    __________      |SN74LS46INS  __|
    |__||SN75189AJ|    | DIPS x 8|   |SN74LS148N        :::::::::   __|
    |__| __________ ..  __________    __________      ___________   __|
    |__||_SN7417N_| ·· |_DM7416N_|   |_PAL_2M__|     | P8254    |   __|
    |    __________ ..  __________    __________     |__________|   __|
    |__ |_DM7416N_| .. |SN74LS74AN   |SN74LS125AN      __________   __|
    |__|            ..    Xtal          Xtal          |_PAL_1N__|  |
    |__|                 16 MHz       4.9152 MHz      ___________  |
    |    __________     __________    __________     |MM58167AN |  |
    |   |SN74LS74AN    |SN74LS74AN   |SN74LS374N     |__________|  |
    |    __________     __________    __________                   |
    |   |SN74S74N_|    |SN74LS165AN  |SN75188J_|                   |
    |______________________________________________________________|



CPU PCB
     _______________________________________________________________
    |   ___                          __________       _________    |
    |  |  |                         |SN74LS74AN      |74LS08N_|    |
    |  |EMPTY                        __________       __________   |
    |  |__|                         |_74LS04N_|      |SN74LS373N   |
    |                                                              |
    |   __________       ______________________       __________   |
    |  |SN74LS21N|      | ZILOG Z8400B PS     |      |SN74LS373N   |
    |                   | Z80B CPU            |                    |
    |   __________      |_____________________|       __________   |
    |  |DIPS x8  |                                   |DM81LS95N|   |___
    |   __________     __________    __________       __________    __|
    |  |SN74LS00N|    |SN74LS373N   |_74LS04N_|      |SN74LS244N    __|
    |   __________   _____________   __________       __________    __|
    |  |SL74LS11N|  | EPROM      |  |DIPS x 8 |      |SN74LS244N    __|
    |   __________  |____________|   __________       __________    __|
    |  |SN74LS74AN   _____________  |DM74LS266N      |SN74LS244N    __|
    |   __________  | EMPTY      |   __________       __________    __|
    |  |SN74LS165AN |____________|  |_PAL_2H__|      |SN74244N_|    __|
    |   __________     __________    __________       __________    __|
    |  |SN74LS00N|    |_74LS04N_|   |_74LS02N_| Xtal |_74LS08N_|    __|
    |   __________     __________    __________ 4MHz  __________    __|
   _|_ |SN74LS10N|    |_74LS02N_|   |_74LS04N_|      |SN74LS367AN   __|
  |   | __________     __________    __________       __________    __|
  |   ||DM74LS161AN   |_74LS04N_|   |SN74LS125AN     |SN74LS367AN   __|
  |   | __________     __________    __________       __________    __|
  |   ||74LS157N_|    |_74LS33N_|   |SN74LS273N      |SN74LS244N    __|
  |   | __________     __________    __________       __________    __|
  |   ||74LS74AN_|    |SN74LS10N|   |SN74LS273N      |_74LS32N_|    __|
  |   | __________     __________    __________       __________    __|
  |   ||_74LS04N_|    |DM74LS245N   |SN74LS148N      |_74LS32N_|    __|
  |___|                __________                                  |
    |     Xtal        |SN74LS244N                                  |
    |    20 MHz                                                    |
    |______________________________________________________________|



PCB SECOINSA SM16B-0200-0010
     _______________________________________________________________
    |                                               __________     |
    |                                __________    SN74LS165AN     |
    |                               |SN74LS157N     __________     |
    |                                __________    |SN74LS04N|     |
    |                            ·· |SN74LS393N     __________     |
   _|_                           ··  __________    |SN74LS32N|     |
  |   |                             |SN74LS393N     __________     |
  |   |                              __________    |SN74LS74N|     |
  |   |                             |74LS138N_|     __________     |
  |   |                                            |LS74LS74AN     |___
  |   |                              __________     __________      __|
  |   |                             |DM74LS374N    |SN74LS240N      __|
  |   |                              __________     __________      __|
  |___|                             |SN74LS245N    |SN74LS240N      __|
    |                                               __________      __|
    |                                              AM25LS2521PC     __|
    |                                __________     __________      __|
    |                               |SN74LS74AN    |SN74LS273N      __|
    |                                __________     __________      __|
    |                               |SN74LS155N    |SN74LS244N      __|
    |                         ··     __________     __________      __|
    |         _____________  ···    |PAL16L8 2K    AM25LS2521PC     __|
    |        |AM9128-15PC |    .     __________     __________  ..  __|
   _|_       |____________|    :    |PAL16L8 2L    |_DIPSx8__|  ..  __|
  |   |       _____________          __________     __________  ..  __|
  |   |      |AM9128-15PC |         |PAL16L8 2M    AM25LS2521PC ..  __|
  |   |      |____________|          __________     __________  ..  __|
  |   |       __________            |SN74LS273N    |_DIPSx8__|  ..  __|
  |   |      |NS74LS374N             __________     __________      __|
  |   |       __________            |SN74LS280N    |SN74LS139AN     __|
  |   |      |SN74LS245N             __________     __________      __|
  |___|       __________            |SN74LS273N    |SN74LS74AN     |
    |        |SN74LS240N                        ··                 |
    |         __________             __________ ··  __________     |
    |        |SN74LS393N            |SN74LS374N ·· |SN74LS74AN     |
    |                                           ··                 |
    |                                           ··                 |
    |______________________________________________________________|


Six RAM PCBs:

RAM PCB 1 (8 x HM6116P-3)
     _______________________________________________________________
    |  _______   _______   _______   _______   _______             |
    | |      |  |      |  |      |  |      |  |      |  ___   ___  |
    | |      |  |      |  |      |  |      |  |      | |  |  |  |  |
    | |HM    |  |HM    |  |HM    |  |EMPTY |  |EMPTY | | <-74LS138N|
    | |6116P |  |6116P |  |6116P |  |      |  |      | |  |  | <-74LS138N
    | |3     |  |3     |  |3     |  |      |  |      | |  |  |  |  |
    | |______|  |______|  |______|  |______|  |______| |__|  |__|  |
    |  _______   _______   _______   _______   _______             |
    | |      |  |      |  |      |  |      |  |      |             |
    | |      |  |      |  |      |  |      |  |      |  _________  |___
    | |HM    |  |HM    |  |HM    |  |EMPTY |  |EMPTY | |SN74LS10N   __|
    | |6116P |  |6116P |  |6116P |  |      |  |      |              __|
    | |3     |  |3     |  |3     |  |      |  |      |  __________  __|
    | |______|  |______|  |______|  |______|  |______| |SN74LS244N  __|
    |  _______   _______   _______   _______   _______              __|
    | |      |  |      |  |      |  |      |  |      |  __________  __|
    | |      |  |      |  |      |  |      |  |      | |SN74LS244N  __|
    | |HM    |  |HM    |  |EMPTY |  |EMPTY |  |EMPTY |              __|
    | |6116P |  |6116P |  |      |  |      |  |      |  __________  __|
    | |3     |  |3     |  |      |  |      |  |      | |SN74LS244N  __|
    | |______|  |______|  |______|  |______|  |______|              __|
    |  _______   _______   _______   _______   _______  __________  __|
    | |      |  |      |  |      |  |      |  |      | |SN74LS244N  __|
    | |      |  |      |  |      |  |      |  |      |              __|
    | |EMPTY |  |EMPTY |  |EMPTY |  |EMPTY |  |EMPTY |  _________   __|
    | |      |  |      |  |      |  |      |  |      | |74LS138N|   __|
    | |      |  |      |  |      |  |      |  |      |              __|
    | |______|  |______|  |______|  |______|  |______|  _________   __|
    |  _______   _______   _______   _______   _______ |SN74LS08N   __|
    | |      |  |      |  |      |  |      |  |      |         :::  __|
    | |      |  |      |  |      |  |      |  |      |  __________  __|
    | |EMPTY |  |EMPTY |  |EMPTY |  |EMPTY |  |EMPTY | AM25LS252IPC __|
    | |      |  |      |  |      |  |      |  |      |              __|
    | |      |  |      |  |      |  |      |  |      |  _________   __|
    | |______|  |______|  |______|  |______|  |______| SN74ALS133N  __|
    |  _______   _______   _______   _______   _______              __|
    | |      |  |      |  |      |  |      |  |      |              __|
    | |      |  |      |  |      |  |      |  |      |  __________  __|
    | |EMPTY |  |EMPTY |  |EMPTY |  |EMPTY |  |EMPTY | |DIPS x 8 |  __|
    | |      |  |      |  |      |  |      |  |      |              __|
    | |      |  |      |  |      |  |      |  |      |             |
    | |______|  |______|  |______|  |______|  |______|  ___   ___  |
    |                                _______   _______ |  |  |  |  |
    |                               |      |  |      | | <-74LS138N|
    |                               |      |  |      | |  |  | <-74LS138N
    |                               |EMPTY |  |EMPTY | |  |  |  |  |
    |                               |      |  |      | |__|  |__|  |
    |                               |      |  |      |             |
    |                               |______|  |______|             |
    |______________________________________________________________|

RAM PCB 2 (24 x HM6116P-3)
     _______________________________________________________________
    |  _______   _______   _______   _______   _______             |
    | |      |  |      |  |      |  |      |  |      |  ___   ___  |
    | |      |  |      |  |      |  |      |  |      | |  |  |  |  |
    | |HM    |  |HM    |  |HM    |  |EMPTY |  |EMPTY | | <-74LS138N|
    | |6116P |  |6116P |  |6116P |  |      |  |      | |  |  | <-74LS138N
    | |3     |  |3     |  |3     |  |      |  |      | |  |  |  |  |
    | |______|  |______|  |______|  |______|  |______| |__|  |__|  |
    |  _______   _______   _______   _______   _______             |
    | |      |  |      |  |      |  |      |  |      |             |
    | |      |  |      |  |      |  |      |  |      |  _________  |___
    | |HM    |  |HM    |  |HM    |  |EMPTY |  |EMPTY | |SN74LS10N   __|
    | |6116P |  |6116P |  |6116P |  |      |  |      |              __|
    | |3     |  |3     |  |3     |  |      |  |      |  __________  __|
    | |______|  |______|  |______|  |______|  |______| |SN74LS244N  __|
    |  _______   _______   _______   _______   _______              __|
    | |      |  |      |  |      |  |      |  |      |  __________  __|
    | |      |  |      |  |      |  |      |  |      | |SN74LS244N  __|
    | |HM    |  |HM    |  |EMPTY |  |EMPTY |  |EMPTY |              __|
    | |6116P |  |6116P |  |      |  |      |  |      |  __________  __|
    | |3     |  |3     |  |      |  |      |  |      | |SN74LS244N  __|
    | |______|  |______|  |______|  |______|  |______|              __|
    |  _______   _______   _______   _______   _______  __________  __|
    | |      |  |      |  |      |  |      |  |      | |SN74LS244N  __|
    | |      |  |      |  |      |  |      |  |      |              __|
    | |HM    |  |HM    |  |HM    |  |HM    |  |EMPTY |  _________   __|
    | |6116P |  |6116P |  |6116P |  |6116P |  |      | |74LS138N|   __|
    | |3     |  |3     |  |3     |  |3     |  |      |              __|
    | |______|  |______|  |______|  |______|  |______|  _________   __|
    |  _______   _______   _______   _______   _______ |SN74LS08N   __|
    | |      |  |      |  |      |  |      |  |      |         :::  __|
    | |      |  |      |  |      |  |      |  |      |  __________  __|
    | |HM    |  |HM    |  |HM    |  |HM    |  |HM    | AM25LS252IPC __|
    | |6116P |  |6116P |  |6116P |  |6116P |  |6116P |              __|
    | |3     |  |3     |  |3     |  |3     |  |3     |  _________   __|
    | |______|  |______|  |______|  |______|  |______| SN74ALS133N  __|
    |  _______   _______   _______   _______   _______              __|
    | |      |  |      |  |      |  |      |  |      |              __|
    | |      |  |      |  |      |  |      |  |      |  __________  __|
    | |HM    |  |HM    |  |HM    |  |HM    |  |HM    | |DIPS x 8 |  __|
    | |6116P |  |6116P |  |6116P |  |6116P |  |6116P |              __|
    | |3     |  |3     |  |3     |  |3     |  |3     |             |
    | |______|  |______|  |______|  |______|  |______|  ___   ___  |
    |                                _______   _______ |  |  |  |  |
    |                               |      |  |      | | <-74LS138N|
    |                               |      |  |      | |  |  | <-74LS138N
    |                               |HM    |  |HM    | |  |  |  |  |
    |                               |6116P |  |6116P | |__|  |__|  |
    |                               |3     |  |3     |             |
    |                               |______|  |______|             |
    |______________________________________________________________|

RAM PCB 3 (24 x HM6116P-3)
     _______________________________________________________________
    |  _______   _______   _______   _______   _______             |
    | |      |  |      |  |      |  |      |  |      |  ___   ___  |
    | |      |  |      |  |      |  |      |  |      | |  |  |  |  |
    | |HM    |  |HM    |  |HM    |  |EMPTY |  |EMPTY | | <-74LS138N|
    | |6116P |  |6116P |  |6116P |  |      |  |      | |  |  | <-74LS138N
    | |3     |  |3     |  |3     |  |      |  |      | |  |  |  |  |
    | |______|  |______|  |______|  |______|  |______| |__|  |__|  |
    |  _______   _______   _______   _______   _______             |
    | |      |  |      |  |      |  |      |  |      |             |
    | |      |  |      |  |      |  |      |  |      |  _________  |___
    | |HM    |  |HM    |  |HM    |  |EMPTY |  |EMPTY | |SN74LS10N   __|
    | |6116P |  |6116P |  |6116P |  |      |  |      |              __|
    | |3     |  |3     |  |3     |  |      |  |      |  __________  __|
    | |______|  |______|  |______|  |______|  |______| |SN74LS244N  __|
    |  _______   _______   _______   _______   _______              __|
    | |      |  |      |  |      |  |      |  |      |  __________  __|
    | |      |  |      |  |      |  |      |  |      | |SN74LS244N  __|
    | |HM    |  |HM    |  |EMPTY |  |EMPTY |  |EMPTY |              __|
    | |6116P |  |6116P |  |      |  |      |  |      |  __________  __|
    | |3     |  |3     |  |      |  |      |  |      | |SN74LS244N  __|
    | |______|  |______|  |______|  |______|  |______|              __|
    |  _______   _______   _______   _______   _______  __________  __|
    | |      |  |      |  |      |  |      |  |      | |SN74LS244N  __|
    | |      |  |      |  |      |  |      |  |      |              __|
    | |HM    |  |HM    |  |HM    |  |HM    |  |EMPTY |  _________   __|
    | |6116P |  |6116P |  |6116P |  |6116P |  |      | |74LS138N|   __|
    | |3     |  |3     |  |3     |  |3     |  |      |              __|
    | |______|  |______|  |______|  |______|  |______|  _________   __|
    |  _______   _______   _______   _______   _______ |SN74LS08N   __|
    | |      |  |      |  |      |  |      |  |      |         :::  __|
    | |      |  |      |  |      |  |      |  |      |  __________  __|
    | |HM    |  |HM    |  |HM    |  |HM    |  |HM    | AM25LS252IPC __|
    | |6116P |  |6116P |  |6116P |  |6116P |  |6116P |              __|
    | |3     |  |3     |  |3     |  |3     |  |3     |  _________   __|
    | |______|  |______|  |______|  |______|  |______| SN74ALS133N  __|
    |  _______   _______   _______   _______   _______              __|
    | |      |  |      |  |      |  |      |  |      |              __|
    | |      |  |      |  |      |  |      |  |      |  __________  __|
    | |HM    |  |HM    |  |HM    |  |HM    |  |HM    | |DIPS x 8 |  __|
    | |6116P |  |6116P |  |6116P |  |6116P |  |6116P |              __|
    | |3     |  |3     |  |3     |  |3     |  |3     |             |
    | |______|  |______|  |______|  |______|  |______|  ___   ___  |
    |                                _______   _______ |  |  |  |  |
    |                               |      |  |      | | <-74LS138N|
    |                               |      |  |      | |  |  | <-74LS138N
    |                               |HM    |  |HM    | |  |  |  |  |
    |                               |6116P |  |6116P | |__|  |__|  |
    |                               |3     |  |3     |             |
    |                               |______|  |______|             |
    |______________________________________________________________|

RAM PCB 4 (24 x HM6116P-3)
     _______________________________________________________________
    |  _______   _______   _______   _______   _______             |
    | |      |  |      |  |      |  |      |  |      |  ___   ___  |
    | |      |  |      |  |      |  |      |  |      | |  |  |  |  |
    | |HM    |  |HM    |  |HM    |  |EMPTY |  |EMPTY | | <-74LS138N|
    | |6116P |  |6116P |  |6116P |  |      |  |      | |  |  | <-74LS138N
    | |3     |  |3     |  |3     |  |      |  |      | |  |  |  |  |
    | |______|  |______|  |______|  |______|  |______| |__|  |__|  |
    |  _______   _______   _______   _______   _______             |
    | |      |  |      |  |      |  |      |  |      |             |
    | |      |  |      |  |      |  |      |  |      |  _________  |___
    | |HM    |  |HM    |  |HM    |  |EMPTY |  |EMPTY | |SN74LS10N   __|
    | |6116P |  |6116P |  |6116P |  |      |  |      |              __|
    | |3     |  |3     |  |3     |  |      |  |      |  __________  __|
    | |______|  |______|  |______|  |______|  |______| |SN74LS244N  __|
    |  _______   _______   _______   _______   _______              __|
    | |      |  |      |  |      |  |      |  |      |  __________  __|
    | |      |  |      |  |      |  |      |  |      | |SN74LS244N  __|
    | |HM    |  |HM    |  |EMPTY |  |EMPTY |  |EMPTY |              __|
    | |6116P |  |6116P |  |      |  |      |  |      |  __________  __|
    | |3     |  |3     |  |      |  |      |  |      | |SN74LS244N  __|
    | |______|  |______|  |______|  |______|  |______|              __|
    |  _______   _______   _______   _______   _______  __________  __|
    | |      |  |      |  |      |  |      |  |      | |SN74LS244N  __|
    | |      |  |      |  |      |  |      |  |      |              __|
    | |HM    |  |HM    |  |HM    |  |HM    |  |EMPTY |  _________   __|
    | |6116P |  |6116P |  |6116P |  |6116P |  |      | |74LS138N|   __|
    | |3     |  |3     |  |3     |  |3     |  |      |              __|
    | |______|  |______|  |______|  |______|  |______|  _________   __|
    |  _______   _______   _______   _______   _______ |SN74LS08N   __|
    | |      |  |      |  |      |  |      |  |      |         :::  __|
    | |      |  |      |  |      |  |      |  |      |  __________  __|
    | |HM    |  |HM    |  |HM    |  |HM    |  |HM    | AM25LS252IPC __|
    | |6116P |  |6116P |  |6116P |  |6116P |  |6116P |              __|
    | |3     |  |3     |  |3     |  |3     |  |3     |  _________   __|
    | |______|  |______|  |______|  |______|  |______| SN74ALS133N  __|
    |  _______   _______   _______   _______   _______              __|
    | |      |  |      |  |      |  |      |  |      |              __|
    | |      |  |      |  |      |  |      |  |      |  __________  __|
    | |HM    |  |HM    |  |HM    |  |HM    |  |HM    | |DIPS x 8 |  __|
    | |6116P |  |6116P |  |6116P |  |6116P |  |6116P |              __|
    | |3     |  |3     |  |3     |  |3     |  |3     |             |
    | |______|  |______|  |______|  |______|  |______|  ___   ___  |
    |                                _______   _______ |  |  |  |  |
    |                               |      |  |      | | <-74LS138N|
    |                               |      |  |      | |  |  | <-74LS138N
    |                               |HM    |  |HM    | |  |  |  |  |
    |                               |6116P |  |6116P | |__|  |__|  |
    |                               |3     |  |3     |             |
    |                               |______|  |______|             |
    |______________________________________________________________|

RAM PCB 5 (32 x HM6116P-3)
     _______________________________________________________________
    |  _______   _______   _______   _______   _______             |
    | |      |  |      |  |      |  |      |  |      |  ___   ___  |
    | |      |  |      |  |      |  |      |  |      | |  |  |  |  |
    | |HM    |  |HM    |  |HM    |  |HM    |  |HM    | | <-74LS138N|
    | |6116P |  |6116P |  |6116P |  |6116P |  |6116P | |  |  | <-74LS138N
    | |3     |  |3     |  |3     |  |3     |  |3     | |  |  |  |  |
    | |______|  |______|  |______|  |______|  |______| |__|  |__|  |
    |  _______   _______   _______   _______   _______             |
    | |      |  |      |  |      |  |      |  |      |             |
    | |      |  |      |  |      |  |      |  |      |  _________  |___
    | |HM    |  |HM    |  |HM    |  |HM    |  |HM    | |SN74LS10N   __|
    | |6116P |  |6116P |  |6116P |  |6116P |  |6116P |              __|
    | |3     |  |3     |  |3     |  |3     |  |3     |  __________  __|
    | |______|  |______|  |______|  |______|  |______| |SN74LS244N  __|
    |  _______   _______   _______   _______   _______              __|
    | |      |  |      |  |      |  |      |  |      |  __________  __|
    | |      |  |      |  |      |  |      |  |      | |SN74LS244N  __|
    | |HM    |  |HM    |  |EMPTY |  |EMPTY |  |EMPTY |              __|
    | |6116P |  |6116P |  |6116P |  |6116P |  |6116P |  __________  __|
    | |3     |  |3     |  |3     |  |3     |  |3     | |SN74LS244N  __|
    | |______|  |______|  |______|  |______|  |______|              __|
    |  _______   _______   _______   _______   _______  __________  __|
    | |      |  |      |  |      |  |      |  |      | |SN74LS244N  __|
    | |      |  |      |  |      |  |      |  |      |              __|
    | |HM    |  |HM    |  |HM    |  |HM    |  |EMPTY |  _________   __|
    | |6116P |  |6116P |  |6116P |  |6116P |  |6116P | |74LS138N|   __|
    | |3     |  |3     |  |3     |  |3     |  |3     |              __|
    | |______|  |______|  |______|  |______|  |______|  _________   __|
    |  _______   _______   _______   _______   _______ |SN74LS08N   __|
    | |      |  |      |  |      |  |      |  |      |         :::  __|
    | |      |  |      |  |      |  |      |  |      |  __________  __|
    | |HM    |  |HM    |  |HM    |  |HM    |  |HM    | AM25LS252IPC __|
    | |6116P |  |6116P |  |6116P |  |6116P |  |6116P |              __|
    | |3     |  |3     |  |3     |  |3     |  |3     |  _________   __|
    | |______|  |______|  |______|  |______|  |______| SN74ALS133N  __|
    |  _______   _______   _______   _______   _______              __|
    | |      |  |      |  |      |  |      |  |      |              __|
    | |      |  |      |  |      |  |      |  |      |  __________  __|
    | |HM    |  |HM    |  |HM    |  |HM    |  |HM    | |DIPS x 8 |  __|
    | |6116P |  |6116P |  |6116P |  |6116P |  |6116P |              __|
    | |3     |  |3     |  |3     |  |3     |  |3     |             |
    | |______|  |______|  |______|  |______|  |______|  ___   ___  |
    |                                _______   _______ |  |  |  |  |
    |                               |      |  |      | | <-74LS138N|
    |                               |      |  |      | |  |  | <-74LS138N
    |                               |HM    |  |HM    | |  |  |  |  |
    |                               |6116P |  |6116P | |__|  |__|  |
    |                               |3     |  |3     |             |
    |                               |______|  |______|             |
    |______________________________________________________________|

RAM PCB 5 (24 x HM6116P-3)
     _______________________________________________________________
    |  _______   _______   _______   _______   _______             |
    | |      |  |      |  |      |  |      |  |      |  ___   ___  |
    | |      |  |      |  |      |  |      |  |      | |  |  |  |  |
    | |HM    |  |HM    |  |HM    |  |EMPTY |  |EMPTY | | <-74LS138N|
    | |6116P |  |6116P |  |6116P |  |      |  |      | |  |  | <-74LS138N
    | |3     |  |3     |  |3     |  |      |  |      | |  |  |  |  |
    | |______|  |______|  |______|  |______|  |______| |__|  |__|  |
    |  _______   _______   _______   _______   _______             |
    | |      |  |      |  |      |  |      |  |      |             |
    | |      |  |      |  |      |  |      |  |      |  _________  |___
    | |HM    |  |HM    |  |HM    |  |EMPTY |  |EMPTY | |SN74LS10N   __|
    | |6116P |  |6116P |  |6116P |  |      |  |      |              __|
    | |3     |  |3     |  |3     |  |      |  |      |  __________  __|
    | |______|  |______|  |______|  |______|  |______| |SN74LS244N  __|
    |  _______   _______   _______   _______   _______              __|
    | |      |  |      |  |      |  |      |  |      |  __________  __|
    | |      |  |      |  |      |  |      |  |      | |SN74LS244N  __|
    | |HM    |  |HM    |  |EMPTY |  |EMPTY |  |EMPTY |              __|
    | |6116P |  |6116P |  |      |  |      |  |      |  __________  __|
    | |3     |  |3     |  |      |  |      |  |      | |SN74LS244N  __|
    | |______|  |______|  |______|  |______|  |______|              __|
    |  _______   _______   _______   _______   _______  __________  __|
    | |      |  |      |  |      |  |      |  |      | |SN74LS244N  __|
    | |      |  |      |  |      |  |      |  |      |              __|
    | |HM    |  |HM    |  |HM    |  |HM    |  |EMPTY |  _________   __|
    | |6116P |  |6116P |  |6116P |  |6116P |  |      | |74LS138N|   __|
    | |3     |  |3     |  |3     |  |3     |  |      |              __|
    | |______|  |______|  |______|  |______|  |______|  _________   __|
    |  _______   _______   _______   _______   _______ |SN74LS08N   __|
    | |      |  |      |  |      |  |      |  |      |         :::  __|
    | |      |  |      |  |      |  |      |  |      |  __________  __|
    | |HM    |  |HM    |  |HM    |  |HM    |  |HM    | AM25LS252IPC __|
    | |6116P |  |6116P |  |6116P |  |6116P |  |6116P |              __|
    | |3     |  |3     |  |3     |  |3     |  |3     |  _________   __|
    | |______|  |______|  |______|  |______|  |______| SN74ALS133N  __|
    |  _______   _______   _______   _______   _______              __|
    | |      |  |      |  |      |  |      |  |      |              __|
    | |      |  |      |  |      |  |      |  |      |  __________  __|
    | |HM    |  |HM    |  |HM    |  |HM    |  |HM    | |DIPS x 8 |  __|
    | |6116P |  |6116P |  |6116P |  |6116P |  |6116P |              __|
    | |3     |  |3     |  |3     |  |3     |  |3     |             |
    | |______|  |______|  |______|  |______|  |______|  ___   ___  |
    |                                _______   _______ |  |  |  |  |
    |                               |      |  |      | | <-74LS138N|
    |                               |      |  |      | |  |  | <-74LS138N
    |                               |HM    |  |HM    | |  |  |  |  |
    |                               |6116P |  |6116P | |__|  |__|  |
    |                               |3     |  |3     |             |
    |                               |______|  |______|             |
    |______________________________________________________________|


WESTERN DIGITAL MFM HDD CONTROLLER WD 100,1-05

    _____________________________________   __________________________   ___________________
   |       _____________                |__|                         |__|                  |
   |   ·· |N8T31N 8324 |   __________   __________   __________    __________________      |
   |   ·· |____________|  |_74LS08N_|  |74LS191N_|  |TMS2149-5NL   | N8X305N  8328   |     |
   |   ··    __________   __________   __________   __________     |                 |     |
   |·· ··   |SN74LS244N  |74LS191N_|  |74LS191N_|  |TMS2149-5NL    |_________________|     |
   |·· ··    __________   __________   __________   __________     _____   _____   _____   |
   |·· ··   |_74LS32N_|  |_74S00PC_|  |SN74LS244N  |WD1100PE-01   |    |  |    |  | <-N82S181N
   |·· ··    __________   __________   __________   __________    | <-N82S181N |  |    |   |
   |·· ··   |SN74LS14N|  |74LS374N_|  |WD1100PE-05 |_74S138N_|    |____|  |__<-N82S181N|   |
   |·· ··    __________   __________   __________   __________   __________                |
   |··      |_74S06PC_|  |74LS273PC|  |WD11C0PE-12 |_74S138N_|  |74LS175PC|  Xtal 8 MHz    |
   |··   __________  _________  __________  __________  __________  __________  __________ |
   |··  |_74S06PC_| |74LS259PC |WD1100V-07 |DM74S04N_| |SN74LS137N WD1100PE-03 |SN74S74N_| |
   | o               _________  __________  __________  __________  __________  __________ |
   | o  .. .. .. .. |DM74S64N| WD1100PE-09 |DTZM1-60_| |WD11CE-08| |DM74S74N_| |SN74S74N_| |
   | o  .. .. .. ..  _________  __________  __________  __________  __________             |
   | o  .. .. .. .. |AM26LS32P |AM26S02PC| |_T74LS54_| |SN74S74N_| |SN74S124N|             |
   |    ·· ·· ·· ··  _________                                                             |
   |    ·· ·· ·· ·· |AM26LS31CN         Xtal 20 MHz                                        |
   |                                     ___                          ___                  |
   |____________________________________|  |_________________________|  |__________________|

***********************************************************************************************************************************/

#include "emu.h"
#include "cpu/z80/z80.h"

namespace {

class secoinsa20_state : public driver_device
{
public:
	secoinsa20_state(const machine_config &mconfig, device_type type, const char *tag)
		: driver_device(mconfig, type, tag)
		, m_maincpu(*this, "maincpu")
	{ }

	void secoinsa20(machine_config &config);

private:
	virtual void machine_start() override ATTR_COLD;
	virtual void machine_reset() override ATTR_COLD;

	required_device<cpu_device> m_maincpu;
};


// Input ports
static INPUT_PORTS_START( secoinsa20 )
INPUT_PORTS_END


void secoinsa20_state::machine_reset()
{
}

void secoinsa20_state::machine_start()
{
}

void secoinsa20_state::secoinsa20(machine_config &config)
{
	// basic machine hardware
	Z80(config, m_maincpu, XTAL(4'000'000));
}



ROM_START( secoinsa20 )
	ROM_REGION(0x20000, "maincpu", 0)
	ROM_LOAD( "secoinsa_iras_3f_vo2.2_2732a.bin", 0x0000, 0x1000, CRC(50b09416) SHA1(9a799e2e6ebad484226778f41b292a66ebe45849) )

	ROM_REGION(0x10000, "wd_mfm_controller", 0)
	ROM_LOAD( "62-000019-00-a0_n82s181n.u16",     0x0000, 0x0400, CRC(d7819b5d) SHA1(c74dc7cbe05796ec787db482f4ccc92d93df2fa4) )
	ROM_LOAD( "62-000019-01-a0_n82s181n.u15",     0x0400, 0x0400, CRC(bb775759) SHA1(dae3fc65e072c5ebec6c545cef4e58ba0dc44ae7) )
	ROM_LOAD( "62-000019-02-a0_n82s181n.u14",     0x0800, 0x0400, CRC(94f9ed35) SHA1(2a14134849213a733b05b9e216b45909c36bd859) )

	ROM_REGION(0x00117, "plds", 0)
	ROM_LOAD( "asp1_2m_v00.1_pal16l8.2m",         0x0000, 0x0117, BAD_DUMP CRC(c20eae34) SHA1(9a879350be345ea757ba7b6636a9c789932b75b4) ) // On PCB SM16B-0200-0010, bruteforced
	ROM_LOAD( "asp2_2l_v00.1_pal16l8.2l",         0x0000, 0x0117, NO_DUMP ) // On PCB SM16B-0200-0010
	ROM_LOAD( "asp_p3_2k_v00.1_pal16l8.2k",       0x0000, 0x0117, BAD_DUMP CRC(4838b218) SHA1(312a2cca2a403776f6d861763258edee17a040b8) ) // On PCB SM16B-0200-0010, bruteforced
	ROM_LOAD( "palp1_1d_v00.1_pal16l8.1d",        0x0000, 0x0117, NO_DUMP ) // On PCB SM16B-0200-0170
	ROM_LOAD( "palp2_1c_v00.1_pal16r4.1c",        0x0000, 0x0117, NO_DUMP ) // On PCB SM16B-0200-0170
	ROM_LOAD( "palp3_3u_v00.1_pal16l8.3u",        0x0000, 0x0117, NO_DUMP ) // On PCB SM16B-0200-0170
	ROM_LOAD( "palp4_1b_v00.1_pal16r8.1b",        0x0000, 0x0117, NO_DUMP ) // On PCB SM16B-0200-0170
	ROM_LOAD( "palp5_2p_v00.2_pal16l8.2p",        0x0000, 0x0117, BAD_DUMP CRC(0a8d42ef) SHA1(5f3af633742fd4859458237bc576671be8f5a898) ) // On PCB SM16B-0200-0170, bruteforced
	ROM_LOAD( "pcpu_2h_v00.1_pal16l8.2h",         0x0000, 0x0117, NO_DUMP ) // On CPU PCB
	ROM_LOAD( "pss1_1h_v00.1_pal16l8.1h",         0x0000, 0x0117, NO_DUMP ) // On PCB SM16B-0200-0140
	ROM_LOAD( "pss2_1k_v00.1_pal16l8.1k",         0x0000, 0x0117, BAD_DUMP CRC(069664b1) SHA1(c5d32b20a3e9985dd1cb23acfe73f1f8be2e270d) ) // On PCB SM16B-0200-0140, bruteforced
	ROM_LOAD( "pss3_1a_v00.2_pal16r4.1a",         0x0000, 0x0117, NO_DUMP ) // On PCB SM16B-0200-0140
	ROM_LOAD( "pss4_2g_v00.1_pal16l8.2g",         0x0000, 0x0117, BAD_DUMP CRC(34f23cf6) SHA1(c67f0c1878e1a93787bcd09647162e174218fd6f) ) // On PCB SM16B-0200-0140, bruteforced
	ROM_LOAD( "pss5_1n_v00.2_pal16r4.1n",         0x0000, 0x0117, NO_DUMP ) // On PCB SM16B-0200-0140
	ROM_LOAD( "pss6_2m_v00.1_pal16l8.2m",         0x0000, 0x0117, NO_DUMP ) // On PCB SM16B-0200-0140
ROM_END

} // Anonymous namespace

//    YEAR  NAME        PARENT  COMPAT  MACHINE     INPUT       CLASS             INIT        COMPANY     FULLNAME    FLAGS
COMP( 1981, secoinsa20, 0,      0,      secoinsa20, secoinsa20, secoinsa20_state, empty_init, "Secoinsa", "Serie 20", MACHINE_NO_SOUND_HW | MACHINE_NOT_WORKING )
